/*
 * jammo-sample-button.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-sample-button.h"

G_DEFINE_TYPE(JammoSampleButton, jammo_sample_button, TANGLE_TYPE_BUTTON);

enum {
	PROP_0,
	PROP_IMAGE_FILENAME,
	PROP_SAMPLE_FILENAME,
	PROP_SAMPLE
};

struct _JammoSampleButtonPrivate {
	gchar* image_filename;
	JammoSample* sample;
};

JammoSampleButton* jammo_sample_button_new() {

	return JAMMO_SAMPLE_BUTTON(g_object_new(JAMMO_TYPE_SAMPLE_BUTTON, NULL));
}

JammoSampleButton* jammo_sample_button_new_from_files(const gchar* image_filename, const gchar* sample_filename) {

	return JAMMO_SAMPLE_BUTTON(g_object_new(JAMMO_TYPE_SAMPLE_BUTTON, "image-filename", image_filename, "sample-filename", sample_filename, NULL));
}

JammoSampleButton* jammo_sample_button_new_from_existing(JammoSampleButton* sample_button) {

	return JAMMO_SAMPLE_BUTTON(g_object_new(JAMMO_TYPE_SAMPLE_BUTTON, "image-filename", sample_button->priv->image_filename, "sample", sample_button->priv->sample, NULL));
}

const gchar* jammo_sample_button_get_image_filename(JammoSampleButton* sample_button) {

	return sample_button->priv->image_filename;
}

const gchar* jammo_sample_button_get_sample_filename(JammoSampleButton* sample_button) {

	return jammo_sample_get_filename(sample_button->priv->sample);
}

JammoSample* jammo_sample_button_get_sample(JammoSampleButton* sample_button) {

	return sample_button->priv->sample;
}

void jammo_sample_button_set_image_filename(JammoSampleButton* sample_button, const gchar* image_filename) {
	if (!sample_button->priv->image_filename ||
	    strcmp(sample_button->priv->image_filename, image_filename)) {
		g_free(sample_button->priv->image_filename);
		sample_button->priv->image_filename = g_strdup(image_filename);

		tangle_button_set_normal_actor(TANGLE_BUTTON(sample_button),
		                               clutter_texture_new_from_file(image_filename, NULL));

		g_object_notify(G_OBJECT(sample_button), "image-filename");
	}
}

void jammo_sample_button_set_sample_filename(JammoSampleButton* sample_button, const gchar* sample_filename) {
	if (!sample_button->priv->sample ||
	    !jammo_sample_get_filename(sample_button->priv->sample) ||
	    strcmp(jammo_sample_get_filename(sample_button->priv->sample), sample_filename)) {
		if (sample_button->priv->sample) {
			g_object_unref(sample_button->priv->sample);
		}
		sample_button->priv->sample = jammo_sample_new_from_file(sample_filename);
		
		g_object_notify(G_OBJECT(sample_button), "sample-filename");
		g_object_notify(G_OBJECT(sample_button), "sample");
	}
}

void jammo_sample_button_set_sample(JammoSampleButton* sample_button, JammoSample* sample) {
	if (sample_button->priv->sample != sample) {
		if (sample_button->priv->sample) {
			g_object_unref(sample_button->priv->sample);
		}
		g_object_ref(sample);
		sample_button->priv->sample = sample;
		
		g_object_notify(G_OBJECT(sample_button), "sample-filename");
		g_object_notify(G_OBJECT(sample_button), "sample");
	
	}
}

static void jammo_sample_button_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoSampleButton* sample_button;
	
	sample_button = JAMMO_SAMPLE_BUTTON(object);

	switch (prop_id) {
		case PROP_IMAGE_FILENAME:
			jammo_sample_button_set_image_filename(sample_button, g_value_get_string(value));
			break;
		case PROP_SAMPLE_FILENAME:
			jammo_sample_button_set_sample_filename(sample_button, g_value_get_string(value));
			break;
		case PROP_SAMPLE:
			jammo_sample_button_set_sample(sample_button, JAMMO_SAMPLE(g_value_get_object(value)));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_sample_button_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoSampleButton* sample_button;

	sample_button = JAMMO_SAMPLE_BUTTON(object);

        switch (prop_id) {
		case PROP_IMAGE_FILENAME:
			g_value_set_string(value, jammo_sample_button_get_image_filename(sample_button));
			break;
		case PROP_SAMPLE_FILENAME:
			g_value_set_string(value, jammo_sample_button_get_sample_filename(sample_button));
			break;
		case PROP_SAMPLE:
			g_value_set_object(value, jammo_sample_button_get_sample(sample_button));
			break;
		default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_sample_button_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_sample_button_parent_class)->finalize(object);
}

static void jammo_sample_button_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_sample_button_parent_class)->dispose(object);
}

static void jammo_sample_button_class_init(JammoSampleButtonClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->finalize = jammo_sample_button_finalize;
	gobject_class->dispose = jammo_sample_button_dispose;
	gobject_class->set_property = jammo_sample_button_set_property;
	gobject_class->get_property = jammo_sample_button_get_property;

	/**
	 * JammoSampleButton:image-filename:
	 */
	g_object_class_install_property(gobject_class, PROP_IMAGE_FILENAME,
	                                g_param_spec_string("image-filename",
	                                "Image filename",
	                                "The filename of the displayed image",
					NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	/**
	 * JammoSampleButton:sample-filename:
	 */
	g_object_class_install_property(gobject_class, PROP_SAMPLE_FILENAME,
	                                g_param_spec_string("sample-filename",
	                                "Sample filename",
	                                "The filename of the played sample",
					NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	/**
	 * JammoSampleButton:sample:
	 */
	g_object_class_install_property(gobject_class, PROP_SAMPLE,
	                                g_param_spec_object ("sample",
	                                "Sample",
	                                "The JammoSample that is played when the button is interacting",
	                                JAMMO_TYPE_SAMPLE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(JammoSampleButtonPrivate));
}

static void jammo_sample_button_init(JammoSampleButton* samplebutton) {
	samplebutton->priv = G_TYPE_INSTANCE_GET_PRIVATE(samplebutton, JAMMO_TYPE_SAMPLE_BUTTON, JammoSampleButtonPrivate);
}

