/*
 * jammo-sample.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-sample.h"
#include "jammo-track.h"
#include "jammo-meam.h"
#include "jammo-meam-private.h"

enum {
	PROP_0,
	PROP_FILENAME,
	PROP_DURATION
};

enum {
	STOPPED,
	LAST_SIGNAL
};

struct _JammoSamplePrivate {
	gchar* filename;
	guint64 duration;
	GstElement* element;
	JammoEditingTrack* editing_track;
};

typedef struct {
	gboolean preroll_only;
	JammoSample* sample;
	GstElement* pipeline;
	GstElement* sink;
} PlayData;

G_DEFINE_TYPE(JammoSample, jammo_sample, G_TYPE_INITIALLY_UNOWNED);

static void play_sample(JammoSample* sample, gboolean preroll_only);
static void stop_and_free_play_data(PlayData* play_data);

static guint signals[LAST_SIGNAL] = { 0 };
static GList* play_datas = NULL;

JammoSample* jammo_sample_new_from_file(const gchar* filename) {

	return JAMMO_SAMPLE(g_object_new(JAMMO_TYPE_SAMPLE, "filename", filename, NULL));
}

JammoSample* jammo_sample_new_from_existing(JammoSample* sample) {

	return JAMMO_SAMPLE(g_object_new(JAMMO_TYPE_SAMPLE,
	                                 "filename", sample->priv->filename,
					 "duration", sample->priv->duration,
					 NULL));
}

const gchar* jammo_sample_get_filename(JammoSample* sample) {

	return sample->priv->filename;
}

guint64 jammo_sample_get_duration(JammoSample* sample) {
	
	return sample->priv->duration;
}

JammoEditingTrack* jammo_sample_get_editing_track(JammoSample* sample) {

	return sample->priv->editing_track;
}

void jammo_sample_play(JammoSample* sample) {
	play_sample(sample, FALSE);	
}

void jammo_sample_stop(JammoSample* sample) {
	GList* list;
	
	for (list = play_datas; list; list = list->next) {
		if (((PlayData*)list->data)->sample == sample) {
			break;
		}
	}
	if (list) {
		stop_and_free_play_data((PlayData*)list->data);
		play_datas = g_list_delete_link(play_datas, list);
	}
}

void jammo_sample_stop_all(void) {
	if (play_datas) {
		g_list_foreach(play_datas, (GFunc)stop_and_free_play_data, NULL);
		g_list_free(play_datas);
		play_datas = NULL;
	}
}

GstElement* _jammo_sample_get_element(JammoSample* sample) {

	return sample->priv->element;
}

void _jammo_sample_set_editing_track(JammoSample* sample, JammoEditingTrack* editing_track) {
	sample->priv->editing_track = editing_track;
}

static void jammo_sample_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoSample* sample;
	
	sample = JAMMO_SAMPLE(object);

	switch (prop_id) {
		case PROP_FILENAME:
			sample->priv->filename = g_strdup(g_value_get_string(value));
			break;
		case PROP_DURATION:
			sample->priv->duration = g_value_get_uint64(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_sample_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoSample* sample;

	sample = JAMMO_SAMPLE(object);

        switch (prop_id) {
		case PROP_FILENAME:
			g_value_set_string(value, sample->priv->filename);
			break;
		case PROP_DURATION:
			g_value_set_uint64(value, sample->priv->duration);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static GObject* jammo_sample_constructor(GType type, guint n_properties, GObjectConstructParam* properties) {
	GObject* object;
	JammoSample* sample;

	object = G_OBJECT_CLASS(jammo_sample_parent_class)->constructor(type, n_properties, properties);

	sample = JAMMO_SAMPLE(object);
	sample->priv->element = gst_element_factory_make("gnlfilesource", NULL);
	g_object_set(sample->priv->element,
		     "location", sample->priv->filename,
		     "media-start", 0 * GST_SECOND,
		     NULL);
	if (sample->priv->duration != JAMMO_DURATION_INVALID) {
		g_object_set(sample->priv->element,
		             "duration", sample->priv->duration,
			     "media-duration", sample->priv->duration,
			     NULL);
	} else {
		play_sample(sample, TRUE);
	}

	return object;
}

static void jammo_sample_finalize(GObject* object) {
	JammoSample* sample;
	
	sample = JAMMO_SAMPLE(object);
	g_free(sample->priv->filename);

	G_OBJECT_CLASS(jammo_sample_parent_class)->finalize(object);
}

static void jammo_sample_dispose(GObject* object) {
	JammoSample* sample;
	
	sample = JAMMO_SAMPLE(object);
	if (sample->priv->element) {
		g_object_unref(sample->priv->element);
		sample->priv->element = NULL;
	}

	G_OBJECT_CLASS(jammo_sample_parent_class)->dispose(object);
}

static void jammo_sample_class_init(JammoSampleClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->constructor = jammo_sample_constructor;
	gobject_class->finalize = jammo_sample_finalize;
	gobject_class->dispose = jammo_sample_dispose;
	gobject_class->set_property = jammo_sample_set_property;
	gobject_class->get_property = jammo_sample_get_property;

	/**
	 * JammoSample:filename:
	 */
	g_object_class_install_property(gobject_class, PROP_FILENAME,
	                                g_param_spec_string("filename",
	                                "File name",
	                                "The file name of the sample",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoSample:duration:
	 */
	g_object_class_install_property(gobject_class, PROP_DURATION,
	                                g_param_spec_uint64("duration",
	                                "Duration",
	                                "The duration of the sample in nanoseconds",
	                                0, G_MAXUINT64, JAMMO_DURATION_INVALID,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoSample::stopped:
	 * @sample: the object which received the signal
	 *
	 * The ::stopped signal is emitted when the playing of the (single) sample has stopped.
	 */
	signals[STOPPED] = g_signal_new("stopped", G_TYPE_FROM_CLASS(gobject_class),
					       G_SIGNAL_RUN_LAST, 0,
					       NULL, NULL,
					       g_cclosure_marshal_VOID__VOID,
					       G_TYPE_NONE, 0);

	g_type_class_add_private (gobject_class, sizeof(JammoSamplePrivate));
}

static void jammo_sample_init(JammoSample* sample) {
	sample->priv = G_TYPE_INSTANCE_GET_PRIVATE(sample, JAMMO_TYPE_SAMPLE, JammoSamplePrivate);
}

static gboolean preroll_bus_callback(GstBus* bus, GstMessage* message, gpointer data) {
	PlayData* play_data;
	gboolean cleanup;
	GstFormat format;
	gint64 duration;
	
	play_data = (PlayData*)data;
	cleanup = FALSE;
	if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_ASYNC_DONE || GST_MESSAGE_TYPE(message) == GST_MESSAGE_EOS) {
		format = GST_FORMAT_TIME;
		if (gst_element_query_duration(play_data->pipeline, &format, &duration)) {
			play_data->sample->priv->duration = (guint64)duration;
			g_object_set(play_data->sample->priv->element,
				     "duration", play_data->sample->priv->duration,
				     "media-duration", play_data->sample->priv->duration,
				     NULL);
			g_object_notify(G_OBJECT(play_data->sample), "duration");
		}
		cleanup = TRUE;
	} else if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_ERROR) {
		play_data->sample->priv->duration = 0;
		cleanup = TRUE;
	}
	if (cleanup) {
		stop_and_free_play_data(play_data);
	}
	
	return !cleanup;
}

static gboolean play_bus_callback(GstBus* bus, GstMessage* message, gpointer data) {
	PlayData* play_data;
	gboolean cleanup;
	
	play_data = (PlayData*)data;
	cleanup = FALSE;
	if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_EOS ||
	    GST_MESSAGE_TYPE(message) == GST_MESSAGE_ERROR) {
		stop_and_free_play_data(play_data);
		play_datas = g_list_remove(play_datas, play_data);
		cleanup = TRUE;
	}
	
	return !cleanup;
}

static void on_new_decoded_pad(GstElement* element, GstPad* pad, gboolean last, gpointer data) {
	PlayData* play_data;
	GstPad* sinkpad;
	GstCaps* caps;

	play_data = (PlayData*)data;
	sinkpad = gst_element_get_pad(play_data->sink, "sink");
	if (!GST_PAD_IS_LINKED(sinkpad)) {
		caps = gst_pad_get_caps(pad);
		if (g_strrstr(gst_structure_get_name(gst_caps_get_structure(caps, 0)), "audio")) {
			gst_pad_link(pad, sinkpad);
		}
		gst_caps_unref(caps);
	}
	g_object_unref(sinkpad);
}

static void play_sample(JammoSample* sample, gboolean preroll_only) {
	PlayData* play_data;
	GstBus* bus;
	GstElement* filesrc;
	GstElement* decodebin;
	GstElement* sink;

	play_data = g_malloc0(sizeof(PlayData));
	play_data->preroll_only = preroll_only;
	play_data->sample = JAMMO_SAMPLE(g_object_ref(sample));
	play_data->pipeline = gst_element_factory_make("pipeline", NULL);
	bus = gst_pipeline_get_bus(GST_PIPELINE(play_data->pipeline));
	gst_bus_add_watch(bus, (preroll_only ? preroll_bus_callback : play_bus_callback), play_data);
	gst_object_unref(bus);

	filesrc = gst_element_factory_make("filesrc", NULL);
	g_object_set(filesrc, "location", sample->priv->filename, NULL);
	decodebin = gst_element_factory_make("decodebin", NULL);
	g_signal_connect(decodebin, "new-decoded-pad", G_CALLBACK(on_new_decoded_pad), play_data);

	if (preroll_only) {
		play_data->sink = gst_element_factory_make("fakesink", NULL);
		gst_bin_add_many(GST_BIN(play_data->pipeline), filesrc, decodebin, play_data->sink, NULL);
	} else {
		play_data->sink = gst_element_factory_make("audioconvert", NULL);
		sink = gst_element_factory_make("autoaudiosink", NULL);
		gst_bin_add_many(GST_BIN(play_data->pipeline), filesrc, decodebin, play_data->sink, sink, NULL);
		gst_element_link(play_data->sink, sink);

		play_datas = g_list_prepend(play_datas, play_data);
	}
	gst_element_link(filesrc, decodebin);

	if (preroll_only) {
		gst_element_set_state(play_data->pipeline, GST_STATE_PAUSED); 
	} else {
		gst_element_set_state(play_data->pipeline, GST_STATE_PLAYING); 
	}
}

static void stop_and_free_play_data(PlayData* play_data) {
	if (!play_data->preroll_only) {
		g_signal_emit(play_data->sample, signals[STOPPED], 0);
	}
	g_object_unref(play_data->sample);
	gst_element_set_state(play_data->pipeline, GST_STATE_NULL);
	gst_object_unref(play_data->pipeline);
	g_free(play_data);
}
