/**track.c is part of JamMo.
License: GPLv2, read more from COPYING

Track is always in pipeline. Track contains samples.
*/
#include "track.h"
#include "../../chum/chum.h" //because of DATA_DIR

//private functions
static void trackbin_and_link(Track *trc,GstElement *audioconvertA,GstElement *audioconvertB);
static void gnl_OnNewPad(GstElement *elem, GstPad *pad, gpointer data);


#ifdef NOT_IN_USE
/*These might be useful */
#define DEFAULT_CHANNELS 1
#define GST_TYPE_PCM "audio/x-raw-int"
#define PCM_RATE 8000
#define PCM_ENDIANNESS 1234
#define PCM_WIDTH 16
#define PCM_DEPTH 16
static GstCaps *createCapsFilter() {
return gst_caps_new_simple(
       GST_TYPE_PCM,
       "rate", G_TYPE_INT, PCM_RATE,
       "signed", G_TYPE_BOOLEAN, TRUE,
       "channels", G_TYPE_INT, DEFAULT_CHANNELS,
       "endianness", G_TYPE_INT, PCM_ENDIANNESS,
       "width", G_TYPE_INT, PCM_WIDTH,
       "depth", G_TYPE_INT, PCM_DEPTH,
       NULL);
}
#endif

/**
 *  create_new_empty_track - Function for adding new empty track, after this function you can add samples 
 *  to the track with add_audiosrc_to_track function.
 */
Track *create_new_empty_track(void)
{
	Track *trc = (Track*) malloc(sizeof(Track));
	if(trc != NULL)
	{
                trc->type = TRACK_SAMPLES;
		int track_number = g_list_length(get_mainPipeline()->tracks)+1;
		gchar track_nbr[10];
		gchar composition_nbr[16];
		gchar silencesrc_nbr[15];
		gchar silence_nbr[12];

		gchar converterA_nbr[15];
#ifdef ENABLE_PITCH_ELEMENT
		gchar pitch_nbr[10];
#endif
		gchar volume_nbr[11];
		gchar converterB_nbr[15];

		GstElement *silenceSource;
		GstElement *silence;
		GstElement *audioconvertA;
		GstElement *audioconvertB;

		sprintf(track_nbr, "track_%d", track_number);
		trc->trackbin = gst_bin_new(track_nbr);

		sprintf(composition_nbr, "composition_%d", track_number);
		trc->composition = gst_element_factory_make("gnlcomposition", composition_nbr);

		// Add silence to the background of the track
		sprintf(silencesrc_nbr, "silencesrc_%d", track_number);
		silenceSource = gst_element_factory_make("gnlsource", silencesrc_nbr);
		sprintf(silence_nbr, "silence_%d", track_number);
		silence = gst_element_factory_make("audiotestsrc", silence_nbr);
		g_object_set(G_OBJECT(silence), "wave", 4, NULL);
		gst_bin_add(GST_BIN(silenceSource), silence);
		g_object_set(G_OBJECT (silenceSource), "start", 0 * GST_SECOND, NULL);
		g_object_set(G_OBJECT (silenceSource), "duration", 9999 * GST_SECOND, NULL);
		g_object_set(G_OBJECT (silenceSource), "media-start", 0 * GST_SECOND, NULL);
		g_object_set(G_OBJECT (silenceSource), "media-duration", 9999 * GST_SECOND, NULL);
		g_object_set(G_OBJECT (silenceSource), "priority", 9999,  NULL);
		gst_bin_add(GST_BIN(trc->composition), silenceSource);

		trc->samples = NULL;

		// It's noticed that there should be an audioconvert before pitch-element to work
		sprintf(converterA_nbr, "converterA_%d", track_number);
		audioconvertA = gst_element_factory_make("audioconvert", converterA_nbr);

#ifdef ENABLE_PITCH_ELEMENT
		sprintf(pitch_nbr, "pitch_%d", track_number);
		trc->pitch = gst_element_factory_make("pitch", pitch_nbr);
#endif

		sprintf(volume_nbr, "volume_%d", track_number);
		trc->volume = gst_element_factory_make("volume", volume_nbr);

		sprintf(converterB_nbr, "converterB_%d", track_number);	
		audioconvertB = gst_element_factory_make("audioconvert", converterB_nbr);

		trackbin_and_link(trc,audioconvertA,audioconvertB);

		get_mainPipeline()->tracks = g_list_append(get_mainPipeline()->tracks, trc);

                //If user thinks to use these, call create_new_empty_track_with_tempo_and_meter
                trc->tempo=130;
                trc->meter=4;

		//malloc space for slot_infos and initialize all to zero
		int i;
		for (i=0;i<SLOT_LIMIT;i++) {
		    trc->slot_info[i] = (SlotInfo*) malloc(sizeof(SlotInfo));
		    trc->slot_info[i]->info=0;
		    trc->slot_info[i]->sample=NULL;
		}

	}
        return trc;
}

void create_new_empty_track_with_tempo_and_meter(guint tempo,guint meter) {
  struct Track *trc = create_new_empty_track();
  trc->tempo=tempo;
  trc->meter=meter;
}


//Testing without gnonlin.
static void
on_pad_added (GstElement *element,
              GstPad     *pad,
              gpointer    data)
{
  GstPad *sinkpad;
  GstElement *decoder = (GstElement *) data;
  /* We can now link this pad with the vorbis-decoder sink pad */
  g_print ("Dynamic pad created, linking demuxer/decoder\n");
  sinkpad = gst_element_get_static_pad (decoder, "sink");
  gst_pad_link (pad, sinkpad);
  gst_object_unref (sinkpad);
}
/**
 *  create_new_empty_track - Function for adding new empty track, after this function you can add samples 
 *  to the track with add_audiosrc_to_track function.
 */
Track *create_new_backing_track(gchar *filename)
{
  Track *trc = (Track*) malloc(sizeof(Track));
  if(trc != NULL)
  {
    trc->type = TRACK_BACKINGTRACK;
    GstElement  *convert, *source1;

    /*track 1 uses these */
    GstElement *demuxer1, *decoder1;

    /* Create gstreamer elements */
    convert  = gst_element_factory_make ("audioconvert",  NULL);
  

    /*track 1*/
    source1   = gst_element_factory_make ("filesrc",       NULL);
    demuxer1  = gst_element_factory_make ("oggdemux",      NULL);        ///
    decoder1  = gst_element_factory_make ("vorbisdec",     NULL);     ///

    g_object_set (G_OBJECT (source1), "location", filename, NULL);
  
    /* add all elements into the pipeline */
   gst_bin_add_many (GST_BIN (get_mainPipeline()->pipeline),
                        source1, demuxer1, decoder1,
                        convert,  NULL);
 
  
   gst_element_link (source1, demuxer1);
   gst_element_link_many (decoder1, convert,  NULL);
   g_signal_connect (demuxer1, "pad-added", G_CALLBACK (on_pad_added), decoder1);

   gst_element_link(convert, get_mainPipeline()->adder);
   get_mainPipeline()->tracks = g_list_append(get_mainPipeline()->tracks, trc);
   }
   return trc;
}

/**
Create track for recording from microphone.
Save wav to file.
This doesn't need samples or volume
*/
void create_new_singing_track(void)
{
	Track *trc = (Track*) malloc(sizeof(Track));
	if(trc != NULL)
	{	
                trc->type = TRACK_SINGING;
		int track_number = g_list_length(get_mainPipeline()->tracks)+1;
		gchar track_nbr[10];
		GstElement *source;

		sprintf(track_nbr, "track_%d", track_number);
		trc->trackbin = gst_bin_new(track_nbr);


	//common for all architectures
	GstElement *sink = NULL;


	sink = gst_element_factory_make("filesink", NULL);
	g_object_set(G_OBJECT(sink),"location", REC_TARGET, NULL);

	GstElement *pitchdetect;
	pitchdetect = gst_element_factory_make ("pitchdetect", "pitchdetect");


	#ifdef N810
	GstElement *parse = NULL;
           source = gst_element_factory_make("dsppcmsrc", NULL);
           g_object_set(G_OBJECT (source),"blocksize","160",NULL); //what is this?
           parse = gst_element_factory_make("wavenc", NULL);

	gst_bin_add_many(GST_BIN(trc->trackbin), source,pitchdetect, parse, sink, NULL);
	if (!gst_element_link (source, pitchdetect))
	  {
	  g_print("gst_element_link 1 !\n");
	  return;
	  }
        if (!gst_element_link ( pitchdetect,parse))
	  {
	  g_print("gst_element_link 2 !\n");
	  return;
	  }
	if (!gst_element_link(parse,sink))
	  {
	  g_print("gst_element_link 3 !\n");
	  return;
	  }


	#else
	GstElement *audioconvertA;
	gchar converterA_nbr[15];
	GstElement *encoder;

         /*
         GstCaps *filter;
         filter = gst_caps_new_simple("audio/x-raw-int",
          "channels", G_TYPE_INT, 1,
          "rate", G_TYPE_INT, 8000,
          "depth", G_TYPE_INT, 16, NULL);
          */
        source = gst_element_factory_make("autoaudiosrc", NULL);
	sprintf(converterA_nbr, "converterA_%d", track_number);
	audioconvertA = gst_element_factory_make("audioconvert", converterA_nbr);
	encoder = gst_element_factory_make("wavenc", NULL);
	//gst_bin_add_many(GST_BIN(trc->trackbin),source, pitchdetect,audioconvertA, filter, encoder,sink,NULL);
        gst_bin_add_many(GST_BIN(trc->trackbin),source, pitchdetect,audioconvertA, encoder,sink,NULL);

	if (!gst_element_link (source, pitchdetect))
	  {
	  g_print("gst_element_link error (source, pitchdetect)!\n");
	  return;
	  }
        if (!gst_element_link (pitchdetect,audioconvertA))
	  {
	  g_print("gst_element_link error (pitchdetect,audioconvertA) !\n");
	  return;
	  }
	if (!gst_element_link (audioconvertA,encoder))
        //if (!gst_element_link_filtered(audioconvertA, encoder, filter))
	  {
	  g_print("gst_element_link_filtered error (audioconvertA,encoder)!\n");
	  return;
	  }

	if (!gst_element_link (encoder,sink))
	  {
	  g_print("gst_element_link (encoder,sink)!\n");
	  return;
	  }

	#endif


  
	//add this track to mainpipeline
	gst_bin_add(GST_BIN(get_mainPipeline()->pipeline), trc->trackbin);
	get_mainPipeline()->tracks = g_list_append(get_mainPipeline()->tracks, trc);
	trc->samples = NULL;
	}

}

#define INSTRUMENT_FOLDER g_strdup_printf("%s/virtual-instruments/",DATA_DIR)
/**
 *  create_new_empty_track - for virtual instruments. 
parameter in this moment:
0=flute
1=drumkit
2=ud
3=slider (slider have own instrument parameter too)
 */
Track *create_instrument_track(gint instrument_nro)
{
	Track *trc = (Track*) malloc(sizeof(Track));
	if(trc != NULL)
	{	
                trc->type = TRACK_INSTRUMENT;
		int track_number = g_list_length(get_mainPipeline()->tracks)+1;
		gchar track_nbr[10];
		gchar composition_nbr[16];
		gchar instrument_nbr[15];
		gchar source_nbr[12];
		//gchar audiosrc_nbr[13];
		gchar converterA_nbr[15];
		gchar volume_nbr[11];

		GstElement *source;
		GstElement *audioconvertA;

		sprintf(track_nbr, "track_%d", track_number);
		trc->trackbin = gst_bin_new(track_nbr);

		sprintf(composition_nbr, "composition_%d", track_number);
		trc->composition = gst_element_factory_make("gnlcomposition", composition_nbr);

		// Add silence to the background of the track
		sprintf(source_nbr, "source_%d", track_number);
		source = gst_element_factory_make("gnlsource", source_nbr);
		sprintf(instrument_nbr, "instrument_%d", track_number);

	//this is  passed parameter
	if (instrument_nro==3){
	  trc->instrument_element = gst_element_factory_make("jammoslider", instrument_nbr);
	  g_object_set(G_OBJECT(trc->instrument_element), "instrument", 2, NULL); //2=karplus-stronger
	  }
	else {
		trc->instrument_element = gst_element_factory_make("jammosampler", instrument_nbr);

	    /////Instrument///
		//We must first define folder of samples
		g_object_set(G_OBJECT(trc->instrument_element), "instrument-folder", INSTRUMENT_FOLDER, NULL);
		char *buffer=NULL;

		if (instrument_nro==0){
		    buffer=g_strdup_printf("%s%s",INSTRUMENT_FOLDER,"flute.txt");
		}

		if (instrument_nro==1){
		      buffer=g_strdup_printf("%s%s",INSTRUMENT_FOLDER,"drumkit.txt");
		}

		if (instrument_nro==2){
		  buffer=g_strdup_printf("%s%s",INSTRUMENT_FOLDER,"ud.txt");
		}


		g_object_set(G_OBJECT(trc->instrument_element), "init-instrument", buffer, NULL);
	  ////////////////////////
	      }

		gst_bin_add(GST_BIN(source), trc->instrument_element);
		g_object_set(G_OBJECT (source), "start", 0 * GST_SECOND, NULL);
		g_object_set(G_OBJECT (source), "duration", 9999 * GST_SECOND, NULL);
		g_object_set(G_OBJECT (source), "media-start", 0 * GST_SECOND, NULL);
		g_object_set(G_OBJECT (source), "media-duration", 9999 * GST_SECOND, NULL);
		g_object_set(G_OBJECT (source), "priority", 9999,  NULL);
		gst_bin_add(GST_BIN(trc->composition), source);

		trc->samples = NULL;


		sprintf(converterA_nbr, "converterA_%d", track_number);
		audioconvertA = gst_element_factory_make("audioconvert", converterA_nbr);

		sprintf(volume_nbr, "volume_%d", track_number);
		trc->volume = gst_element_factory_make("volume", volume_nbr);

	
		gst_bin_add_many(GST_BIN(trc->trackbin),trc->composition, audioconvertA,trc->volume, NULL);

		gst_element_link(audioconvertA,trc->volume);
		g_signal_connect(trc->composition, "pad-added", G_CALLBACK(gnl_OnNewPad), audioconvertA);

		//the output of tracks trackbin should be output of volume-element:
		gst_element_add_pad(trc->trackbin,
			gst_ghost_pad_new("src", gst_element_get_pad(trc->volume, "src")));

		//add this track to mainpipeline
		gst_bin_add(GST_BIN(get_mainPipeline()->pipeline), trc->trackbin);
		gst_element_link(trc->trackbin, get_mainPipeline()->adder);
		if(GST_PAD_LINK_SUCCESSFUL(GST_PAD_LINK_OK))
			g_print("Linking successfully completed (Track -> Adder)\n");
		else 
			g_print("Linking failed! (Track -> Adder)\n");

		get_mainPipeline()->tracks = g_list_append(get_mainPipeline()->tracks, trc);

	}
return trc;
}

/**
 *  trackbin_and_link - Function for adding GstElements to one bin and linking them together.
 *  @param trc - a pointer to the track which will be "binned" and linked.
 */
static void trackbin_and_link(Track *trc, GstElement *audioconvertA,GstElement *audioconvertB)
{
	gst_bin_add(GST_BIN(trc->trackbin), trc->composition);

	g_signal_connect(trc->composition, "pad-added", G_CALLBACK(gnl_OnNewPad), audioconvertA);
	
	// Get source pad from audioconvertB to be made to ghostpad for trackbin element 
	GstPad *audiopad;
	audiopad = gst_element_get_pad(audioconvertB, "src");

#ifdef ENABLE_PITCH_ELEMENT
	gst_bin_add_many(GST_BIN(trc->trackbin), audioconvertA, trc->pitch, trc->volume, audioconvertB, NULL);
	gst_element_link_many(audioconvertA, trc->pitch, trc->volume, audioconvertB, NULL);

#else
	gst_bin_add_many(GST_BIN(trc->trackbin), audioconvertA, trc->volume, audioconvertB, NULL);
	gst_element_link_many(audioconvertA, trc->volume, audioconvertB, NULL);
#endif

	gst_element_add_pad(trc->trackbin,
		gst_ghost_pad_new("src", audiopad));
  	gst_object_unref(audiopad);

	// Add newly created trackbin to the pipeline and link trackbins ghostpad to the adder element 
	gst_bin_add(GST_BIN(get_mainPipeline()->pipeline), trc->trackbin);
	gst_element_link(trc->trackbin, get_mainPipeline()->adder);
	if(GST_PAD_LINK_SUCCESSFUL(GST_PAD_LINK_OK))
    		g_print("Linking successfully completed (Track -> Adder)\n");
  	else 
    		g_print("Linking failed! (Track -> Adder)\n");

}

/**
 *  gnl_OnNewPad - For dynamic pad adding.
 *  @param elem - a pointer to the element where the new pad was created
 *  @param pad - a pointer to the newly created pad
 *  @param data - a pointer to the user defined data (in our case this is usually tracks audioconvertA) 
 */
static void gnl_OnNewPad(GstElement *elem, GstPad *pad, gpointer data)
{
	GstPad *convsink;

	convsink = gst_element_get_compatible_pad(data, pad, gst_pad_get_caps(pad));
	if(convsink)
		g_print("Compatible pad found...\n");
	else
	    	g_print("Compatible pad not found!\n");
	
	gst_pad_link(pad, convsink);
	if(GST_PAD_LINK_SUCCESSFUL(GST_PAD_LINK_OK))
	    	g_print("Linking successfully completed\n"); 

	gst_object_unref(convsink);
} 

/**
 *  free_track - function for freeing the allocated memory of track elements
 *  @param trc - a pointer to the track which should be freed
 */
void free_track(Track *trc)
{
	// Free all the allocated memory from tracks and then free the GList itself
	if(trc->samples != NULL) {
		g_list_foreach(trc->samples, (GFunc)free_sample, NULL);
		g_list_free(trc->samples);	
	}

	g_print("Freeing track...\n");
	free(trc);
	trc = NULL;
	g_print("\t-> Track freed\n");
}

/**
 *  set_track_pitch - function for setting the pitch of the given track
 *  @param trackNumber - index number of the track (from pipelines track list)
 *  @param pitchlevel - new value for pitch (0.1 -3.0). default=1.
 */
void set_track_pitch(guint trackNumber, double pitchlevel)
{
#ifdef ENABLE_PITCH_ELEMENT
	Pipeline *pipe = get_mainPipeline();
	Track *trc = (Track*) g_list_nth_data(pipe->tracks, trackNumber);
	g_object_set(G_OBJECT(trc->pitch), "pitch", pitchlevel, NULL);
#endif
}


/**
 *  set_track_speed - function for setting the speed of the given track
 * Durations are still reported normally!
 *  @param trackNumber - index number of the track (from pipelines track list)
 *  @param speedlevel - new value for speed. 
 */
void set_track_speed(guint trackNumber, double speedlevel)
{
#ifdef ENABLE_PITCH_ELEMENT
	Pipeline *pipe = get_mainPipeline();
	Track *trc = (Track*) g_list_nth_data(pipe->tracks, trackNumber);
	g_object_set(G_OBJECT(trc->pitch), "tempo", speedlevel, NULL);
#endif
}

/**
 *  set_track_mute - function for setting the given track to muted
 *  @param trackNumber - index number of the track (from pipelines track list)
 *  @param state - TRUE == muted, FALSE == not muted
 */
void set_track_mute(guint trackNumber, gboolean state)
{
	Pipeline *pipe = get_mainPipeline();

	Track *trc = (Track*) g_list_nth_data(pipe->tracks, trackNumber);
	g_object_set(G_OBJECT(trc->volume), "mute", state, NULL);
}

/**
 *  remove_samples_from_tracks - Function for removing all samples from tracks
 *  NOTE: this function sets the pipelines state to NULL because samples can't
 *  be removed in any other state.
 *  Maybe this can be merged to use: remove_samples_from_track(guint track_number) ?
 */
void remove_samples_from_all_tracks(void)
{
	Pipeline *pipeline = get_mainPipeline();
	
	gst_element_set_state(GST_ELEMENT(get_mainPipeline()->pipeline), GST_STATE_NULL);

	if(pipeline->tracks != NULL) {
		gint i = g_list_length(pipeline->tracks)-1;
		for(; i >= 0; i--) {
			Track *trc = g_list_nth_data(pipeline->tracks, i);
			g_print("Track: %s\n", gst_element_get_name(trc->trackbin));
			if(g_list_length(trc->samples) != 0) {
				g_list_foreach(trc->samples, (GFunc)remove_sample_from_track, NULL);
			}
		}
		// Set the pipelines end time to 0
		pipeline->pipelines_end_time = 0LLU;
	}
}


/*Checks all samples in asked track and returns biggest end_time. This can be called lenght of this track.
Returns -1, if there aren't such a track.
returned value is wrong, but I think they should be same?

*/
guint64 count_track_endtime(guint track_number,guint64 *value) {
//g_print("counting end of track\n\n");
guint64 biggest =0;
Pipeline *pipeline = get_mainPipeline();
if(pipeline->tracks != NULL) {
	guint i = g_list_length(pipeline->tracks)-1;
	if (track_number<=i){
	    Track *track = g_list_nth_data(pipeline->tracks, track_number);
	    GList *samples = track->samples;
	    GList *list;

	    for (list = samples; list; list = list->next){
	      Sample *sample =list->data;
 	      if (biggest < sample->end_time) 
		  biggest = sample->end_time;
	    }
	}
}

//g_print ("Biggest is: %" GST_TIME_FORMAT " \n",GST_TIME_ARGS (biggest)); 
*value=biggest;
return biggest;
}

/**
 *  remove_samples_from_tracks - Function for removing all samples from tracks
 *  NOTE: this function sets the pipelines state to NULL because samples can't
 *  be removed in any other state.
 */
void remove_samples_from_track(guint track_number)
{
	Pipeline *pipeline = get_mainPipeline();
	
	gst_element_set_state(GST_ELEMENT(get_mainPipeline()->pipeline), GST_STATE_NULL);

	if(pipeline->tracks != NULL) {
			Track *trc = g_list_nth_data(pipeline->tracks, track_number);
                        if (trc != NULL)
                        {
			g_print("Track: %s\n", gst_element_get_name(trc->trackbin));

                        gint i;
			//empty all slot_infos (not all track type has slot_info)
                        if (trc->type == TRACK_SAMPLES)
                          {
                          for (i=0;i<SLOT_LIMIT;i++) {
                                trc->slot_info[i]->info=0;
                                trc->slot_info[i]->sample=NULL;
                            }
                          }

			if(g_list_length(trc->samples) != 0) {
				g_list_foreach(trc->samples, (GFunc)remove_sample_from_track, NULL);
			}

      //If this was longest track in pipeline, we must calculate pipelies lenght.
      //search all tracks and put longest to pipelines length
		guint64 biggest =0;
		i = g_list_length(pipeline->tracks)-1;
		for(; i >= 0; i--) {
			guint64 length;
			count_track_endtime(i,&length);
			if (biggest < length) 
			biggest = length;
			}
		pipeline->pipelines_end_time = biggest;
		}
        }
 }

