/**meam.c is part of JamMo.
License: GPLv2, read more from COPYING

(from D2.4)
MEAM will be composed of several different modules which are started as they are
needed. The sequencer is the most important component of MEAM and it is used as
the basis of all composing. It will be running throughout the composing process.
Virtual instruments are used in jamming sessions and to create parts of music to
be added on sequencer tracks. Virtual instrument components are started as the
user request for them.
*/

#include "meam.h"
#include "sequencer/sequencer.h"
#include "stand_alone_player/stand_alone_player.h"
#include <sys/time.h>

void meam_init(){
    gst_init(NULL, NULL); /* (at least) sequencer and stand_alone_player uses gst*/
    sequencer_init();
}

void meam_create_new_empty_track() {
  create_new_empty_track();
}

void meam_get_duration_of_song(guint64 *value){
seq_get_lenght_of_pipeline(value);
}

void meam_seq_stop(){
    seq_stop();
}
void meam_seq_play(){
    seq_play();
}

void meam_seq_remove_samples_from_track(int track) {
    remove_samples_from_track(track);
}

Sample *meam_seq_add_new_sample_to_track(int track_number, const gchar srcLocation[], guint64 startTime){
return add_new_sample_to_track(track_number, srcLocation, startTime);
}

Sample *meam_seq_add_new_sample_to_track_nth_slot(int track_number, const gchar srcLocation[], guint slot){
return add_new_sample_to_track_nth_slot(track_number, srcLocation, slot);
}

void meam_remove_sample_from_slot(guint track_number,int slot){
empty_track_from_slot(track_number,slot);
}

Track *instrument_track=NULL;
struct timeval start_of_instrument_track;;
 
/* At this moment there can be only one instrument track
Type of virtual instrument 0=flute, 1=drum, 2=ud
*/
void meam_add_instrument_track(int type){
instrument_track=create_instrument_track(type); 
  if (instrument_track==NULL) {
    printf("made instrument track\n");  
  } else {
    printf("There was instrument track already, new created.\n");
  }

gettimeofday(&start_of_instrument_track, NULL);
}

/*octaves are from 0 to 7.
It is defined on virtual-instrument plugin*/
int octave=3;

void meam_octave_up() {
if (octave<=6)
  octave++;
printf("meam.c: octave is now %d\n",octave);
}

void meam_octave_down(){
if (octave>=1)
  octave--;
printf("meam.c: octave is now %d\n",octave);
}

#define DEFAULT_INSTRUMENT 0
void meam_state_of_note(gboolean state, char note) {
if (instrument_track==NULL){
  meam_add_instrument_track(DEFAULT_INSTRUMENT);
}

gchar *buffer0;
buffer0 = g_strdup_printf("%c%d",note,octave);
//printf("toggling %s\n",buffer0);

if (state)
  g_object_set(G_OBJECT(instrument_track->instrument_element), "note-on", buffer0, NULL);
else
  g_object_set(G_OBJECT(instrument_track->instrument_element), "note-off", buffer0, NULL);
}

void meam_toggle_note_on_octave(char note, int new_octave){
octave=new_octave;
meam_toggle_note(note);
}

/*
 Returns time difference between 
 start_of_audio_track_initialization and current time
*/
struct timeval calc_timestamp () {
  struct timeval result;

  struct timeval now;
  gettimeofday(&now, NULL);

  if (now.tv_usec < start_of_instrument_track.tv_usec) {
    int nsec = (start_of_instrument_track.tv_usec - now.tv_usec) / 1000000 + 1;
    start_of_instrument_track.tv_usec -= 1000000 * nsec;
    start_of_instrument_track.tv_sec += nsec;
  }
  if (now.tv_usec - start_of_instrument_track.tv_usec > 1000000) {
    int nsec = (now.tv_usec - start_of_instrument_track.tv_usec) / 1000000;
    start_of_instrument_track.tv_usec += 1000000 * nsec;
    start_of_instrument_track.tv_sec -= nsec;
  }

  /* Compute the time.
     tv_usec is certainly positive. */
  result.tv_sec = now.tv_sec - start_of_instrument_track.tv_sec;
  result.tv_usec = now.tv_usec - start_of_instrument_track.tv_usec;

  return result;
}


/*Toggling note On and Off*/
void meam_toggle_note(char note) {
if (instrument_track==NULL){
  meam_add_instrument_track(DEFAULT_INSTRUMENT);
}
gchar *buffer0;
buffer0 = g_strdup_printf("%c%d",note,octave);
//printf("toggling %s\n",buffer0);

g_object_set(G_OBJECT(instrument_track->instrument_element), "note-toggle", buffer0, NULL);

struct timeval timestamp;
timestamp=calc_timestamp();
printf("MEAM: %ld.%ld: note %c%d toggled\n", timestamp.tv_sec,timestamp.tv_usec,note,octave); 
}


void meam_print_note_table(){
gchar *buffer0;
g_object_get(G_OBJECT(instrument_track->instrument_element), "note-toggle", &buffer0, NULL);
printf("%s\n",buffer0);
}

void meam_get_all_slot_info(int track_number){
struct Track *trc=get_nth_track_from_pipeline(track_number);
if (trc==NULL) {
  printf("asked track is NULL\n");
  return;
}

int i;
for (i=0;i<SLOT_LIMIT;i++)
  {
  const gchar *location;
  SlotInfo *si= trc->slot_info[i];
  Sample *sample = si->sample;
  if (sample!=NULL){
    location = sample->location;
    printf("%2d: '%d' = '%s'\n",i,trc->slot_info[i]->info, location);
    }
  }
}

void meam_play_one_sample(char* filename){
    stand_alone_play_one_sample(filename);
}

void meam_stop_one_sample(){
    stand_alone_stop_one_sample();
}


void meam_slider_state(gboolean b){
g_object_set(G_OBJECT(instrument_track->instrument_element), "slider-state", b, NULL);
}

void meam_slider_freq(int a){
gfloat a2=(gfloat)a+100;
g_object_set(G_OBJECT(instrument_track->instrument_element), "slider-freq", a2, NULL);
}



guint meam_get_meter_of_track(int track_number){
return get_nth_track_from_pipeline(track_number)->meter;
}

guint meam_get_tempo_of_track(int track_number){
return get_nth_track_from_pipeline(track_number)->tempo;
}

