/**pipeline.c is part of JamMo.
License: GPLv2, read more from COPYING

We have one pipeline which contains all tracks. When pipeline is
started all tracs are played.
*/
#include "pipeline.h"
#include "../../chum/chum.h"    //chum_report_melodic_contour(float freq)
#include <string.h>            //we use strcmp
//private functions
static gboolean my_bus_callback(GstBus *bus, GstMessage *msg, gpointer data);

// private Variables
static Pipeline *mainPipeline;

/**
 * Pipeline_init function allocates the needed memory for pipeline and
 * then initializes it. After this stage we have our pipeline ready for
 * tracks to be added in it 
 * @return TRUE if pipeline initialization was successful else FALSE
 */
gboolean pipeline_init(void) 
{ 	
	mainPipeline = (Pipeline*) malloc(sizeof(Pipeline));
	if(mainPipeline != NULL) 
	{
		mainPipeline->pipeline = gst_pipeline_new("pipeline");
		mainPipeline->bus = gst_pipeline_get_bus(GST_PIPELINE(mainPipeline->pipeline));
		gst_bus_add_watch(mainPipeline->bus, my_bus_callback, NULL);
		gst_object_unref(mainPipeline->bus);

		gst_element_set_state(GST_ELEMENT(mainPipeline->pipeline), GST_STATE_PAUSED);	

		mainPipeline->tracks = NULL;	

		mainPipeline->adder = gst_element_factory_make("adder", "adder");
		if(!mainPipeline->adder)
	    		g_print("Couldn't create adder element!\n");

		mainPipeline->audiosink = gst_element_factory_make("autoaudiosink", "sink");
		if(!mainPipeline->audiosink)
	    		g_print("Couldn't create adder element!\n");

		gst_bin_add_many(GST_BIN(mainPipeline->pipeline), mainPipeline->adder, mainPipeline->audiosink, NULL);

		gst_element_link(mainPipeline->adder, mainPipeline->audiosink);
	  	if(GST_PAD_LINK_SUCCESSFUL(GST_PAD_LINK_OK))
	    		g_print("Linking successfully completed (Adder -> Sink)\n");
	  	else 
	    		g_print("Linking failed! (Adder -> Sink)\n");

		mainPipeline->pipelines_end_time = 0LLU; //Long long unsigned for 64bit integer
	}	
	else
		return FALSE;

	return TRUE;
}

/**
 *  free_pipeline - This function is responsible of freeing all the allocated memory within
 *  the mainPipeline. This function also frees the memory from dynamically allocated track objects
 *  using the function free_track for every track object in GList. 
 */
void free_pipeline(void)
{
	// Free all the allocated memory from tracks and then free the GList itself
	if(mainPipeline->tracks != NULL) {
		g_list_foreach(mainPipeline->tracks, (GFunc)free_track, NULL);
		g_list_free(mainPipeline->tracks);	
	}

	// Frees mainPipeline
	free(mainPipeline);
	mainPipeline = NULL;
	g_print("Pipeline - allocated memory freed\nNow closing program...\n");
}

/**
 *  get_mainPipeline - Getter function for getting the pointer to the mainPipeline structure
 *  @return Pipeline* - a pointer to the mainPipeline struct
 */
Pipeline* get_mainPipeline(void)
{
	return mainPipeline;
}



/**
 *  my_bus_callback handles debug information about the pipeline. It 
 *  catches different messages like EOS (end of stream) and notifies if 
 *  some other errors do occur in the pipeline.
 */
static gboolean my_bus_callback (GstBus *bus, GstMessage *msg, gpointer data)
{
	//g_print ("Got %s message\n", GST_MESSAGE_TYPE_NAME (msg));
	switch (GST_MESSAGE_TYPE (msg)) {
	case GST_MESSAGE_EOS:
      		g_print ("End of stream\n");
      		//g_main_loop_quit (loop);
      		break;

	/* //USED FOR DEBUGGING
	case GST_MESSAGE_STATE_CHANGED: {
     		GstState state , pending;
      		gst_message_parse_state_changed (msg, NULL, &state, &pending);
      		g_print("SEQUENCER_BUS: %s is %s, pending %s\n", GST_ELEMENT_NAME(GST_MESSAGE_SRC(msg)), 
			gst_element_state_get_name(state), gst_element_state_get_name(pending));
		break;
	}
	*/
    	case GST_MESSAGE_ERROR: {
      		gchar  *debug;
      		GError *error;

      		gst_message_parse_error (msg, &error, &debug);
      		g_free (debug);

      		g_printerr ("Error: %s\n", error->message);
      		g_error_free (error);

      		//g_main_loop_quit (loop);
      		break;
	}
	case GST_MESSAGE_ELEMENT: {
	    const GstStructure *structure = gst_message_get_structure (msg);
	    const gchar *name = gst_structure_get_name (structure);

	    if (strcmp (name, "pitchdetect") == 0 ) {
	       float frequency;
	       frequency = g_value_get_float( gst_structure_get_value (structure, "frequency") );
	       //printf("pipeline:freq=%f\n",frequency);
	       //gui_visualize_contour(frequency); //This is from gui.h
	       chum_report_melodic_contour(frequency);
	       break;
	    }
	}
    	default:
      		break;
	}

  return TRUE;
}

