/**composing_game.c is part of JamMo.
License: GPLv2, read more from COPYING

This file is for clutter based gui.
This is the composing game. It contains theme selection and game playing.
*/

#include <glib-object.h>
#include <gtk/gtk.h>
#include <clutter/clutter.h>


#include "jammo-texture.h"
#include "clutter_jammo.h"
#include "../chum/chum.h" //THEMES_DIR
#include "object.h"
#include "config.h"
#include "button.h"
#include "preloader.h"

GList* actors_in_track = NULL;

GList* composing_game_get_actors_in_track(){
 return actors_in_track; 
}

ClutterActor* timeline_star_actor;
ClutterActor* objects_in_this_stage;

static void on_timeline_completed(ClutterTimeline *timeline,  gpointer data) {
  //printf("star reached end\n");
  clutter_actor_hide(timeline_star_actor);
  //chum_stop_sequencer();
}

static void on_timeline_marker_reached(ClutterTimeline *timeline,  gchar *name, gint frame, gpointer data) {
  //printf("frame_number:%s. next_key_frame %d\n",frame_number,times[timeline_index]);
  int X = atoi(name);
  printf("X is %d\n",X);
  clutter_actor_set_position (timeline_star_actor, X , Y_PLACE_OF_TIMELINE);
}

void composing_game_visualize_playing() {
  printf("composing_game: visualize_playing starts!\n");

  //timeline_star is initialized from start_theme(). If we want play_sequencer before that
  //(e.g. testing virtual instrument) we want skip star visualization.
  if (timeline_star_actor==NULL)
    return;

  guint64 length_of_song;
  chum_get_duration_of_song(&length_of_song);

  guint64 a0 = 60000;
  guint64 a1= 1000000;
  guint64 a2=a0*a1;
  gint64 length_of_beat = a2 * GST_NSECOND * chum_get_meter_of_track(0)/chum_get_tempo_of_track(0);

  printf ("composing_game: Duration: %" GST_TIME_FORMAT " .one beat: %" GST_TIME_FORMAT "\n",
  GST_TIME_ARGS (length_of_song), GST_TIME_ARGS (length_of_beat));

  //printf("composing_game: start_star timeline\n");
  int length_ms=(int)(length_of_song/1000000);
  ClutterTimeline *timeline = clutter_timeline_new (length_ms);

  clutter_timeline_set_loop(timeline, FALSE); 
  g_signal_connect (timeline, "marker-reached", G_CALLBACK (on_timeline_marker_reached),timeline_star_actor);
  g_signal_connect (timeline, "completed", G_CALLBACK (on_timeline_completed),timeline_star_actor);

  gint64 i;
  int place=0;
  for (i=0;i<length_of_song;i+=length_of_beat) {
      gchar *place_ =g_strdup_printf("%d",place);

      guint time_ns = i / GST_NSECOND /1000000;
      //printf("time(ns):%d, place%d\n",time_ns,place);
      clutter_timeline_add_marker_at_time (timeline, place_, time_ns);
      place+=WIDTH_OF_SLOT;
  }



  clutter_actor_show (timeline_star_actor);
  clutter_actor_raise_top(timeline_star_actor);
  clutter_actor_set_position (timeline_star_actor, 0 , Y_PLACE_OF_TIMELINE);

printf("composing_game: start_star timeline\n");
clutter_timeline_start(timeline);
}




ClutterActor* drop_area=NULL;
ClutterActor* get_drop_area()
{
	return(drop_area);
}


GList *current_behaviours_and_actors = NULL;
GList *get_current_behaviours_and_actors(){
return current_behaviours_and_actors;
}

/**
 * stage_start_behaviour() is used to initialize and start timeline
 * behaviours.
 * 
 * This is for objects and buttons in config.json
 *
 * @param		actor			actor for behaviour
 * @param		behaviour		behaviour which is used
 */
void stage_start_behaviour(ClutterActor* actor, ClutterBehaviour* behaviour)
{
	ClutterAlpha* alpha;
	ClutterTimeline* timeline;
	
	g_return_if_fail(CLUTTER_IS_ACTOR(actor));
	g_return_if_fail(CLUTTER_IS_BEHAVIOUR(behaviour));
	
	alpha = clutter_behaviour_get_alpha(behaviour);
	timeline = clutter_alpha_get_timeline(alpha);
	clutter_timeline_start(timeline);

	gchar *name;
	g_object_get(G_OBJECT(actor), "name", &name, NULL);
	if (name==NULL) //objects doesn't have names. buttons have
	  {
	  ActorBehaviour *ab = malloc(sizeof(ActorBehaviour));
	  ab->behaviour=behaviour;
	  ab->actor=actor;
	  current_behaviours_and_actors = g_list_append(current_behaviours_and_actors, ab); 
	  }
//clutter_behaviour_remove (behaviour,actor); //this helps debugging
}


/**
 * @param		stage_nr	number of stage
 * @return		actor		background actor of asked stage. NULL if asked too big index
 */
ClutterActor* load_background(const char* theme_folder)
{
	ClutterActor* actor = clutter_texture_new ();;
	gchar* filepath1;
	gchar* dir = THEMES_DIR;
	filepath1 = g_strdup_printf("%s/%s/background.png", dir, theme_folder);

        g_object_set(G_OBJECT(actor), "load-async", TRUE, NULL);
        clutter_texture_set_from_file (CLUTTER_TEXTURE (actor), filepath1, NULL);

	return actor;
}

gboolean composing_game_is_game_initialized(){
  return (drop_area != NULL); //Last initialized actor.
}

GList *themes = NULL;
/**
 * stage_init_clutter_stage() function is used to initialize the
 * stage given as parameter. Function sets the corresponding
 * background for the stage in order. Function also initializes
 * all stage relevant actors and all buttons to objects_group
 * (ClutterGroup).
 * 
 */
void start_theme() {
    if (!composing_game_is_game_initialized())
        preloader_init_this(SCREEN_COMPOSING);

    actors_in_track = NULL; //make this empty

    //List containing lists.
    GList *actors = (GList*)g_list_nth_data (themes,chum_get_current_theme()-1); //Themes started from 1, but GList from 0

    if (actors==NULL) printf("actors is NULL\n");


    GList *l;
    for (l = actors; l; l = l->next)
        {
        ClutterActor *actor = CLUTTER_ACTOR(l->data);
        gchar *name;
        g_object_get(G_OBJECT(actor), "name", &name, NULL);
        printf("name is %s\n",name);
        clutter_actor_set_reactive(actor, TRUE);
        clutter_actor_raise_top(actor);
        clutter_actor_show (actor);
        }

    clutter_actor_raise_top(drop_area);
    clutter_actor_show (drop_area);
}


/**
 * Initialization function for composing games
 */
void composing_game_init_games() {
    ClutterActor* stage = main_getStage();

     int number_of_stages=chum_get_number_of_themes();
    //Then we take background image for each and make theme-selection
    int i;
    for (i=1;i<number_of_stages+1;i++)
    //for (i=1;i<1+1;i++)
        {
        GList *actors_in_this_theme = NULL;

        ClutterActor*  bgr = load_background(chum_get_theme_folder(i));
        clutter_actor_set_name(bgr, "background");
        //if (bgr==NULL)    printf("Background is NULL\n");
        clutter_actor_set_position (bgr, 0, 0);
        clutter_actor_hide (bgr);
        clutter_actor_set_reactive(bgr, FALSE);
        clutter_container_add_actor (CLUTTER_CONTAINER (stage), bgr);

        actors_in_this_theme = g_list_append(actors_in_this_theme,  CLUTTER_ACTOR (bgr));

        ClutterActor* objects_group;
        objects_group = clutter_group_new();
        object_create_stage_object_group (objects_group, chum_get_theme_folder(i));

        button_create_buttons (objects_group);

        //ClutterGroup is not passed forward. Unroll it to GList.
        int j;
        int max=clutter_group_get_n_children (CLUTTER_GROUP(objects_group)); //this number decrease in loop
        for (j=0;j<max;j++)
          {
          ClutterActor* actor =clutter_group_get_nth_child (CLUTTER_GROUP(objects_group),0); //always take first (it is then removed from list)
          if (actor==NULL) printf("actor==NULL, j=%d\n",j);
          clutter_actor_set_reactive(actor, FALSE);
          g_object_ref (actor); //removing from parent will delete this otherwise
          clutter_container_remove_actor (CLUTTER_CONTAINER (objects_group), actor);
          clutter_actor_hide(actor);
          clutter_container_add_actor (CLUTTER_CONTAINER (stage), actor);
          actors_in_this_theme = g_list_append(actors_in_this_theme,  CLUTTER_ACTOR (actor));
          //gchar *name; name =clutter_actor_get_name(actor);
          }
 
    themes = g_list_append(themes,  (GList*) actors_in_this_theme);
    }

    timeline_star_actor = config_get_configured_actor(NULL,"timeline_star");
    clutter_actor_set_name(timeline_star_actor, "timeline_star_actor");
    clutter_actor_set_reactive(timeline_star_actor, FALSE);
    clutter_actor_hide (timeline_star_actor);
    clutter_container_add_actor(CLUTTER_CONTAINER(stage), timeline_star_actor);

    drop_area = config_get_configured_actor(NULL,"drop-area");
    clutter_actor_set_name(drop_area, "drop_area");
    clutter_actor_set_reactive(drop_area, FALSE);
    clutter_actor_hide (drop_area);
    clutter_container_add_actor (CLUTTER_CONTAINER (stage), drop_area);
}


//////////////////////
//Theme selection/////
//////////////////////
static void theme_selection_mouse_release (JammoTexture *actor, ClutterEvent *event, gpointer data){
int value = GPOINTER_TO_INT(data);
chum_change_theme(value);
chum_go_screen(SCREEN_COMPOSING);
}


static void from_theme_selection_to_game_selection(){
chum_go_back();
}

GList *group; /*There are variable number of themes and border lines.*/
ClutterActor *borders;  /*This is outer border*/
ClutterActor *back_to_mainmenu; /*Button back to main_menu.*/

gboolean composing_game_is_menu_initialized(){
  return (back_to_mainmenu != NULL); //Last initialized actor.
}

void show_theme_selection() {
    if (!composing_game_is_menu_initialized())
        preloader_init_this(SCREEN_THEME_SELECTION);

    GList *l;
    for (l = group; l; l = l->next){
        ClutterActor *actor = CLUTTER_ACTOR(l->data);
        clutter_actor_set_reactive(actor, TRUE);
        clutter_actor_raise_top(actor);
        clutter_actor_show (actor);
    }

    clutter_actor_raise_top(borders);
    clutter_actor_show (borders);

    clutter_actor_set_reactive(back_to_mainmenu, TRUE);
    clutter_actor_raise_top(back_to_mainmenu);
    clutter_actor_show (back_to_mainmenu);
}

void composing_game_init_theme_selection() {
  ClutterActor* stage = main_getStage();
  group = NULL;

  //First we count how many themes we have in theme-folder
  int number_of_stages=chum_get_number_of_themes();
  //Then we take background image for each and make theme-selection
  int i;
  for (i=1;i<number_of_stages+1;i++)
      {
      ClutterActor*  actor= load_background(chum_get_theme_folder(i));
      //if (actor==NULL)   printf("Background is NULL\n");

      //there are empty white area bottom of background:
      //first make image taller and then clip from bottom
      clutter_actor_set_height (actor,700); //FIXED NUMBER. FIXME
      clutter_actor_set_clip (actor,WIDTH_OF_WINDOW/number_of_stages,0,WIDTH_OF_WINDOW/number_of_stages,HEIGHT_OF_WINDOW);
      clutter_actor_set_position (actor,WIDTH_OF_WINDOW/number_of_stages*(i-2),0);		

      clutter_container_add_actor (CLUTTER_CONTAINER (stage), actor);
      clutter_actor_hide (actor);
      clutter_actor_set_reactive(actor, FALSE);
      //clutter_actor_set_name(actor, "");
      g_signal_connect (actor, "button-press-event", G_CALLBACK (theme_selection_mouse_release),GINT_TO_POINTER(i));

      //Add it to list.
      group = g_list_append( group,  CLUTTER_ACTOR (actor) );

      //make vertical bar between each theme
      ClutterActor* line=clutter_rectangle_new_with_color (get_transparency_color());
      clutter_rectangle_set_border_color(CLUTTER_RECTANGLE(line),get_dark_color());
      clutter_rectangle_set_border_width(CLUTTER_RECTANGLE(line),10);
      clutter_actor_set_size (line, 10, HEIGHT_OF_WINDOW);
      clutter_actor_set_position (line, WIDTH_OF_WINDOW/number_of_stages*(i-1), 0);
      clutter_actor_hide (line);
      clutter_container_add_actor (CLUTTER_CONTAINER (stage), line);

      group = g_list_append( group,  CLUTTER_ACTOR (line) );
      }


  //border all over the theme-selection
  borders=clutter_rectangle_new_with_color (get_transparency_color());
  clutter_actor_set_size (borders, WIDTH_OF_WINDOW,HEIGHT_OF_WINDOW);
  clutter_actor_set_position (borders, 0, 0);
  clutter_rectangle_set_border_color(CLUTTER_RECTANGLE(borders),get_dark_color());
  clutter_rectangle_set_border_width(CLUTTER_RECTANGLE(borders),10);
  clutter_actor_hide (borders);
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), borders);

  //Back to main-menu button.
  back_to_mainmenu = config_get_configured_actor(NULL,"house");
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), back_to_mainmenu);
  clutter_actor_set_position (back_to_mainmenu, 720, 5); 
  clutter_actor_set_size (back_to_mainmenu, 66, 60);
  clutter_actor_hide (back_to_mainmenu);
  clutter_actor_set_name(back_to_mainmenu, "back_to_mainmenu");
  clutter_actor_set_reactive(back_to_mainmenu, FALSE);
  g_signal_connect (back_to_mainmenu, "button-press-event", G_CALLBACK (from_theme_selection_to_game_selection),NULL);
}
