/** chum.c is part of JamMo.
License: GPLv2, read more from COPYING

(from D2.4)
"CHUM controls the user interface and will communicate with all other modules. 
It will use MEAM to create music. It will use GEMS to identify and authorize
the user and to send data to other users and teacher's server. It also sends
all the user actions to CEM fo logging/mentor input/game input."
*/

#include <fcntl.h> //we check some files
#include <string.h> //we use strcmp
#include <pthread.h> // we use threads when debugging

#include "gui.h"
#include "chum.h"
#include "file_helper.h"

#include "../cem/cem.h"
#include "../meam/meam.h"

#include "../gems/gems.h"

static int chum_count_themes();
static void chum_play_recorded_and_backing_track();

/*Chum is like state-machine*/
int state_of_chum=SCREEN_WELCOME;
int current_theme; /*for composing game*/
int current_song;  /*for karaoke game*/

gint number_of_themes = 0;
gint chum_get_number_of_themes(){
return number_of_themes;
}

int chum_get_state_of_chum(){
return state_of_chum;
}

//THIS IS THE MAIN LOOP OF JAMMO!!!
void chum_start_jammo_main_loop() {
    cem_add_to_log("starting main loop",LOG_DEBUG);
    //meam_start_gstreamer_loop();

    //This is not good for jammoterminal but it works.
    //gtk_main();
    gui_start_main_loop();
    //clutter_init();
    //clutter_main();
    cem_add_to_log("main loop is over",LOG_DEBUG);
}


void *debug_test_wait_and_start_seq(){
  printf("wait_and_start_seq\n");
  g_usleep( 1 * G_USEC_PER_SEC );
  chum_play_sequencer();
  pthread_exit(NULL);
}

void debug_test_minimal_recording(int different_way) {
    int song_number=4;
    cem_add_to_log(g_strdup_printf("debug_test_minimal_recording(%d)",different_way),LOG_DEBUG);
    current_song=song_number;

    const gchar *song_folder=chum_get_song_folder(song_number);
    gchar *filename = g_strdup_printf("%s/comping.ogg", song_folder);
    
    if (different_way==0)
      {
      cem_add_to_log(g_strdup_printf("making new_empty_track and put sample to it"),LOG_DEBUG);
      chum_create_new_empty_track(); //for comping track
      chum_add_new_sample_to_track_nth_slot(0,filename,0);
      }
    else
      {
      cem_add_to_log(g_strdup_printf("making own track type _without_ gnonlin."),LOG_DEBUG);
      create_new_backing_track(filename);
      }


    create_new_singing_track();

    pthread_t loop_thread;
    pthread_create(&loop_thread, NULL, debug_test_wait_and_start_seq, NULL);

    GMainLoop *loop;
    loop = g_main_loop_new (NULL, FALSE);
    g_main_loop_run (loop);
    //This is useless:
    pthread_exit(NULL);
}


void chum_start_chum(int argc, char** argv){
    int i;
    cem_add_to_log("chum_start_chum()",LOG_DEBUG);
    for (i=1;i<argc;i++) 
      {
      cem_add_to_log(g_strdup_printf("chum_start_chum(%s)",argv[i]),LOG_DEBUG);
      }
    //init for sequencer
    meam_init();
    number_of_themes=chum_count_themes();

    if (argc==2 && strcmp(argv[1],"--help")==0) {
        printf("\nJamMo help:\n\n");
        printf("flags:\n");
        printf("editor  \t- start midi-editor\n");
        printf("theme n \t- start composing game in theme n\n\n");
        exit(0);
        }

    else if (argc==2 && strcmp(argv[1],"debug0")==0) { 
        debug_test_minimal_recording(0);
        exit(0);
        }
    else if (argc==2 && strcmp(argv[1],"debug1")==0) { 
        debug_test_minimal_recording(1);
        exit(0);
        }



    gui_init();
    if (argc==2 && strcmp(argv[1],"sing")==0) { 
        state_of_chum=SCREEN_SONG_SELECTION;
        }

    else if (argc==2 && strcmp(argv[1],"editor")==0) { 
        state_of_chum=SCREEN_MIDI_EDITOR;
        }

    else if (argc==2 && strcmp(argv[1],"theme")==0) { 
        state_of_chum=SCREEN_THEME_SELECTION;
        }


    else if (argc==3 && strcmp(argv[1],"theme")==0) { 
        chum_change_theme(atoi(argv[2]));
        state_of_chum=SCREEN_COMPOSING;
        }


    gui_go_screen(state_of_chum);

    chum_start_jammo_main_loop();
}

void chum_go_back() {
    cem_add_to_log(g_strdup_printf("chum_go_back(). state_of_chum was %d...",state_of_chum),LOG_DEBUG);

    switch (chum_get_state_of_chum()) {
      //singing
      case SCREEN_SONG_SELECTION:
	  state_of_chum=SCREEN_GAME_SELECTION;
	  break;
      case SCREEN_SONG_RECORDING:
	  state_of_chum=SCREEN_SONG_SELECTION;
	  break;
      case SCREEN_PLAY_RECORDED:
	  state_of_chum=SCREEN_SONG_RECORDING;
	  break;

      //composing
      case SCREEN_COMPOSING:
	  state_of_chum=SCREEN_THEME_SELECTION;
	  break;
      case SCREEN_THEME_SELECTION:
	  state_of_chum=SCREEN_GAME_SELECTION;
	  break;
      }


    cem_add_to_log(g_strdup_printf("...chum_go_back(). new state_of_chum is %d",state_of_chum),LOG_DEBUG);

    gui_go_screen(state_of_chum);
}


void chum_go_next() {
    cem_add_to_log(g_strdup_printf("chum_go_next(). state_of_chum was %d...",state_of_chum),LOG_DEBUG);

    //singing game is handled with switch-case. Composing game doens't have such a thing like next.
    switch (chum_get_state_of_chum()) {
      case SCREEN_SONG_RECORDING:
	  state_of_chum=SCREEN_PLAY_RECORDED;
	  chum_play_recorded_and_backing_track();
	  break;

      case SCREEN_PLAY_RECORDED:
	  state_of_chum=SCREEN_SONG_SELECTION;
	  break;
      default:
	state_of_chum++;
	break;
    }

    cem_add_to_log(g_strdup_printf("...chum_go_next(). new state_of_chum is %d",state_of_chum),LOG_DEBUG);

    gui_go_screen(state_of_chum);
}

void chum_go_screen(int screen_number){
    cem_add_to_log(g_strdup_printf("chum_go_screen(%d)",screen_number),LOG_DEBUG);
    state_of_chum=screen_number;
    gui_go_screen(state_of_chum);
}

//STUB
void chum_change_to_game(int game_number){
    cem_add_to_log(g_strdup_printf("chum_change_to_game(%d)",game_number),LOG_DEBUG);
    state_of_chum=SCREEN_THEME_SELECTION;
}

int chum_get_current_theme(){
    cem_add_to_log(g_strdup_printf("get_current_theme(). will return %d",current_theme),LOG_DEBUG);
    return current_theme;
}

void chum_create_new_empty_track() {
  cem_add_to_log(g_strdup_printf("chum_create_new_empty_track()"),LOG_DEBUG);
  meam_create_new_empty_track();
}
void chum_change_theme(int theme_number){
    cem_add_to_log(g_strdup_printf("chum_change_theme(%d)",theme_number),LOG_DEBUG);
    current_theme=theme_number;

    g_list_foreach(get_mainPipeline()->tracks, (GFunc)remove_track, NULL);
    chum_create_new_empty_track(); //for samples
    chum_create_new_empty_track(); //for backing track


    //Add backing track if there are.
    gchar* filepath1 = g_strdup_printf("%s/%s/backing_track.wav", THEMES_DIR, chum_get_theme_folder(chum_get_current_theme()));
    int fd = open(filepath1,O_RDONLY);
    if (fd != -1)
	meam_seq_add_new_sample_to_track(1,filepath1,0); //trac-number,file,starting position

    state_of_chum=SCREEN_COMPOSING;
    //gui_go_screen(state_of_chum);
}

void chum_change_to_theme(int theme_number){
chum_change_theme(theme_number);
    state_of_chum=SCREEN_COMPOSING;
    gui_go_screen(state_of_chum);
}

/* If this is needed, it should be moved to track.c or pipeline.c
It doesn't call free_track, so it can waste memory */
void remove_track (Track *trc){
printf("remove_track begin\n");
get_mainPipeline()->tracks =  g_list_remove(get_mainPipeline()->tracks, trc);
	gst_bin_remove(GST_BIN(get_mainPipeline()->pipeline), trc->trackbin);
}

int chum_get_current_song(){
cem_add_to_log(g_strdup_printf("get_current_song(). will return %d",current_song),LOG_DEBUG);
    return current_song;
}

const gchar *chum_get_song_folder( int i){
gchar *name =  (gchar *) g_list_nth_data(chum_get_all_songs(),i); //FIXME. now it checks it everytime in disk
return name;
}


/*STATIC
Always call this before gui_go_screen(SCREEN_PLAY_RECORDED);
*/
static void chum_play_recorded_and_backing_track(){
    chum_stop_sequencer();
    g_list_foreach(get_mainPipeline()->tracks, (GFunc)remove_track, NULL);


    chum_create_new_empty_track(); //for comping track
    const gchar *song_folder=chum_get_song_folder(chum_get_current_song());
    gchar *filename = g_strdup_printf("%s/comping.ogg", song_folder);
    chum_add_new_sample_to_track_nth_slot(0,filename,0);


    chum_create_new_empty_track(); //for recorded singing
    chum_add_new_sample_to_track_nth_slot(1,REC_TARGET,0);
}


void chum_change_to_song(int song_number){
    cem_add_to_log(g_strdup_printf("chum_change_to_song(%d)",song_number),LOG_DEBUG);
    current_song=song_number;

    meam_seq_remove_samples_from_track(0);
    meam_seq_remove_samples_from_track(1);

    const gchar *song_folder=chum_get_song_folder(song_number);

    g_list_foreach(get_mainPipeline()->tracks, (GFunc)remove_track, NULL);
    chum_create_new_empty_track(); //for comping track
    gchar *filename = g_strdup_printf("%s/comping.ogg", song_folder);
    chum_add_new_sample_to_track_nth_slot(0,filename,0);

    create_new_singing_track();
    state_of_chum=SCREEN_SONG_RECORDING;
    gui_go_screen(state_of_chum);
}

void chum_play_one_sample(char* filename){
    cem_add_to_log(g_strdup_printf("chum_play_one_sample(%s)",filename),LOG_DEBUG);
    meam_play_one_sample(filename);
}

void chum_stop_one_sample(){
    cem_add_to_log("chum_stop_one_sample()",LOG_DEBUG);
    meam_stop_one_sample();
}

void chum_stop_sequencer(){
    cem_add_to_log("chum_stop_sequencer()",LOG_DEBUG);
    meam_seq_stop();
}

void chum_play_sequencer(){
    chum_stop_sequencer();
    cem_add_to_log("chum_play_sequencer()",LOG_DEBUG);
    gui_visualize_playing(state_of_chum);
    meam_seq_play();
}

void chum_report_melodic_contour(float freq){
gui_visualize_contour(freq);
}


void chum_get_duration_of_song(guint64 *value) {
    meam_get_duration_of_song(value);
}

//FIX: implement this.
//void chum_pause_sequencer();

Sample *chum_add_new_sample_to_track(int track_number, const gchar srcLocation[], guint64 startTime){
    cem_add_to_log(g_strdup_printf("chum_add_new_sample_to_track(%d,%s,%" GST_TIME_FORMAT ")",track_number, srcLocation, GST_TIME_ARGS (startTime)),LOG_DEBUG);
    return meam_seq_add_new_sample_to_track(track_number, srcLocation, startTime);
}


Sample *chum_add_new_sample_to_track_nth_slot(int track_number, const gchar srcLocation[], guint slot){
    cem_add_to_log(g_strdup_printf("chum_add_new_sample_to_track_nth_slot(%d,%s,%d)",track_number, srcLocation, slot),LOG_DEBUG);
    return meam_seq_add_new_sample_to_track_nth_slot(track_number, srcLocation, slot);
}


void chum_remove_sample_from_slot(guint track_number,int slot){
cem_add_to_log(g_strdup_printf("chum_remove_sample_from_slot(track=%d,slot=%d)",track_number,slot),LOG_DEBUG);
meam_remove_sample_from_slot(track_number,slot);
}

//FIX: we should not use fixed value
#define MAX_NUMBER_OF_THEMES 10
const gchar* theme_folders[MAX_NUMBER_OF_THEMES];

const gchar* chum_get_theme_folder(int index){
//cem_add_to_log(g_strdup_printf("chum_get_theme_folder(%d). we will return %s",index,theme_folders[index]),LOG_DEBUG);
    return theme_folders[index];
}



static int chum_count_themes(){
    g_return_val_if_fail(THEMES_DIR,-1);
    GError* error = NULL;
    GDir* dir = g_dir_open (THEMES_DIR, 0, &error);
    if (error) {
	g_warning ("g_dir_open() failed: %s\n", error->message);
	g_clear_error (&error);
	return -1;
    }

    int number_of_themes = 0;
    const gchar* filename = NULL;
    while ((filename = g_dir_read_name(dir))) {
	if (filename[0]!='.') { //do not count hidden files (folders)
	      number_of_themes++;
	      theme_folders[number_of_themes]=filename;
	}
   }
return number_of_themes;
}

void chum_get_all_slot_info(int track_number){
cem_add_to_log(g_strdup_printf("chum_get_all_slot_info(%d)",track_number),LOG_DEBUG);
meam_get_all_slot_info(track_number);
}

void chum_add_instrument_track(int type) {
cem_add_to_log(g_strdup_printf("chum_add_instrument_track(%d)",type),LOG_DEBUG);
meam_add_instrument_track(type);
}

void chum_state_of_note(gboolean state, char note){
cem_add_to_log(g_strdup_printf("chum_state_of_note(%d,%c)",state,note),LOG_DEBUG);
meam_state_of_note( state,  note);
}

void chum_toggle_note(char note){
cem_add_to_log(g_strdup_printf("chum_toggle_note(%c)",note),LOG_DEBUG);
meam_toggle_note(note);
}

void chum_toggle_note_on_octave(char note, int octave){
cem_add_to_log(g_strdup_printf("chum_toggle_note_on_octave(%c,%d)",note,octave),LOG_DEBUG);
meam_toggle_note_on_octave(note,octave);
}

void chum_print_note_table(){
cem_add_to_log(g_strdup_printf("chum_print_note_table()"),LOG_DEBUG);
meam_print_note_table();
}


void chum_octave_up() {
cem_add_to_log(g_strdup_printf("chum_octave_up()"),LOG_DEBUG);
meam_octave_up();
}

void chum_octave_down(){
cem_add_to_log(g_strdup_printf("chum_octave_down()"),LOG_DEBUG);
meam_octave_down();
}

GList* chum_get_all_songs(){
cem_add_to_log(g_strdup_printf("chum_get_all_songs()"),LOG_DEBUG);
return file_helper_get_all_songs();
}



guint chum_get_meter_of_track(int track_number){
guint meter=meam_get_meter_of_track(track_number);
cem_add_to_log(g_strdup_printf("chum_get_meter_of_track(%d) [%u]",track_number,meter),LOG_DEBUG);
return meter;
}

guint chum_get_tempo_of_track(int track_number){
guint tempo=meam_get_tempo_of_track(track_number);
cem_add_to_log(g_strdup_printf("chum_get_tempo_of_track(%d) [%u]",track_number,tempo),LOG_DEBUG);
return tempo;
}
