/**config.c is part of JamMo.
License: GPLv2, read more from COPYING

This file is for clutter based gui and it handles config file reading
*/

#include "config.h"
#include <clutter/clutter.h>
#include <unistd.h> //for getcwd
#include <string.h>
#include "../chum/chum.h" //because of DATA_DIR

/**
 * The default ClutterScript config file.
 */
#define CONFIG_FILE_NAME g_strdup_printf("%s/config.json",DATA_DIR)

/*
Script is loaded everytime when new object is asked. 
If it is global we can't use same behaviour in two different buttons.
 And creating new object-actor doesn't work.
*/

/**
 * config_return_script() returns a ClutterScript from the
 * parameter file given. Function returns NULL value if an
 * error occured.
 * 
 * @param		config_file		JSON-type file
 * @return		script			ClutterScript returned
 */
static ClutterScript* config_return_script (gchar* config_file)
{
	ClutterScript* script;
	char buffer[PATH_MAX];
	gchar* path_file;
	GError* error = NULL;
	
	script = clutter_script_new();
	
	if (!getcwd(buffer, PATH_MAX)) {
		fprintf(stderr, "Error: Current working directory path is too long.\n");
		return NULL;
	}
	
	path_file = g_strdup_printf("%s%s", buffer,config_file);
	//printf("path_file:'%s'\n", path_file);

	if (!clutter_script_load_from_file(script, config_file, &error)) {
		fprintf(stderr, "return_script: Error: Could not load from file '%s' (%s).\n", config_file, error->message);
		g_error_free(error);
		return NULL;
	}
	
	clutter_script_connect_signals(script, NULL);
	
	return script;
}


/**
 * config_get_configured_actor() returns a ClutterActor according
 * to the parameter actor name given. The actor is returned from
 * the default config file, which is specified in the beginning
 * of this file. Function returns NULL value if an error occured.
 * Function uses config_return_script() function from this file.
 * 
 * @param		actor_name		actor wanted to be found
 * @return		actor			actor returned
 */
ClutterActor* config_get_configured_actor (const gchar *theme_folder, gchar* actor_name)
{
	ClutterActor* actor;
	ClutterScript* script;
	
	gchar *buffer;

	if (theme_folder==NULL) {
	    buffer=g_strdup_printf("%s/config.json",DATA_DIR);
	}
	  else{
/*when theme starts, all objects are loaded to object_group (not with this function)
when one object are respawned, this function and this else-branch are used.*/
	    buffer=g_strdup_printf("%s/themes/%s/config.json", DATA_DIR,theme_folder);
	}

	if(!(script = config_return_script(buffer))) {
		fprintf(stderr, "get_configured_actor: Error: Could not find config file.\n");
		return NULL;
	}
	
	if(!(actor = CLUTTER_ACTOR(clutter_script_get_object(script, actor_name)))) {
		fprintf(stderr, "get_configured_actor: Error: Could not find %s from config file.\n", actor_name);
		return NULL;
	}
	
	return actor;
}

GList *config_get_objects_for_theme(const gchar *theme_folder){
GList *actors = NULL;
gchar* type="object";
gchar *buffer;
buffer=g_strdup_printf("%s/themes/%s/config.json", DATA_DIR,theme_folder);

  ClutterScript* script;	
  if(!(script = config_return_script(buffer))) {
    fprintf(stderr, "get_objects_for_theme:Error: Could not find config file.\n");
    return NULL;
  }


  GList *all= clutter_script_list_objects (script);
  GList *l;
  for (l = all; l; l = l->next){
      if (strncmp(clutter_get_script_id(l->data),type,strlen(type)) ==0){
	  actors = g_list_append(actors, l->data); 
	  //printf("%s\n",clutter_get_script_id(l->data) );
      }
  }

return actors;
}






/**
 This function counts how many actor in config-file starts with given string.
 example:
 "button-" how many buttons
 
FIX: only buttons are counted in this way, and they should not be either.
*/
gint count_actors_in_script (gchar* type){
  ClutterScript* script;	

  char *buffer;
  char working_directory[PATH_MAX];
  if (!getcwd(working_directory, PATH_MAX)) {
    fprintf(stderr, "Error: Current working directory path is too long.\n");
    return -1;
  }
  buffer=g_strdup_printf("%s",CONFIG_FILE_NAME);



  if(!(script = config_return_script(buffer))) {
    fprintf(stderr, "Error: Could not find config file.\n");
    return -1;
  }
	
  gint count =0;
  GList *actors= clutter_script_list_objects (script);

  GList *l;
  for (l = actors; l; l = l->next){
    //printf("%s\n",clutter_get_script_id(l->data) );
    if (strncmp(clutter_get_script_id(l->data),type,strlen(type)) ==0){
      //printf("  count this\n");
      count++;
    }
  }
  return count;
}
