
#include <sys/select.h>


#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include "test-teacher-profile-service.h"

volatile gboolean run = TRUE;
volatile gboolean docleanup = FALSE;


serverdata* new_data_container()
{
	serverdata* data = (serverdata*)g_malloc(sizeof(serverdata));
	data->database = gems_teacher_server_new_profile_data_container();
	data->clients = NULL;
	data->ph = NULL;
	data->ph_cb = NULL;
	
	return data;
}

void clear_data_container(serverdata* data)
{
	if(data)
	{
		if(data->clients)
		{
			g_list_foreach(data->clients,(GFunc)gems_clear_gems_connection,NULL);
			g_list_free(data->clients);
		}
		
		if(data->ph) ph_c_delete_peerhood(data->ph);
		if(data->ph_cb) ph_c_delete_callback(data->ph_cb);
		if(data->database) gems_teacher_server_clear_profile_data_container(data->database);
		g_free(data);
		data = NULL;
	}
}

void sighandler(int sig)
{
	cem_add_to_log("Caught kill/term/int signal. Shutting down.",J_LOG_INFO);
	run = FALSE;
}

void ph_callback_notify(short aEvent, const char* aAddress,void* aData)
{
	gchar* logmsg = g_strdup_printf("gems_ph_callback_notify: event notify (%d) from %s.", aEvent,aAddress);
	cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
	g_free(logmsg);
}

gint compare_connections(gems_connection* a, gems_connection* b)
{
	if(ph_c_connection_get_device_checksum(a->connection) == ph_c_connection_get_device_checksum(b->connection)) return 0;
	else if(ph_c_connection_get_device_checksum(a->connection) < ph_c_connection_get_device_checksum(b->connection)) return -1;
	else return 1;
}

void ph_callback_newconnection(const unsigned short aPort, MAbstractConnection* aConnection, int aConnectionId, void* aData)
{
	serverdata* data = NULL;
	
	if(aData) data = (serverdata*)aData;
	gems_connection *c = gems_new_gems_connection(aConnection,aPort,aConnectionId,1,0);
	if(g_list_find_custom(data->clients,c,(GCompareFunc)compare_connections) == NULL)
	{
		data->clients = g_list_append(data->clients,c);
		cem_add_to_log("New connection arrived",J_LOG_INFO);
	}
	else
	{
		cem_add_to_log("Is already connected",J_LOG_INFO);
		gems_clear_gems_connection(c);
	}
}

gboolean process_data(gems_connection* c,serverdata* data)
{
	if(!c || !data) return FALSE;
	
	if(ntohs(gems_connection_get_16(c,0)) == JAMMO_PACKET_PRIVATE)
	{
		if(gems_security_extract_envelope(c) != SECURITY_OK)
		{
			// Omnom, devour the message since it was failurously delicious
			gems_connection_clear_buffer(c);
			return FALSE;
		}
	}
	
	gems_connection_localize_data_in_buffer(c);
	gchar* log = NULL;
	// Process
	switch(gems_connection_get_16(c,0))
	{
		case TEACHER_PROFILE_SERVICE_ID:
			if(gems_connection_get_16(c,sizeof(gint16)+sizeof(gint32)) == REQ_ENC_PROFILE)
			{				
				gint unamelen = gems_connection_get_32(c,sizeof(gint16)) - sizeof(gint16) - sizeof(gint32) - sizeof(gint16);
				gchar* username = gems_connection_get_data(c,sizeof(gint16)+sizeof(gint32)+sizeof(gint16),unamelen);
				
				log = g_strdup_printf("Got profile request, username: \"%s\"",username);
				cem_add_to_log(log,J_LOG_INFO);
				g_free(log);
				
				jammo_profile_container* profile = gems_teacher_server_get_profile_for_user(data->database,username);
				if(profile)
				{
					gems_message* msg = gems_create_message_teacher_service_profile_reply(profile->encrypted,profile->encrypted_length,username);
					if(msg)
					{
						if(!gems_communication_write_encrypted_data(JAMMO_PACKET_ADMIN,c,msg))
						{
							ph_c_connection_delete(c->connection);
							docleanup = TRUE;
						}
						else
						{
							log = g_strdup_printf("Sent profile for username \"%s\"",username);
							cem_add_to_log(log,J_LOG_INFO);
							g_free(log);
						}
					}
					gems_clear_message(msg);
				}
				else
				{
					log = g_strdup_printf("No profile found for username \"%s\"",username);
					cem_add_to_log(log,J_LOG_INFO);
					g_free(log);
				}
				g_free(username);
			}
			break;
		case ERROR_MSG:
			break;
		default:
			// TODO handle errors
			break;
	}
	gems_connection_clear_buffer(c);
	return TRUE;
}

void check_connection(gems_connection* c, serverdata* data)
{	
	gint bytes = 0;
	if(!c) return;
	if(!c->connection) return;
	
	if(ph_c_connection_is_connected(c->connection))
	{		
		if(ph_c_connection_has_data(c->connection))
		{
			cem_add_to_log("Processing connection: has data", J_LOG_INFO);
			// read service - 16 bits
			if(gems_connection_partial_data_amount(c) == 0)
			{
				gint16 cid = -1;
				bytes = 0;
				
				if((bytes = gems_communication_read_data(ph_c_connection_get_fd(c->connection), &cid, sizeof(gint16))) != sizeof(gint16))
				{
					cem_add_to_log("Read less than service id",J_LOG_ERROR);
					if(bytes <= 0)
					{
						ph_c_connection_disconnect(c->connection);
						docleanup = TRUE;
					}
				}
				else gems_connection_add_16(c,cid);
			}
			// read length
			else if(gems_connection_partial_data_amount(c) == sizeof(gint16))
			{
				gint32 length = 0;
				bytes = 0;
				
				if((bytes = gems_communication_read_data(ph_c_connection_get_fd(c->connection), &length, sizeof(gint32))) != sizeof(gint32))
				{
					cem_add_to_log("Read less than length id",J_LOG_ERROR);
					if(bytes <= 0)
					{
						ph_c_connection_disconnect(c->connection);
						docleanup = TRUE;
					}
				}
				else gems_connection_add_32(c,length);
			}
			// read rest
			else if(gems_connection_partial_data_amount(c) == sizeof(gint16) + sizeof(gint32))
			{
				gint remaining = ntohl(gems_connection_get_32(c,sizeof(gint16))) - gems_connection_partial_data_amount(c);
				gchar remainingdata[remaining];
				memset(&remainingdata,0,remaining);
				
				if((bytes = gems_communication_read_data(ph_c_connection_get_fd(c->connection), &remainingdata, remaining)) != remaining)
				{
					cem_add_to_log("All data not completely read!",J_LOG_ERROR);
					if(bytes <= 0)
					{
						ph_c_connection_disconnect(c->connection);
						docleanup = TRUE;
					}
					else gems_connection_add_data(c,remainingdata,bytes);
				}
				else
				{
					cem_add_to_log("All data read!",J_LOG_DEBUG);
					gems_connection_add_data(c,remainingdata,remaining);
				}
			}
			else
			{
				gchar rbyte = '\0';
				if(gems_communication_read_data(ph_c_connection_get_fd(c->connection), &rbyte, sizeof(gchar)) < sizeof(gchar))
				{
					cem_add_to_log("Cannot read a byte! Closing connection.",J_LOG_ERROR);
					ph_c_connection_disconnect(c->connection);
					docleanup = TRUE;
				}
				else gems_connection_add_data(c,&rbyte,sizeof(gchar));
			}
			
			if(gems_connection_partial_data_amount(c) == ntohl(gems_connection_get_32(c,sizeof(gint16))))
			{
				if(!process_data(c,data))
				{
					ph_c_connection_disconnect(c->connection);
					docleanup = TRUE;
				}
			}
		}
	}
}

void check_connections(serverdata* data)
{
	g_list_foreach(data->clients,(GFunc)check_connection,data);
}

void search_clients(serverdata* data)
{
	// Get all devices with control service (this is the service teacher connects to)
	TDeviceList* list = ph_c_get_devicelist_with_services(data->ph,CONTROL_SERVICE_NAME);
	if(list)
	{
		if(ph_c_devicelist_is_empty(list) == FALSE)
		{
			DeviceIterator* iterator = NULL;
			
			for(iterator = ph_c_new_device_iterator(list); !ph_c_device_iterator_is_last(iterator,list); ph_c_device_iterator_next_iterator(iterator))
			{
				MAbstractDevice* dev = ph_c_device_iterator_get_device(iterator);
				
				// Check if connection exists
				if(gems_communication_get_connection(data->clients,ph_c_device_get_checksum(dev)) == NULL)
				{
					// Through WLAN and has Peerhood
					if((g_strcmp0(ph_c_device_get_prototype(dev),"wlan-base") == 0) && (ph_c_device_has_peerhood(dev) == TRUE))
					{
						MAbstractConnection* conn = ph_c_connect_remoteservice(data->ph, dev, CONTROL_SERVICE_NAME);
					
						if(conn)
						{
							// Use gems_connection structs in list
							gems_connection* new_conn = gems_new_gems_connection(conn, 0, 0, 1, 0);
							data->clients = g_list_append(data->clients, new_conn);
							cem_add_to_log("New connection established!", J_LOG_INFO);
						}
					}
				}
			} // for
			ph_c_delete_device_iterator(iterator);
		}
		ph_c_delete_devicelist(list);
	}
}

teacher_server_profile_data* load_database(teacher_server_profile_data* data)
{
	// Read profile db
	switch(gems_teacher_server_read_profile_database(data))
	{
		case READ_OK:
			cem_add_to_log("Teacher database read success!",J_LOG_INFO);
			break;
		case READ_FAIL_FILE_NOT_EXIST:
			cem_add_to_log("Teacher database read failure! No database!",J_LOG_INFO);
			data = gems_teacher_server_new_profile_data_container();
			return data;
		default:
			cem_add_to_log("Teacher database read failure!",J_LOG_INFO);
			return NULL;
	}
	
	// Decrypt data with password
	if(gems_teacher_server_decrypt_profiles(data,"mypassword")) cem_add_to_log("Teacher database decryption success!",J_LOG_INFO);
	else cem_add_to_log("Teacher database decryption failure!",J_LOG_INFO);
	
	return data;
}

void save_database(teacher_server_profile_data* data)
{
	// Encrypt profiles (all!)
	if(gems_teacher_server_encrypt_profiles(data, "mypassword")) cem_add_to_log("Teacher database encryption success!", J_LOG_INFO);
	else cem_add_to_log("Teacher database encryption failure!", J_LOG_INFO);
	
	if(gems_teacher_server_write_profile_database(data) == WRITE_OK) cem_add_to_log("Teacher database saving success!", J_LOG_INFO);
	else cem_add_to_log("Teacher database saving failure!", J_LOG_INFO);
}

void create_profile_to_database(teacher_server_profile_data* data)
{
	gboolean do_more = TRUE;
	GRand* grand = g_rand_new_with_seed(time(NULL));
	// Username
	gchar* username = (gchar*)g_malloc0(sizeof(gchar*)*50);
	while(do_more)
	{
		printf("Username: ");
		if(fgets(username,50,stdin))
		{
			erase_newline(username);
			// Check for illegal characters
			if(!gems_teacher_server_check_illegal_characters(username))
			{
				// Check if username taken
				if(!gems_teacher_server_get_profile_for_user(data,username)) do_more = FALSE;
				else memset(username,0,50);
			}
			else memset(username,0,50);
		}
	}
	do_more = TRUE;

	gchar* password = (gchar*)g_malloc0(sizeof(gchar*)*PASSLEN);
	while(do_more)
	{
		printf("Password: ");
		if(fgets(password,PASSLEN,stdin))
		{
			erase_newline(password);
			// Check
			if(!gems_teacher_server_check_illegal_characters(password))
			{
				// Check if numbers
				if(is_all_integers(password)) do_more = FALSE;
				else memset(password,0,50);
			}
			else memset(password,0,PASSLEN);
		}
	}
	do_more = TRUE;
	
	gchar* firstname = (gchar*)g_malloc0(sizeof(gchar*)*50);
	while(do_more)
	{
		printf("Firstname: ");
		if(fgets(firstname,50,stdin))
		{
			erase_newline(firstname);
			// Check for illegal characters
			if(!gems_teacher_server_check_illegal_characters(firstname)) do_more = FALSE;
			else memset(firstname,0,50);
		}
	}
	do_more = TRUE;
	
	gchar* lastname = (gchar*)g_malloc0(sizeof(gchar*)*50);
	while(do_more)
	{
		printf("Lastname: ");
		if(fgets(lastname,50,stdin))
		{
			erase_newline(lastname);
			// Check for illegal characters
			if(!gems_teacher_server_check_illegal_characters(lastname))	do_more = FALSE;
			else memset(lastname,0,50);
		}
	}
	do_more = TRUE;
	
	guint32 id = g_rand_int(grand);
	while(gems_teacher_server_get_profile_for_user_id(data,id)) id = g_rand_int(grand);
	
	gchar* agestring = (gchar*)g_malloc0(sizeof(gchar*)*3);
	guint16 age = 0;
	while(do_more)
	{
		printf("Age (2 char max!): ");
		if(fgets(agestring,3,stdin))
		{
			erase_newline(agestring);
			// Check for integers
			if(is_all_integers(agestring))
			{
				age = (guint16)atoi(agestring);
				do_more = FALSE;
			}
			else memset(lastname,0,3);
		}
	}
	do_more = TRUE;
	// Search for id
	
	jammo_profile_container* profile = gems_teacher_server_create_profile(id,username,password,"none",firstname,lastname,age);
	if(profile)
	{
		data->profiles = g_list_append(data->profiles,profile);
	
		cem_add_to_log("Profile added",J_LOG_INFO);
	
		gems_teacher_server_print_profile_list(data->profiles);
	}
	else cem_add_to_log("Profile not added, parameters invalid",J_LOG_INFO);
	g_free(username);
	g_free(password);
	g_free(firstname);
	g_free(lastname);
	g_free(grand);
	cem_add_to_log("done",J_LOG_INFO);
}

gboolean is_all_integers(gchar* c)
{
	for(gint i = 0; i < strlen(c); i++) if(!g_ascii_isdigit(c[i])) return FALSE;
	return TRUE;
}

void erase_newline(gchar* c)
{
	gchar* pos = g_strrstr(c,"\n");
	if(pos) pos[0] = '\0';
}

int main(int argc,char* argv[])
{
	gint port = 0;
	serverdata* data = new_data_container();
	
	cem_set_log_level(J_LOG_DEBUG+J_LOG_INFO+J_LOG_ERROR+J_LOG_NETWORK_DEBUG);
	signal(SIGTERM,sighandler);
	signal(SIGINT,sighandler);
	
	void (*notify)(short, const char*,void*) = NULL;
	notify = &ph_callback_notify;

	void (*newconnection)(const unsigned short, MAbstractConnection*, int,void*) = NULL;
	newconnection = &ph_callback_newconnection;
	
	// Callback and instance of PeerHood
	data->ph_cb = ph_c_create_callback(notify, newconnection,NULL,(void*)data);
	data->ph = ph_c_get_instance(data->ph_cb);
	
	// Load database and print it
	data->database = load_database(data->database);
	if(!data->database) exit(1);
	gems_teacher_server_print_profile_list(data->database->profiles);
	cem_add_to_log("Teacher profile service, press Ctrl+C to terminate, press Enter to start entering new profile to database", J_LOG_INFO);
	
	if(ph_c_init(data->ph,argc,argv) == TRUE)
	{
		cem_add_to_log("PeerHood initialized",J_LOG_INFO);
		if((port = ph_c_register_service(data->ph,TEACHER_PROFILE_SERVICE_NAME,"application:jammoteacher,method:ALL_YOUR_BASE_ARE_BELONG_TO_US")) != 0)
		{
			cem_add_to_log("Registered Teacher profile service",J_LOG_INFO);
			fd_set s;
			struct timeval tv;
			tv.tv_sec = 0;
			tv.tv_usec = 0;
			while(run)
			{
				check_connections(data); // Check for incoming data
				search_clients(data); // Search for JamMos
				
				if(docleanup)
				{
					cem_add_to_log("Connection cleanup",J_LOG_INFO);
					data->clients = gems_cleanup_connections_from_list(data->clients);
					docleanup = FALSE;
				}
				
				// Enter pressed, entering a new profile data
				
				FD_SET(0,&s);
				
				switch(select(0+1,&s,NULL,NULL,&tv))
				{
					case -1:
						exit(1);
						break;
					case 0:
						break;
					default:
						if(FD_ISSET(0,&s))
						{
							if(getc(stdin) == '\n') create_profile_to_database(data->database);
						}
						break;
				}
				FD_CLR(0,&s);
				
				g_usleep(500000);
			}
		}
		else cem_add_to_log("Cannot register Teacher service",J_LOG_INFO);
	}
	else
	{
		cem_add_to_log("Cannot initialize PeerHood",J_LOG_ERROR);
	}
	
	if(port != 0) ph_c_unregister_service(data->ph,TEACHER_PROFILE_SERVICE_NAME);
	if(data)
	{
		save_database(data->database);
		clear_data_container(data);
	}
	
	return 0;
}
