/*
 * gems_structures.h
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 * 	    Tommi Kallonen  <tommi.kallonen@lut.fi>		
 */

#ifndef __GEMS_STRUCTURES_
#define __GEMS_STRUCTURES_

#include <peerhood/PeerHood.h>
#include <peerhood/IteratorWrapper.h>
#include <glib.h>
#include <glib-object.h>
#include <netinet/in.h>
#include <openssl/evp.h>
#include "gems_definitions.h"
#include "ProfileManager.h"

// States of connecting devices
typedef enum {
	JAMMO_CONNECTING, 				// Added, waiting for CONNECT_REQ
	JAMMO_WAITING_VERSION, 		// VERSION_REQ sent, waiting for VERSION_REPLY
	JAMMO_WAITING_REQUEST,		// CONNECT_REPLY sent, waiting for VERSION_REQ
	JAMMO_WAITING_REPLY,			// VERSION_REPLY sent, waiting for CONNECT_REPLY
	JAMMO_VERSION_MISMATCH,		// versions mismatch - ERROR is sent
	JAMMO_CONNECTED_NO_AUTH,	// CONNECT_REPLY received, device is connected
	JAMMO_CONNECTED_AUTH			// Authenticated - ready to roll
	// TODO authentication states
} jammo_state;

/* Profile of some other user */
typedef struct {
	guint32 id;
	gchar* username;
	guint16 avatarid;
	guint16 age;
} gems_peer_profile;



// Group info
typedef struct {
	// Group id
	guint32 id;
	// Group owner;
	guint32 owner;
	// Group type, 
	gint16 type;
	// Group theme id
	gint16 theme;
	// Size of the group (2 = pair game -> 1 available, 4 = group -> 3 available)
	gint16 size;
	// Other members
	guint32 peers[GROUP_MAX_SIZE-1]; // TODO change to use gems_connection* - less list operations
	// free spaces
	gint16 spaces;
	// Locked?
	gboolean is_locked;
	// own status in this group (IN_GROUP, JOINING, LEAVING)
	gint16 own_state;
	// For periodic group advertizing and checking when the last advert was received
	GTimer* group_advert_timer; 
} gems_group_info;

// Connection item holder with additional info
typedef struct {
	MAbstractConnection* connection;
	MAbstractConnection* game_connection; // For collaboration / games 
	unsigned short port; // TODO necessary?
	int connectionid; // TODO necessary?
	
	// TODO Add JamMo data!
	jammo_state connection_state; // State of the connecting device
	guint32 jammoid; // Id of the user - should be retrieved from certificate?
	
	/*** PROFILE ***/
	gems_peer_profile* profile; // Profile of peer
	gint16 profile_requests;
	GTimer* profile_request_timer;
	
	/*** GROUP REQUEST TIMER ***/
	GTimer* group_request_timer;
	
	/*** LAST ACTION TIMER ***/
	GTimer* last_action; // Use for last actions & time in rejected list
	
	/* TODO Add last action made (what was sent) and timeout for possible re-send
	   - use also for removing items from rejected list after some time to keep
	   list size fairly small. Service & device as parameters. */
	
	/* network buffering */
	gchar nwbuffer[JAMMO_NETWORK_BUFFER_MAX_SIZE]; // buffer for partial reads
	guint add_position; // adding position, should be 0 when there is no data!
} gems_connection;


typedef struct {
	gems_connection* connection;
	// TODO certificates + other stuff here
} gems_teacher_connection;

// JamMo service 
typedef struct {
	GList* connections; // gems_connection x N
	gboolean enabled;
	gint port;
} gems_service_jammo;

// Profile service
typedef struct {
	gboolean enabled;
	gint port;
} gems_service_profile;

// Collaboration service
typedef struct {
	gboolean enabled;
	gboolean active;
	gint port;
	// collaboration_game can be any multiplayer game
	GObject * collaboration_game;
} gems_service_collaboration;

// Group management service
typedef struct {
	gboolean enabled;
	gems_group_info* group_info;
	GList* other_groups; // contains gems_group_info
	gint port;
} gems_service_group;

// Control service
typedef struct {
	gboolean enabled;
	gboolean active; //is there a connection to teacher
	gint port;
} gems_service_control;

// Communication items - 
typedef struct {
	GList* connections; // contains gems_connection, connected items
	GList* connecting; // contains gems_connection, connecting items
	GList* rejected; // contains gems_connection, rejected/version mismatch
} gems_communication_data;

typedef struct{
//	gems_service_songbank* mySBobj;	
	MPeerHood* peerHood;	
	unsigned short sbPort; //songbank service port on this device
	GList* connections;    // connections to SongBankComm
//	MAbstractConnection *connection;
//	pthread_t connThread;
//	int threadId;
}SongBankComm;

typedef struct{
	char* baseDirectory;
	SongBankComm* sbComm;
}gems_service_songbank;

/* Connection error items, every errorous connection to a device
   increases the errorcount, more errors -> increase connection interval
   to the device */
typedef struct {
	guint checksum;
	guint connection_errors;
	gboolean version_mismatch;
	gboolean already_connected;
	// TODO add "invalid messages" counter
	GTimer* connection_timer;
} gems_connection_error;


typedef struct {
	EVP_CIPHER_CTX* encryption;
	EVP_CIPHER_CTX* decryption;
	EVP_MD_CTX* digest;
} gems_security_context;

// High level struct for GEMS
typedef struct {
	guint timeout_functions[9];
	// Profile Manager
	ProfileManager* pm;
	
	/* PeerHood */
	MPeerHood* ph;
	C_Callback* ph_cb;
	
	/* Initialized connections - gems_connection items*/
	GList* initialized_connections;
	
	/* Communication */
	gems_communication_data* communication;
	
	gems_security_context* security_context;
	
	/* Services*/
	gems_service_jammo* service_jammo;
	//gems_service_aa* service_aa;
	gems_service_profile* service_profile;

	gems_service_group* service_group;

	gems_service_collaboration* service_collaboration;

	gems_service_control* service_control;
	
	GList* connection_errorlist; // Contains gems_connection_error items

	gems_teacher_connection* teacher_connection;
	// Set values to .jammo-profiles/<id>.csv

	gems_service_songbank* service_songbank;
	
} gems_components;

/* Generic GEMS message */
typedef struct {
	gchar* message;
	gint32 length;
} gems_message;



void gems_clear_gems_connection(gems_connection* element);

void gems_connection_clear_buffer(gems_connection* element);

void gems_connection_set_action(gems_connection* element);
double gems_connection_get_last_action(gems_connection* element);

void gems_connection_add_16(gems_connection* element, guint16 value);
void gems_connection_add_32(gems_connection* element, guint32 value);
void gems_connection_add_64(gems_connection* element, guint64 value);
void gems_connection_add_float(gems_connection* element, gfloat value);
void gems_connection_add_char(gems_connection* element, gchar* value, gint length, gboolean pad);
void gems_connection_add_padding(gems_connection* element);
void gems_connection_add_data(gems_connection* element, gchar* data, gint length);
guint16 gems_connection_get_16(gems_connection* element, gint from);
guint32 gems_connection_get_32(gems_connection* element, gint from);
guint64 gems_connection_get_64(gems_connection* element, gint from);
gfloat gems_connection_get_float(gems_connection* element, gint from);
gfloat gems_connection_get_float_for_localization(gems_connection* element, gint from);
gchar* gems_connection_get_char(gems_connection* element, gint from);
gchar* gems_connection_get_data(gems_connection* element, gint from, gint length);

void gems_connection_set_content(gems_connection* element, gchar* newcontent, gint length);

gint gems_connection_partial_data_amount(gems_connection* element);

void gems_connection_localize_data_in_buffer(gems_connection* element);

gems_connection* gems_new_gems_connection(MAbstractConnection* newconn,
	unsigned short newport,
	int newconnectionid,
	jammo_state newstate,
	gint32 newjammoid);

gems_peer_profile* gems_new_peer_profile(gchar* username, guint32 id, guint16 age, guint16 avatarid);
void gems_peer_profile_set_values(gems_peer_profile* profile, gchar* username, guint32 id, guint16 age, guint16 avatarid);

void gems_clear_peer_profile(gems_peer_profile* profile);

GList* gems_cleanup_connections_from_list(GList* list);

GList* gems_connection_error_add(GList* list, guint device_checksum, guint errortype);
gems_connection_error* gems_connection_error_get(GList* list, guint device_checksum);
guint gems_connection_error_get_connection_error_count(GList* list, guint device_cheksum);
gboolean gems_connection_error_attempt_connection(GList* list, guint device_checksum);
GList* gems_connection_error_remove(GList* list, guint device_checksum);
void gems_clear_connection_error(gems_connection_error* element);

GList* gems_process_rejected(GList* list);

guint32 htonf(gfloat value);
gfloat ntohf(guint32 value);

#endif // __GEMS_STRUCTURES
