/*
 * gems_teacher_server_utils.h
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */
 
#ifndef __GEMS_TEACHER_SERVER_UTILS_H_
#define __GEMS_TEACHER_SERVER_UTILS_H_

#define WRITE_FAIL_CONTENT_NOT_SET 1
#define WRITE_FAIL_CANNOT_REMOVE_FILE 2 
#define WRITE_FAIL_CANNOT_SET_FILE_CONTENT 3
#define WRITE_OK 0

#include <glib.h>
#include "gems_definitions.h"

typedef struct {
	guint32 userid;
	gchar* username;
	gchar* password;
	guchar* salt;
	gchar* firstname;
	gchar* lastname;
	gchar* authlevel;
	guint16 age;
	guint32 points;
	gchar* encrypted; // Contains the whole encrypted profile (salt+pwhash+phash+profile)
	guint encrypted_length; // Length of encrypted -field
} jammo_profile_container;

/* Allocate new jammo_profile_container
 *
 * Returns:
 *  pointer to jammo_profile_container struct
 */
jammo_profile_container* gems_teacher_server_new_profile_container();

/* Free the given jammo_profile_container struct
 *
 * Params:
 *  container - pointer to container struct
 */
void gems_teacher_server_clear_profile_container(jammo_profile_container* container);

/* Create a new profile with given information, profile will be also encrypted - calls
 * gems_teacher_profile_encrypt_profile().
 *
 * Params (duplicates are created from gchars):
 *  _userid - user id to use, should be something else than 0
 *  _username - username
 *  _password - password that user gives
 *  _authlevel - level of authentication (atm. "none" should be ok..)
 *  _firstname - firstname of the user
 *  _lastname - lastname of the user
 *  _age - age of the user
 * Returns:
 *  pointer to jammo_profile_container struct when all could be done
 *  NULL if some parameter was not set or the profile couldn't be encrypted
 */
jammo_profile_container* gems_teacher_server_create_profile(guint32 _userid, gchar* _username, gchar* _password,
	gchar* _authlevel, gchar* _firstname, gchar* _lastname, guint16 _age);
	
/* Encrypt the profile content, uses data in container and sets the encrypted content + length
 * Normally used through other functions (create profile & change password) - no checks for container data !
 *
 * Params:
 *  container - pointer to container struct, should have all profile details set!
 * Returns:
 *  TRUE when profile could be encrypted
 *  FALSE if something was missing
 */
gboolean gems_teacher_server_encrypt_profile(jammo_profile_container* container);

/* Change password for given profile, profile data will be re-encrypted with new password
 *
 * Params:
 *  container - profile to use, pointer to jammo_profile_container struct
 *  password - new password
 * Returns:
 *  TRUE when password could be changed
 *  FALSE when there are problems setting password (cannot initialize security or encryption fails)
 */ 
gboolean gems_teacher_server_change_password(jammo_profile_container* container, gchar* password);

/* Write encrypted profile to "username.jpf" into $HOME/.jammo/profile/
 *
 * Params:
 *   container - pointer to jammo_profile_container struct containing encrypted data
 * Returns:
 *  WRITE_FAIL_CONTENT_NOT_SET for invalid parameters
 *  WRITE_FAIL_CANNOT_REMOVE_FILE if previous profile cannot be removed
 *  WRITE_FAIL_CANNOT_SET_FILE_CONTENT if cannot set the encrypted data to file
 *  WRITE_OK when profile was written
 */
gint gems_teacher_server_write_profile(jammo_profile_container* container);

#endif
