/*
 * gems_teacher_server_utils.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */

#include <glib/gstdio.h>
#include "gems_teacher_server_utils.h"
#include "ProfileManager.h"
#include "gems_security.h"
#include "../configure.h"

jammo_profile_container* gems_teacher_server_new_profile_container()
{
	jammo_profile_container* container = (jammo_profile_container*)g_malloc(sizeof(jammo_profile_container));
	// Allocation failed?
	if(container == NULL) return NULL;
	
	container->userid = 0;
	container->username = NULL;
	container->password = NULL;
	container->salt = NULL;
	container->firstname = NULL;
	container->lastname = NULL;
	container->authlevel = NULL;
	container->age = 0;
	container->points = 0;
	container->encrypted = NULL;
	container->encrypted_length = 0;
	return container;	
}

void gems_teacher_server_clear_profile_container(jammo_profile_container* container)
{
	if(container)
	{
		if(container->username) g_free(container->username);
		if(container->password) g_free(container->password);
		if(container->salt) g_free(container->salt);
		if(container->firstname) g_free(container->firstname);
		if(container->lastname) g_free(container->lastname);
		if(container->authlevel) g_free(container->authlevel);
		if(container->encrypted) g_free(container->encrypted);
		
		g_free(container);
	}
}

jammo_profile_container* gems_teacher_server_create_profile(guint32 _userid, gchar* _username, gchar* _password,
	gchar* _authlevel, gchar* _firstname, gchar* _lastname, guint16 _age)
{
	// check input
	if((_userid == 0) || (_username == NULL) || (_password == NULL) || (_authlevel == NULL) || (_firstname == NULL) 
		|| (_lastname == NULL) || (_age == 0)) return NULL;

	jammo_profile_container* container = gems_teacher_server_new_profile_container();
	
	// Set details
	container->userid = _userid;
	container->age = _age;
	container->username = g_strdup(_username);
	container->firstname = g_strdup(_firstname);
	container->lastname = g_strdup(_lastname);
	container->authlevel = g_strdup(_authlevel);
	container->password = g_strdup(_password);

	if(gems_teacher_server_encrypt_profile(container)) return container;
	else
	{
		gems_teacher_server_clear_profile_container(container);
		return NULL;
	}
}

gboolean gems_teacher_server_encrypt_profile(jammo_profile_container* container)
{
	// vars
	guint encproflen = 0, hashlen = 0, profilelen = 0, passlen = 0;
	gchar* userid = NULL;
	gchar* age = NULL;
	gchar* points = NULL;
	guchar* profilehash = NULL;
	guchar* encprofile = NULL;
	guchar* decprofile = NULL;
	guchar* decprofhash = NULL;
	guchar* profile = NULL;
	guchar* passwd = NULL;
	guchar* passhash = NULL;
	gboolean rval = TRUE;	

	if(container)
	{
		// Set profile data
		userid = g_strdup_printf("%u",container->userid);
		age = g_strdup_printf("%u",container->age);
		points = g_strdup_printf("%u",container->points);
		profile = (guchar*)g_strjoin(",", userid, container->username, container->authlevel, container->firstname, container->lastname, points, age, NULL);
		profilelen = strlen((gchar*)profile)+1;
	
		// Create salt and password
		if(container->salt == NULL) container->salt = gems_security_create_password_salt();
		passwd = gems_security_create_password(container->password,container->salt);
		passhash = NULL;
		passlen = PASSLEN;
	
		// Initialize security
		if(gems_security_init_security(passwd,container->salt))
		{
			passhash = gems_security_calculate_hash(passwd,&hashlen);
			if(hashlen != SHA_HASH_LENGTH) rval = FALSE; // Invalid size hash, profile might not be valid
			
			profilehash = gems_security_calculate_hash(profile,&hashlen);
			if(hashlen != SHA_HASH_LENGTH) rval = FALSE; // Invalid size hash, profile might not be valid
			
			encprofile = gems_security_encrypt_data(profile,&profilelen);
			
			encproflen = profilelen;
			decprofile = gems_security_decrypt_data(encprofile,&encproflen);
			
			decprofhash = gems_security_calculate_hash(decprofile,&hashlen);
			if(hashlen != SHA_HASH_LENGTH) rval = FALSE; // Invalid size hash, profile might not be valid
		
			if(gems_security_verify_hash(profilehash,decprofhash))
			{
				container->encrypted_length = AES_SALT_LENGTH + 32 + 32 + profilelen;
				
				// Allocate content, free old if it was allocated
				if(container->encrypted != NULL) g_free(container->encrypted);
				container->encrypted = (gchar*)g_malloc(sizeof(gchar*) * container->encrypted_length);
				memset(container->encrypted,0,container->encrypted_length);
	
				// Set salt
				*(guint32*)&container->encrypted[0] = *(guint32*)&container->salt[0];
				*(guint32*)&container->encrypted[sizeof(guint32)] = *(guint32*)&container->salt[sizeof(guint32)];
	
				// Set password hash
				g_memmove(&container->encrypted[AES_SALT_LENGTH],passhash,32);
				// Set profile hash
				g_memmove(&container->encrypted[AES_SALT_LENGTH + 32],profilehash,32);
				// Set profile
				g_memmove(&container->encrypted[AES_SALT_LENGTH + 64],encprofile,profilelen);
				
			}
			else rval = FALSE;
		}
		else rval = FALSE;

		// Free all
		if(userid != NULL) g_free(userid);
		if(age != NULL) g_free(age);
		if(points != NULL) g_free(points);
		if(profile!= NULL) g_free(profile);
		if(profilehash != NULL) g_free(profilehash);
		if(encprofile != NULL) g_free(encprofile);
		if(decprofile != NULL) g_free(decprofile);
		if(decprofhash != NULL) g_free(decprofhash);
		if(passwd != NULL) g_free(passwd);
		if(passhash != NULL) g_free(passhash);
	
		gems_security_clear_security();
	}
	else rval = FALSE;
		
	return rval;
}

gboolean gems_teacher_server_change_password(jammo_profile_container* container, gchar* password)
{	
	if(container)
	{
		if(container->password != NULL) g_free(container->password);
		container->password = g_strdup(password);
		return gems_teacher_server_encrypt_profile(container);
	}
	else return FALSE;
}

// Write encrypted profile
gint gems_teacher_server_write_profile(jammo_profile_container* container)
{
	if(!container) return WRITE_FAIL_CONTENT_NOT_SET;
	if(!container->username) return WRITE_FAIL_CONTENT_NOT_SET;
	if(!container->encrypted) return WRITE_FAIL_CONTENT_NOT_SET;
	
	gchar* profilefile = g_strjoin(NULL,container->username,PROFILE_FILE_EXT,NULL);
	gchar* filename = g_build_filename(configure_get_jammo_directory(),JAMMOPROFILE,profilefile,NULL);
	GError* error = NULL;
	
	// Check for previous profile file
	if(g_file_test(filename,(G_FILE_TEST_EXISTS)))
	{
		if(g_remove(filename) != 0)
		{
			// Error, change username?
			g_free(filename);
			g_free(profilefile);
			return WRITE_FAIL_CANNOT_REMOVE_FILE;
		}
	}

	// Set content
  if(!g_file_set_contents(filename,container->encrypted,container->encrypted_length,&error))
  {
  	if(error)
		{
			// print error
			g_error_free(error);
			g_free(filename);
			g_free(profilefile);
			return WRITE_FAIL_CANNOT_SET_FILE_CONTENT;
		}
  }
  
	g_free(filename);
	g_free(profilefile);
	
	return WRITE_OK;
}
