/*
 * jammo-mentor-activity.h
 *
 * This file is part of JamMo.
 *
 * (c) 2010-2011 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-mentor-activity.h"

G_DEFINE_TYPE(JammoMentorActivity, jammo_mentor_activity, TANGLE_TYPE_ACTIVITY);

enum {
	PROP_0,
	PROP_MENTOR,
	PROP_SPEECH,
	PROP_SPEAK_ONCE,
	PROP_IDLE_SPEECH,
	PROP_HIGHLIGHT,
	PROP_OFFSET_X,
	PROP_OFFSET_Y
};

enum {
	SPOKEN,
	INTERRUPTED,
	LAST_SIGNAL
};

struct _JammoMentorActivityPrivate {
	JammoMentor* mentor;
	gchar* speech;
	gchar* idle_speech;
	gfloat offset_x;
	gfloat offset_y;
	guint speak_once : 1;
	guint highlight : 1;
};

static void on_mentor_spoken(JammoMentor* mentor, const gchar* speech, gboolean interrupted, gpointer user_data);

static guint signals[LAST_SIGNAL] = { 0 };

TangleActivity* jammo_mentor_activity_new(void) {

	return TANGLE_ACTIVITY(g_object_new(JAMMO_TYPE_MENTOR_ACTIVITY, NULL));
}

JammoMentor* jammo_mentor_activity_get_mentor(JammoMentorActivity* mentor_activity) {
	g_return_val_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity), NULL);

	return mentor_activity->priv->mentor;
}

void jammo_mentor_activity_set_mentor(JammoMentorActivity* mentor_activity, JammoMentor* mentor) {
	g_return_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity));
	g_return_if_fail(JAMMO_IS_MENTOR(mentor));

	if (mentor_activity->priv->mentor != mentor) {
		if (mentor_activity->priv->mentor) {
			g_object_unref(mentor_activity->priv->mentor);
		}
		mentor_activity->priv->mentor = mentor;
		g_object_ref(mentor_activity->priv->mentor);
		g_object_notify(G_OBJECT(mentor_activity), "mentor");
	}
}

const gchar* jammo_mentor_activity_get_speech(JammoMentorActivity* mentor_activity) {
	g_return_val_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity), NULL);

	return mentor_activity->priv->speech;
}

void jammo_mentor_activity_set_speech(JammoMentorActivity* mentor_activity, const gchar* speech) {
	g_return_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity));

	if (g_strcmp0(mentor_activity->priv->speech, speech)) {
		g_free(mentor_activity->priv->speech);
		mentor_activity->priv->speech = g_strdup(speech);
		g_object_notify(G_OBJECT(mentor_activity), "speech");	
	}
}

gboolean jammo_mentor_activity_get_speak_once(JammoMentorActivity* mentor_activity) {
	g_return_val_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity), FALSE);

	return mentor_activity->priv->speak_once;
}

void jammo_mentor_activity_set_speak_once(JammoMentorActivity* mentor_activity, gboolean once) {
	g_return_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity));

	if (mentor_activity->priv->speak_once != once) {
		mentor_activity->priv->speak_once = once;
		g_object_notify(G_OBJECT(mentor_activity), "speak-once");	
	}
}

gboolean jammo_mentor_activity_get_highlight(JammoMentorActivity* mentor_activity) {
	g_return_val_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity), FALSE);

	return mentor_activity->priv->highlight;
}

void jammo_mentor_activity_set_highlight(JammoMentorActivity* mentor_activity, gboolean highlight) {
	g_return_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity));

	if (mentor_activity->priv->highlight != highlight) {
		mentor_activity->priv->highlight = highlight;
		g_object_notify(G_OBJECT(mentor_activity), "highlight");
	}
}

gfloat jammo_mentor_activity_get_offset_x(JammoMentorActivity* mentor_activity) {
	g_return_val_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity), 0.0);

	return mentor_activity->priv->offset_x;
}

void jammo_mentor_activity_set_offset_x(JammoMentorActivity* mentor_activity, gfloat offset_x) {
	g_return_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity));

	if (mentor_activity->priv->offset_x != offset_x) {
		mentor_activity->priv->offset_x = offset_x;
		g_object_notify(G_OBJECT(mentor_activity), "offset-x");
	}
}

gfloat jammo_mentor_activity_get_offset_y(JammoMentorActivity* mentor_activity) {
	g_return_val_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity), 0.0);

	return mentor_activity->priv->offset_y;
}

void jammo_mentor_activity_set_offset_y(JammoMentorActivity* mentor_activity, gfloat offset_y) {
	g_return_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity));

	if (mentor_activity->priv->offset_y != offset_y) {
		mentor_activity->priv->offset_y = offset_y;
		g_object_notify(G_OBJECT(mentor_activity), "offset-y");
	}
}

const gchar* jammo_mentor_activity_get_idle_speech(JammoMentorActivity* mentor_activity) {
	g_return_val_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity), NULL);

	return mentor_activity->priv->idle_speech;
}

void jammo_mentor_activity_set_idle_speech(JammoMentorActivity* mentor_activity, const gchar* speech) {
	g_return_if_fail(JAMMO_IS_MENTOR_ACTIVITY(mentor_activity));

	if (g_strcmp0(mentor_activity->priv->idle_speech, speech)) {
		g_free(mentor_activity->priv->idle_speech);
		mentor_activity->priv->idle_speech = g_strdup(speech);
		g_object_notify(G_OBJECT(mentor_activity), "idle-speech");	
	}
}

static void jammo_mentor_activity_activate(TangleActivity* activity) {
	JammoMentorActivity* mentor_activity;
	JammoMentor* mentor;
	
	mentor_activity = JAMMO_MENTOR_ACTIVITY(activity);
	if (!(mentor = mentor_activity->priv->mentor)) {
		mentor = jammo_mentor_get_default();
	}
	
	g_return_if_fail(mentor != NULL);
	g_return_if_fail(mentor_activity->priv->speech != NULL || mentor_activity->priv->idle_speech != NULL);
	
	jammo_mentor_shut_up(mentor);
	if (mentor_activity->priv->speech) {
		if (mentor_activity->priv->highlight) {
			if (mentor_activity->priv->speak_once) {
				jammo_mentor_speak_and_highlight_once_with_callback(mentor, mentor_activity->priv->speech, mentor_activity->priv->offset_x, mentor_activity->priv->offset_y, on_mentor_spoken, mentor_activity);
			} else {
				jammo_mentor_speak_and_highlight_with_callback(mentor, mentor_activity->priv->speech, mentor_activity->priv->offset_x, mentor_activity->priv->offset_y, on_mentor_spoken, mentor_activity);
			}
		 } else {
			if (mentor_activity->priv->speak_once) {
				jammo_mentor_speak_once_with_callback(mentor, mentor_activity->priv->speech, on_mentor_spoken, mentor_activity);
			} else {
				jammo_mentor_speak_with_callback(mentor, mentor_activity->priv->speech, on_mentor_spoken, mentor_activity);
			}
		}
	}
	if (mentor_activity->priv->idle_speech) {
		jammo_mentor_set_idle_speech(mentor, mentor_activity->priv->idle_speech);
	}
}

static void jammo_mentor_activity_spoken(JammoMentorActivity* mentor_activity, const gchar* speech) {
}

static void jammo_mentor_activity_interrupted(JammoMentorActivity* mentor_activity, const gchar* speech) {
}

static void jammo_mentor_activity_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoMentorActivity* mentor_activity;
	
	mentor_activity = JAMMO_MENTOR_ACTIVITY(object);

	switch (prop_id) {
		case PROP_MENTOR:
			jammo_mentor_activity_set_mentor(mentor_activity, JAMMO_MENTOR(g_value_get_object(value)));
			break;
		case PROP_SPEECH:
			jammo_mentor_activity_set_speech(mentor_activity, g_value_get_string(value));
			break;
		case PROP_SPEAK_ONCE:
			jammo_mentor_activity_set_speak_once(mentor_activity, g_value_get_boolean(value));
			break;
		case PROP_IDLE_SPEECH:
			jammo_mentor_activity_set_idle_speech(mentor_activity, g_value_get_string(value));
			break;
		case PROP_HIGHLIGHT:
			jammo_mentor_activity_set_highlight(mentor_activity, g_value_get_boolean(value));
			break;
		case PROP_OFFSET_X:
			jammo_mentor_activity_set_offset_x(mentor_activity, g_value_get_float(value));
			break;
		case PROP_OFFSET_Y:
			jammo_mentor_activity_set_offset_y(mentor_activity, g_value_get_float(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_mentor_activity_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoMentorActivity* mentor_activity;

	mentor_activity = JAMMO_MENTOR_ACTIVITY(object);

        switch (prop_id) {
		case PROP_MENTOR:
			g_value_set_object(value, mentor_activity->priv->mentor);
			break;
		case PROP_SPEECH:
			g_value_set_string(value, mentor_activity->priv->speech);
			break;
		case PROP_SPEAK_ONCE:
			g_value_set_boolean(value, mentor_activity->priv->speak_once);
			break;
		case PROP_IDLE_SPEECH:
			g_value_set_string(value, mentor_activity->priv->idle_speech);
			break;
		case PROP_HIGHLIGHT:
			g_value_set_boolean(value, mentor_activity->priv->highlight);
			break;
		case PROP_OFFSET_X:
			g_value_set_float(value, mentor_activity->priv->offset_x);
			break;
		case PROP_OFFSET_Y:
			g_value_set_float(value, mentor_activity->priv->offset_y);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_mentor_activity_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_mentor_activity_parent_class)->finalize(object);
}

static void jammo_mentor_activity_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_mentor_activity_parent_class)->dispose(object);
}

static void jammo_mentor_activity_class_init(JammoMentorActivityClass* mentor_activity_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(mentor_activity_class);
	TangleActivityClass* activity_class = TANGLE_ACTIVITY_CLASS(mentor_activity_class);

	gobject_class->finalize = jammo_mentor_activity_finalize;
	gobject_class->dispose = jammo_mentor_activity_dispose;
	gobject_class->set_property = jammo_mentor_activity_set_property;
	gobject_class->get_property = jammo_mentor_activity_get_property;

	activity_class->activate = jammo_mentor_activity_activate;
	
	mentor_activity_class->spoken = jammo_mentor_activity_spoken;
	mentor_activity_class->interrupted = jammo_mentor_activity_interrupted;

	/**
	 * JammoMentorActivity:mentor:
	 *
	 * If :mentor is NULL, the default mentor (see jammo_mentor_get_default()) is used.
	 */
	g_object_class_install_property(gobject_class, PROP_MENTOR,
	                                g_param_spec_object("mentor",
	                                                    "Mentor",
	                                                    "The mentor",
	                                                    JAMMO_TYPE_MENTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * JammoMentorActivity:speech:
	 *
	 * The speech that the mentor should speak when the mentor_activity is executed.
	 * Can be NULL, if the purpose of the mentor_activity is only to set idle speech.
	 */
	g_object_class_install_property(gobject_class, PROP_SPEECH,
	                                g_param_spec_string("speech",
	                                                    "Speech",
	                                                    "The speech that the mentor should speak when the mentor_activity is executed",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * JammoMentorActivity:speak-once:
	 *
	 * Whether the speech should be spoken only once in a lifetime of a mentor
	 */
	g_object_class_install_property(gobject_class, PROP_SPEAK_ONCE,
	                                g_param_spec_boolean("speak-once",
	                                                     "Speak once",
	                                                     "Whether the speech should be spoken only once in a lifetime of a mentor",
	                                                     FALSE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * JammoMentorActivity:idle-speech:
	 *
	 * The speech that the mentor should speak when an user is idling.
	 * Leave this NULL, if you do not want to change idle speech when the mentor_activity is executed.
	 */
	g_object_class_install_property(gobject_class, PROP_IDLE_SPEECH,
	                                g_param_spec_string("idle-speech",
	                                                    "Idle speech",
	                                                    "The speech that the mentor should speak when an user is idling",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	/**
	 * JammoMentorActivity::spoken:
	 * @mentor_activity: the object which received the signal
	 *
	 * The ::spoken signal is emitted when the mentor_activity with :speech property has
	 * been completed (the mentor has spoken).
	 */
	signals[SPOKEN] = g_signal_new("spoken", G_TYPE_FROM_CLASS(gobject_class),
	                               G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(JammoMentorActivityClass, spoken),
				       NULL, NULL,
				       g_cclosure_marshal_VOID__STRING,
				       G_TYPE_NONE, 1,
				       G_TYPE_STRING);
	/**
	 * JammoMentorActivity::interrupted:
	 * @mentor_activity: the object which received the signal
	 *
	 * The ::interrupted signal is emitted when the mentor_activity with :speech property has
	 * been completed (the mentor has been interrupted).
	 */
	signals[INTERRUPTED] = g_signal_new("interrupted", G_TYPE_FROM_CLASS(gobject_class),
	                        	    G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(JammoMentorActivityClass, interrupted),
					    NULL, NULL,
					    g_cclosure_marshal_VOID__STRING,
					    G_TYPE_NONE, 1,
					    G_TYPE_STRING);
						   
	g_type_class_add_private (gobject_class, sizeof (JammoMentorActivityPrivate));
}

static void jammo_mentor_activity_init(JammoMentorActivity* mentor_activity) {
	mentor_activity->priv = G_TYPE_INSTANCE_GET_PRIVATE(mentor_activity, JAMMO_TYPE_MENTOR_ACTIVITY, JammoMentorActivityPrivate);
}

static void on_mentor_spoken(JammoMentor* mentor, const gchar* speech, gboolean interrupted, gpointer user_data) {
	JammoMentorActivity* mentor_activity;
	
	mentor_activity = JAMMO_MENTOR_ACTIVITY(user_data);
	
	if (interrupted) {
		g_signal_emit(mentor_activity, signals[INTERRUPTED], 0, speech);
	} else {
		g_signal_emit(mentor_activity, signals[SPOKEN], 0, speech);
	}
}
