/*
 * gems_service_control.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Tommi Kallonen <tommi.kallonen@lut.fi>
 *          Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */


#include "gems_service_control.h"
#include "gems_message_functions.h"
#include "communication.h"
#include "../cem/cem.h"


gboolean gems_service_control_process_request(gems_connection* element)
{
	gems_components* data = gems_get_data();
	
	gems_teacher_connection* tc = gems_get_teacher_connection();
	
	// Connection to teacher exists, control messages should be coming from teacher only!
	if(tc)
	{
		// Is connected
		if(ph_c_connection_is_connected(tc->connection->connection) == TRUE)
		{
			// Message is coming from teacher
			if(ph_c_connection_get_device_checksum(tc->connection->connection) == ph_c_connection_get_device_checksum(element->connection))
			{	
				switch(gems_connection_get_16(element,6)) // command
				{
					case CONTROL_MODE:
						gems_service_control_handle_control_mode(data,element);
						break;
					case CONTROL_OK:
						gems_service_control_handle_control_ok(data,element);
						break;
					case CONTROL_SET_LOGIN:
						gems_service_control_handle_set_login(data,element);
						break;
					case CONTROL_GET_LOGIN:
						gems_service_control_handle_get_login(data,element);
					default:
						break;
				}
			}
			else
			{
				cem_add_to_log("gems_service_control_process_request: Got control message from some other than teacher, ignoring.",J_LOG_NETWORK_DEBUG);
				return FALSE;
			}
		}
		else
		{
			cem_add_to_log("gems_service_control_process_request: Got control message but no teacher is connected, ignoring.",J_LOG_NETWORK_DEBUG);
			return FALSE;
		}
	}
	else
	{
		cem_add_to_log("gems_service_control_process_request: Got control message but teacher is not initialized, ignoring.",J_LOG_NETWORK_DEBUG);
		return FALSE;
	}
	
	return TRUE;
}

gems_service_control* gems_service_control_init()
{
	gems_service_control* data = (gems_service_control*)g_malloc(sizeof(gems_service_control));
	
	data->enabled = FALSE;

	data->active=FALSE;
	
	data->port = 0;

	return data;
}

void gems_service_control_cleanup()
{
	gems_service_control* data = gems_get_data()->service_control;
	
	
	if(data != NULL)
	{
		g_free(data);
		data = NULL;
	}
}

void gems_service_control_handle_control_mode(gems_components* data, gems_connection* element)
{
	gchar* logmsg = NULL;
	guint position=8;
	guint16 cntrl_type;
	guint16 lock_mode;

	// Control type
	cntrl_type = gems_connection_get_16(element,position);
	position = position + sizeof(guint16);

	// Lock mode
	lock_mode = gems_connection_get_16(element,position);
	position = position + sizeof(guint16);

	// TODO do something with new control mode
	// TODO use gems_set_waiting_for_teacher() when lock_mode == lock
	// TODO use gems_unset_waiting_for_teacher() when lock_mode == unlock

	logmsg = g_strdup_printf("gems_service_control_handle_control_mode: got CONTROL_MODE ");
	cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
	g_free(logmsg);
}

void gems_service_control_handle_control_ok(gems_components* data, gems_connection* element)
{
	gchar* logmsg = NULL;

	logmsg = g_strdup_printf("gems_service_control_handle_control_ok: got CONTROL_OK ");
	cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
	g_free(logmsg);
}

void gems_service_control_handle_set_login(gems_components* data, gems_connection* element)
{
	gchar* logmsg = NULL;
	guint position = sizeof(guint16)+sizeof(guint32)+sizeof(guint16);
	guint32 length = gems_connection_get_32(element,sizeof(guint16)) - position; // length - header
	
	gchar* loginname = gems_connection_get_data(element,position,length);
	
	// TODO if logged in, logout the user AND INFORM ! -> CALLBACK TO CHUM? Return to login screen and stop all activity?
	// if(gems_profile_manager_logout()) cem_add_to_log("gems_service_control_handle_set_login: logged out previous user",J_LOG_DEBUG);
	
	if(!gems_profile_manager_is_authenticated())
	{
		logmsg = g_strdup_printf("gems_service_control_handle_set_login: username \"%s\" will be used as login name.",loginname);
		cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
		g_free(logmsg);
		
		// load profile with loginname
		switch(gems_profile_manager_set_user(loginname))
		{
			// Found & loaded
			case CHANGE_USER_OK:
				cem_add_to_log("gems_service_control_handle_set_login: profile found",J_LOG_DEBUG);
				break;
			// Not found but request sent
			case CHANGE_USER_WAITING_PROFILE:
				cem_add_to_log("gems_service_control_handle_set_login: profile requested.",J_LOG_DEBUG);
				break;
			// Should not happen!
			case CHANGE_USER_NO_USERNAME:
				cem_add_to_log("gems_service_control_handle_set_login: teacher sent username as NULL.",J_LOG_DEBUG);
				break;
			default:
				break;
		}
	}
	else
	{
		gems_set_login_username(loginname,TRUE);
		cem_add_to_log("gems_service_control_handle_set_login: user logged in, the username will be changed after logout",J_LOG_DEBUG);
	}
	g_free(loginname);
}

void gems_service_control_handle_get_login(gems_components* data, gems_connection* element)
{
	gems_message* msg = gems_create_message_service_profile(CONTROL_REPLY_LOGIN);
	
	// not logged in, msg is NULL -> create a general message with basic header only
	if(msg == NULL) msg = gems_create_general_message(NULL,PROFILE_SERVICE_ID,0,CONTROL_REPLY_LOGIN);
		
	if(!gems_communication_write_encrypted_data(JAMMO_PACKET_PRIVATE,element,msg))
		cem_add_to_log("gems_service_control_handle_get_login: cannot write login name reply as public profile to teacher",J_LOG_NETWORK_DEBUG);
	else cem_add_to_log("gems_service_control_handle_get_login: wrote login name reply to teacher", J_LOG_NETWORK_DEBUG);

}
