/*
 * gems_teacher_server_utils.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */

#include <glib/gstdio.h>
#include "gems_teacher_server_utils.h"
#include "ProfileManager.h"
#include "gems_security.h"
#include "../configure.h"

jammo_profile_container* gems_teacher_server_new_profile_container()
{
	jammo_profile_container* container = (jammo_profile_container*)g_malloc(sizeof(jammo_profile_container));
	// Allocation failed?
	if(container == NULL) return NULL;
	
	container->userid = 0;
	container->username = NULL;
	container->password = NULL;
	container->salt = NULL;
	container->firstname = NULL;
	container->lastname = NULL;
	container->authlevel = NULL;
	container->age = 0;
	container->points = 0;
	container->avatarid = PROF_DEFAULT_AVATARID;
	container->encrypted = NULL;
	container->encrypted_length = 0;
	return container;	
}

teacher_server_profile_data* gems_teacher_server_new_profile_data_container()
{
	teacher_server_profile_data* data = (teacher_server_profile_data*)g_malloc(sizeof(teacher_server_profile_data));
	
	if(data == NULL) return NULL;
	
	data->profiles = NULL;
	data->salt = NULL;
	data->pwhash = NULL;
	data->datahash = NULL;
	data->data = NULL;
	data->datalength = 0;
	
	return data;
}

void gems_teacher_server_clear_profile_container(jammo_profile_container* container)
{
	if(container)
	{
		if(container->username) g_free(container->username);
		if(container->password) g_free(container->password);
		if(container->salt) g_free(container->salt);
		if(container->firstname) g_free(container->firstname);
		if(container->lastname) g_free(container->lastname);
		if(container->authlevel) g_free(container->authlevel);
		if(container->encrypted) g_free(container->encrypted);
		
		g_free(container);
		container = NULL;
	}
}

void gems_teacher_server_clear_profile_data_container(teacher_server_profile_data* data)
{
	if(data)
	{
		if(data->profiles) gems_teacher_server_clear_profile_list(data->profiles);
		if(data->salt) g_free(data->salt);
		if(data->pwhash) g_free(data->pwhash);
		if(data->datahash) g_free(data->datahash);
		if(data->data) g_free(data->data);
		
		g_free(data);
		data = NULL;
	}	
}

jammo_profile_container* gems_teacher_server_create_profile(guint32 _userid, gchar* _username, gchar* _password,
	gchar* _authlevel, gchar* _firstname, gchar* _lastname, guint16 _age, guint32 _avatarid)
{
	// Call setup function with NULL salt and 0 points, salt will be generated when NULL
	return gems_teacher_server_setup_profile(_userid,_username,_password,NULL,_authlevel,_firstname,_lastname,_age,0,_avatarid);
}

jammo_profile_container* gems_teacher_server_setup_profile(guint32 _userid, gchar* _username, gchar* _password,
	guchar* _salt, gchar* _authlevel, gchar* _firstname, gchar* _lastname, guint16 _age, guint32 _points, guint32 _avatarid)
{
	// check input
	if((_userid == 0) || (_username == NULL) || (_password == NULL) || (_authlevel == NULL) 
		|| (_firstname == NULL) || (_lastname == NULL) || (_age == 0)) return NULL;
	
	// Check params for illegal characters
	if(gems_check_illegal_characters(_username)) return NULL;
	if(gems_check_illegal_characters(_password)) return NULL;
	if(gems_check_illegal_characters(_authlevel)) return NULL;
	if(gems_check_illegal_characters(_firstname)) return NULL;
	if(gems_check_illegal_characters(_lastname)) return NULL;

	jammo_profile_container* container = gems_teacher_server_new_profile_container();
	
	// Set details
	container->userid = _userid;
	container->age = _age;
	container->points = _points;
	container->avatarid = _avatarid;
	container->username = g_strdup(_username);
	container->firstname = g_strdup(_firstname);
	container->lastname = g_strdup(_lastname);
	container->authlevel = g_strdup(_authlevel);
	container->password = g_strdup(_password);
	
	// No salt, used probably through gems_teacher_server_create_profile
	if(_salt == NULL) container->salt = gems_security_create_password_salt();
	else
	{
		// set salt
		container->salt = (guchar*)g_malloc0(sizeof(guchar*)*AES_SALT_LENGTH);
		g_memmove(&container->salt[0],_salt,AES_SALT_LENGTH);
	}

	if(gems_teacher_server_encrypt_profile(container)) return container;
	else
	{
		gems_teacher_server_clear_profile_container(container);
		return NULL;
	}
}

jammo_profile_container* gems_teacher_server_get_profile_for_user(teacher_server_profile_data* data, gchar* username)
{
	GList* iter = NULL;
	if(!data) return NULL;
	if(!username) return NULL;
	if(!data->profiles) return NULL;
	
	for(iter = g_list_first(data->profiles); iter ; iter = g_list_next(iter))
	{
		jammo_profile_container* profile = (jammo_profile_container*)iter->data;
		if(g_strcmp0(username,profile->username) == 0) return profile;
	}
	return NULL;
}

jammo_profile_container* gems_teacher_server_get_profile_for_user_id(teacher_server_profile_data* data, guint32 userid)
{
	GList* iter = NULL;
	if(!data) return NULL;
	if(userid == 0) return NULL;
	if(!data->profiles) return NULL;
	
	for(iter = g_list_first(data->profiles); iter ; iter = g_list_next(iter))
	{
		jammo_profile_container* profile = (jammo_profile_container*)iter->data;
		if(profile->userid == userid) return profile;
	}
	return NULL;
}

gboolean gems_teacher_server_encrypt_profile(jammo_profile_container* container)
{
	// vars
	guint encproflen = 0, hashlen = 0, profilelen = 0, passlen = 0;
	gchar* userid = NULL;
	gchar* age = NULL;
	gchar* points = NULL;
	gchar* avatarid = NULL;
	guchar* profilehash = NULL;
	guchar* encprofile = NULL;
	guchar* decprofile = NULL;
	guchar* decprofhash = NULL;
	guchar* profile = NULL;
	guchar* passwd = NULL;
	guchar* passhash = NULL;
	gboolean rval = TRUE;	

	if(container)
	{
		// Set profile data
		userid = g_strdup_printf("%u",container->userid);
		age = g_strdup_printf("%u",container->age);
		points = g_strdup_printf("%u",container->points);
		avatarid = g_strdup_printf("%u",container->avatarid);
		profile = (guchar*)g_strjoin(",", userid, container->username, container->authlevel, container->firstname, container->lastname, points, age, avatarid, NULL);
		profilelen = strlen((gchar*)profile)+1;
	
		// Create salt and password
		if(container->salt == NULL) container->salt = gems_security_create_password_salt();
		passwd = gems_security_create_password(container->password,container->salt);
		passhash = NULL;
		passlen = PASSLEN;
	
		// Initialize security
		if(gems_security_init_security(passwd,container->salt))
		{
			hashlen = PASSLEN;
			passhash = gems_security_calculate_hash(passwd,&hashlen);
			if(hashlen != SHA_HASH_LENGTH) rval = FALSE; // Invalid size hash, profile might not be valid
			
			hashlen = profilelen;
			profilehash = gems_security_calculate_hash(profile,&hashlen);
			if(hashlen != SHA_HASH_LENGTH) rval = FALSE; // Invalid size hash, profile might not be valid
			
			encprofile = gems_security_encrypt_data(profile,&profilelen);
			
			encproflen = profilelen;
			decprofile = gems_security_decrypt_data(encprofile,&encproflen);
			
			hashlen = encproflen;
			decprofhash = gems_security_calculate_hash(decprofile,&hashlen);
			if(hashlen != SHA_HASH_LENGTH) rval = FALSE; // Invalid size hash, profile might not be valid
		
			if(gems_security_verify_hash(profilehash,decprofhash,SHA_HASH_LENGTH))
			{
				container->encrypted_length = AES_SALT_LENGTH + 32 + 32 + profilelen;
				
				// Allocate content, free old if it was allocated
				if(container->encrypted != NULL) g_free(container->encrypted);
				container->encrypted = (gchar*)g_malloc(sizeof(gchar*) * container->encrypted_length);
				memset(container->encrypted,0,container->encrypted_length);
	
				// Set salt
				*(guint32*)&container->encrypted[0] = *(guint32*)&container->salt[0];
				*(guint32*)&container->encrypted[sizeof(guint32)] = *(guint32*)&container->salt[sizeof(guint32)];
	
				// Set password hash
				g_memmove(&container->encrypted[AES_SALT_LENGTH],passhash,32);
				// Set profile hash
				g_memmove(&container->encrypted[AES_SALT_LENGTH + 32],profilehash,32);
				// Set profile
				g_memmove(&container->encrypted[AES_SALT_LENGTH + 64],encprofile,profilelen);
				
			}
			else rval = FALSE;
		}
		else rval = FALSE;

		// Free all
		if(userid != NULL) g_free(userid);
		if(age != NULL) g_free(age);
		if(points != NULL) g_free(points);
		if(avatarid != NULL) g_free(avatarid);
		if(profile!= NULL) g_free(profile);
		if(profilehash != NULL) g_free(profilehash);
		if(encprofile != NULL) g_free(encprofile);
		if(decprofile != NULL) g_free(decprofile);
		if(decprofhash != NULL) g_free(decprofhash);
		if(passwd != NULL) g_free(passwd);
		if(passhash != NULL) g_free(passhash);
	
		gems_security_clear_security();
	}
	else rval = FALSE;
		
	return rval;
}

gboolean gems_teacher_server_change_password(jammo_profile_container* container, gchar* password)
{	
	if(container)
	{
		if(container->password != NULL) g_free(container->password);
		container->password = g_strdup(password);
		return gems_teacher_server_encrypt_profile(container);
	}
	else return FALSE;
}

// Write encrypted profile
gint gems_teacher_server_write_profile(jammo_profile_container* container)
{
	if(!container) return WRITE_FAIL_CONTENT_NOT_SET;
	if(!container->username) return WRITE_FAIL_CONTENT_NOT_SET;
	if(!container->encrypted) return WRITE_FAIL_CONTENT_NOT_SET;
	
	gchar* profilefile = g_strjoin(NULL,container->username,PROFILE_FILE_EXT,NULL);
	gchar* filename = g_build_filename(configure_get_jammo_directory(),JAMMOPROFILE,profilefile,NULL);
	GError* error = NULL;
	
	// Check for previous profile file
	if(g_file_test(filename,(G_FILE_TEST_EXISTS)))
	{
		if(g_remove(filename) != 0)
		{
			// Error, change username?
			g_free(filename);
			g_free(profilefile);
			return WRITE_FAIL_CANNOT_REMOVE_FILE;
		}
	}

	// Set content
  if(!g_file_set_contents(filename,container->encrypted,container->encrypted_length,&error))
  {
  	if(error)
		{
			// print error
			g_error_free(error);
			g_free(filename);
			g_free(profilefile);
			return WRITE_FAIL_CANNOT_SET_FILE_CONTENT;
		}
  }
  
	g_free(filename);
	g_free(profilefile);
	
	return WRITE_OK;
}

gboolean gems_teacher_server_save_encrypted_profile(gchar* username, gchar* profiledata, guint datalength)
{
	gboolean rval = TRUE;
	jammo_profile_container* container = gems_teacher_server_new_profile_container();
	
	container->username = g_strdup(username);
	container->encrypted = (gchar*)g_malloc0(sizeof(gchar*)*datalength);
	g_memmove(container->encrypted,profiledata,datalength);
	container->encrypted_length = datalength;

	if(gems_teacher_server_write_profile(container) != WRITE_OK) rval = FALSE;
				
	gems_teacher_server_clear_profile_container(container);
	
	return rval;
}

gchar* gems_teacher_server_serialize_all_profiles(GList* list, guint* length)
{
	gchar* all_profiles = NULL;
	jammo_profile_container* profile = NULL;
	guint total_length = 0, position = 0;
	GList* iter = NULL;
	
	for(iter = g_list_first(list); iter; iter = g_list_next(iter))
	{
		profile = (jammo_profile_container*)iter->data;
		
		// Calculate length for the huge gchar*
		total_length = total_length + sizeof(guint32) + sizeof(guint8) + strlen(profile->username) + 1 + 
			sizeof(guint8) + strlen(profile->password) + 1 + AES_SALT_LENGTH + sizeof(guint8) + 
			strlen(profile->firstname) + 1 + sizeof(guint8) + strlen(profile->lastname) + 1 +sizeof(guint8) + 
			strlen(profile->authlevel) + 1 + sizeof(guint16) + sizeof(guint32) + sizeof(guint32);
	}
	
	if((all_profiles = (gchar*)g_try_malloc0(sizeof(gchar*)*total_length)) == NULL) return NULL;
	
	for(iter = g_list_first(list); iter; iter = g_list_next(iter))
	{
		profile = (jammo_profile_container*)iter->data;
		
		// id
		*(guint32*)&all_profiles[position] = profile->userid;
		position = position + sizeof(guint32);
		
		// username length 
		all_profiles[position] = (guint8)strlen(profile->username) + 1;
		position++;
		
		// username
		g_memmove(&all_profiles[position],profile->username,strlen(profile->username) + 1);
		position = position + strlen(profile->username) + 1;
		
		// password length
		all_profiles[position] = (guint8)strlen(profile->password) + 1;
		position++;
		
		// password
		g_memmove(&all_profiles[position],profile->password,strlen(profile->password) + 1);
		position = position + strlen(profile->password) + 1;
		
		// salt - fixed size
		g_memmove(&all_profiles[position],profile->salt,AES_SALT_LENGTH);
		position = position + AES_SALT_LENGTH;
		
		// firstname length
		all_profiles[position] = (guint8)strlen(profile->firstname) + 1;
		position++;
		
		// firstname
		g_memmove(&all_profiles[position],profile->firstname,strlen(profile->firstname) + 1);
		position = position + strlen(profile->firstname) + 1;
		
		// lastname length
		all_profiles[position] = (guint8)strlen(profile->lastname) + 1;
		position++;
		
		// lastname
		g_memmove(&all_profiles[position],profile->lastname,strlen(profile->lastname) + 1);
		position = position + strlen(profile->lastname) + 1;
		
		// authlevel length
		all_profiles[position] = (guint8)strlen(profile->authlevel) + 1;
		position++;
		
		// authlevel
		g_memmove(&all_profiles[position],profile->authlevel,strlen(profile->authlevel) + 1);
		position = position + strlen(profile->authlevel) + 1 ;
		
		// age
		*(guint16*)&all_profiles[position] = profile->age;
		position = position + sizeof(guint16);
		
		// points
		*(guint32*)&all_profiles[position] = profile->points;
		position = position + sizeof(guint32);
		
		// avatar id
		*(guint32*)&all_profiles[position] = profile->avatarid;
		position = position + sizeof(guint32);
	}
	
	*length = total_length;
	return all_profiles;
}

GList* gems_teacher_server_deserialize_profiles(GList* list, gchar* serialized_data, guint length)
{
	jammo_profile_container* profile = NULL;
	
	guint32 id = 0;
	gchar* username = NULL;
	gchar* password = NULL;
	guchar* salt = NULL;
	gchar* firstname = NULL;
	gchar* lastname = NULL;
	gchar* authlevel = NULL;
	guint16 age = 0;
	guint32 points = 0;
	guint32 avatarid = 0;
	
	guint position = 0;
	guint8 valuelength = 0;
	
	while(position < length)
	{
		// user id
		id = *(guint32*)&serialized_data[position];
		position = position + sizeof(guint32);
		
		// length of username
		valuelength = *(guint8*)&serialized_data[position];
		position++;
		
		// username
		username = (gchar*)g_malloc0(sizeof(gchar*)*(valuelength));
		g_memmove(username,&serialized_data[position],valuelength);
		position = position + valuelength;
		
		// length of password
		valuelength = *(guint8*)&serialized_data[position];
		position++;
		
		// password
		password = (gchar*)g_malloc0(sizeof(gchar*)*(valuelength));
		g_memmove(password,&serialized_data[position],valuelength);
		position = position + valuelength;
		
		// salt
		salt = (guchar*)g_malloc0(sizeof(guchar*)*AES_SALT_LENGTH);
		g_memmove(&salt[0],&serialized_data[position],AES_SALT_LENGTH);
		position = position + AES_SALT_LENGTH;
		
		// length of firstname
		valuelength = *(guint8*)&serialized_data[position];
		position++;
		
		// firstname
		firstname = (gchar*)g_malloc0(sizeof(gchar*)*(valuelength));
		g_memmove(firstname,&serialized_data[position],valuelength);
		position = position + valuelength;
		
		// length of lastname
		valuelength = *(guint8*)&serialized_data[position];
		position++;
		
		// lastname
		lastname = (gchar*)g_malloc0(sizeof(gchar*)*(valuelength));
		g_memmove(lastname,&serialized_data[position],valuelength);
		position = position + valuelength;
		
		// length of authlevel
		valuelength = *(guint8*)&serialized_data[position];
		position++;
		
		// authlevel
		authlevel = (gchar*)g_malloc0(sizeof(gchar*)*(valuelength));
		g_memmove(authlevel,&serialized_data[position],valuelength);
		position = position + valuelength;
		
		// age
		age = *(guint16*)&serialized_data[position];
		position = position + sizeof(guint16);
		
		// points
		points = *(guint32*)&serialized_data[position];
		position = position + sizeof(guint32);
		
		// avatarid
		avatarid = *(guint32*)&serialized_data[position];
		position = position + sizeof(guint32);
		
		// create profile and add to list
		if((profile = gems_teacher_server_setup_profile(id,username,password,salt,authlevel,firstname,
			lastname,age,points,avatarid)) != NULL) list = g_list_append(list,profile);
		
		g_free(username);
		username = NULL;
		g_free(password);
		password = NULL;
		g_free(salt);
		salt = NULL;
		g_free(authlevel);
		authlevel = NULL;
		g_free(firstname);
		firstname = NULL;
		g_free(lastname);
		lastname = NULL;
		
	}
	return list;
}

// Clear list
void gems_teacher_server_clear_profile_list(GList* list)
{
	if(list) g_list_foreach(list,(GFunc)gems_teacher_server_clear_profile_container,NULL);
	g_list_free(list);
	list = NULL;
}

// For debugging
void gems_teacher_server_print_profile_list(GList* list)
{
	if(list) g_list_foreach(list,(GFunc)gems_teacher_server_print_profile,NULL);
}
void gems_teacher_server_print_profile(jammo_profile_container* profile)
{
	printf("%u:%s:%s:%s:%s:%s:%u:%u:%u\n",profile->userid, profile->username, profile->password,profile->authlevel,
		profile->firstname,profile->lastname,profile->age,profile->points,profile->avatarid);
}

// serialize all data and encrypt
gboolean gems_teacher_server_encrypt_profiles(teacher_server_profile_data* data, gchar* passwd)
{
	gboolean rval = TRUE;
	gchar* serialized = NULL;
	guchar* decrypted = NULL;
	guchar* dechash = NULL;
	guchar* password = NULL;
	guint encrypted_length = 0;
	guint hash_length = 0;
	
	// check params and content
	if(data == NULL || passwd == NULL) return FALSE;
	if(data->profiles == NULL) return FALSE;	
	
	// serialize data
	if((serialized = gems_teacher_server_serialize_all_profiles(data->profiles, &data->datalength)) == NULL) return FALSE;
	
	// init salt and passwd
	if(data->salt == NULL) data->salt = gems_security_create_password_salt();
	password = gems_security_create_password(passwd,data->salt);
	
	// init security
	if(gems_security_init_security(password,data->salt))
	{
		// Password hash
		if(data->pwhash != NULL) g_free(data->pwhash);
		hash_length = PASSLEN;
		data->pwhash = gems_security_calculate_hash(password,&hash_length);
		if(hash_length != SHA_HASH_LENGTH) rval = FALSE; // Invalid size hash, profile might not be valid
		
		// Profile data hash
		if(data->datahash != NULL) g_free(data->datahash);
		hash_length = data->datalength;
		data->datahash = gems_security_calculate_hash((guchar*)serialized,&hash_length);
		if(hash_length != SHA_HASH_LENGTH) rval = FALSE; // Invalid size hash, profile might not be valid
		
		// encrypt
		if(data->data != NULL) g_free(data->data);
		data->data = gems_security_encrypt_data((guchar*)serialized,&(data->datalength));
		
		// decrypt
		encrypted_length = data->datalength;
		decrypted = gems_security_decrypt_data(data->data,&encrypted_length);
		
		// hash decrypted
		hash_length = encrypted_length;
		dechash = gems_security_calculate_hash(decrypted,&hash_length);
		if(hash_length != SHA_HASH_LENGTH) rval = FALSE; // Invalid size hash, profile might not be valid
	
		// verify hashes
		if(!gems_security_verify_hash(data->datahash,dechash,SHA_HASH_LENGTH)) rval = FALSE;
		gems_security_clear_security();
	}
	else rval = FALSE;
	
	if(serialized != NULL) g_free(serialized);
	if(decrypted != NULL) g_free(decrypted);
	if(dechash != NULL) g_free(dechash);
	if(password != NULL) g_free(password);
	
	return rval;
}

// decrypt data-field content and add pass decrypted to deserialization
gboolean gems_teacher_server_decrypt_profiles(teacher_server_profile_data* data, gchar* passwd)
{
	gboolean rval = TRUE;
	guchar* password = NULL;
	guchar* pwhash = NULL;
	guchar* decrypted = NULL;
	guchar* dechash = NULL;
	guint hashlength = 0;
	
	if(!data || !passwd) return FALSE;
	if((!data->salt) || (!data->pwhash) || (!data->datahash) || (!data->data) || (data->datalength == 0))
		return FALSE;
		
	// form password
	password = gems_security_create_password(passwd,data->salt);
	
	if(gems_security_init_security(password,data->salt))
	{
		// hash pass
		hashlength = PASSLEN;
		pwhash = gems_security_calculate_hash(password,&hashlength);
		if(hashlength != SHA_HASH_LENGTH) rval = FALSE;
		
		// password verified
		if(gems_security_verify_hash(pwhash,data->pwhash,SHA_HASH_LENGTH))
		{
			// decrypt
			decrypted = gems_security_decrypt_data(data->data,&(data->datalength));
			
			// hash
			hashlength = data->datalength;
			dechash = gems_security_calculate_hash(decrypted,&hashlength);
			if(hashlength != SHA_HASH_LENGTH) rval = FALSE;
			
			// verify hash
			if(gems_security_verify_hash(data->datahash,dechash,SHA_HASH_LENGTH))
			{
				if(data->profiles != NULL)
				{
					gems_teacher_server_clear_profile_list(data->profiles);
					data->profiles = NULL;
				}
				// Clear own security contexts BEFORE deserializing profiles
				gems_security_clear_security();
				// deserialize
				data->profiles = gems_teacher_server_deserialize_profiles(data->profiles,(gchar*)decrypted,data->datalength);
			}
			else rval = FALSE;
		}
		else rval = FALSE;
	}
	else rval = FALSE;
	
	if(password) g_free(password);
	if(pwhash) g_free(pwhash);
	if(decrypted) g_free(decrypted);
	if(dechash) g_free(dechash);
	
	return rval;
}

// write encrypted data to disk
gint gems_teacher_server_write_profile_database(teacher_server_profile_data* data)
{
	if(!data) return WRITE_FAIL_CONTENT_NOT_SET;
	if(!data->salt) return WRITE_FAIL_CONTENT_NOT_SET;
	if(!data->pwhash) return WRITE_FAIL_CONTENT_NOT_SET;
	if(!data->datahash) return WRITE_FAIL_CONTENT_NOT_SET;
	if(!data->data) return WRITE_FAIL_CONTENT_NOT_SET;
	if(data->datalength == 0) return WRITE_FAIL_CONTENT_NOT_SET;
	
	gchar* filename = g_build_filename(configure_get_jammo_directory(),TSRV_DATABASE,NULL);
	GError* error = NULL;
	gchar* content = NULL;
	guint contentlength = 0;
	
	// Check for previous profile file
	if(g_file_test(filename,(G_FILE_TEST_EXISTS)))
	{
		if(g_remove(filename) != 0)
		{
			// Error, change username?
			g_free(filename);
			return WRITE_FAIL_CANNOT_REMOVE_FILE;
		}
	}
	
	contentlength = AES_SALT_LENGTH + SHA_HASH_LENGTH + SHA_HASH_LENGTH + data->datalength;
			
	content = (gchar*)g_malloc0(sizeof(gchar*) * contentlength);

	// Set salt
	*(guint32*)&content[0] = *(guint32*)&data->salt[0];
	*(guint32*)&content[sizeof(guint32)] = *(guint32*)&data->salt[sizeof(guint32)];

	// Set password hash
	g_memmove(&content[AES_SALT_LENGTH],data->pwhash,SHA_HASH_LENGTH);
	// Set profile hash
	g_memmove(&content[AES_SALT_LENGTH + SHA_HASH_LENGTH],data->datahash,SHA_HASH_LENGTH);
	// Set profile
	g_memmove(&content[AES_SALT_LENGTH + SHA_HASH_LENGTH + SHA_HASH_LENGTH],data->data,data->datalength);
	

	// Set content
  if(!g_file_set_contents(filename,content,contentlength,&error))
  {
  	if(error)
		{
			// print error
			g_error_free(error);
			g_free(content);
			g_free(filename);
			return WRITE_FAIL_CANNOT_SET_FILE_CONTENT;
		}
  }
  g_free(content);
	g_free(filename);
	return WRITE_OK;
}

// read the database from disk
gint gems_teacher_server_read_profile_database(teacher_server_profile_data** data)
{
	gchar* filename = g_build_filename(configure_get_jammo_directory(),TSRV_DATABASE,NULL);
	gchar* contents = NULL;
	guint length = 0;
	GError* error = NULL;
	gint rval = READ_OK;
	
	// Test if profile file exists;
	if(g_file_test(filename,G_FILE_TEST_IS_REGULAR))
	{
		// Get the file content
		if(!g_file_get_contents(filename,&contents,(gsize*)&length,&error))
		{
			// Cannot read, TODO report error.
			if(error) g_error_free(error);
			rval = READ_FAIL_CANNOT_READ;
		}
		else
		{
			// Extract salt
			if((*data)->salt != NULL) g_free((*data)->salt);
			(*data)->salt = (guchar*)g_malloc0(sizeof(guchar*) * AES_SALT_LENGTH);
			*(guint32*)&(*data)->salt[0] = *(guint32*)&contents[0];
			*(guint32*)&(*data)->salt[sizeof(guint32)] = *(guint32*)&contents[sizeof(guint32)];
			
			// Extract password hash
			if((*data)->pwhash != NULL) g_free((*data)->pwhash);
			(*data)->pwhash = (guchar*)g_malloc0(sizeof(guchar*) * (SHA_HASH_LENGTH + 1));
			g_memmove((*data)->pwhash,&contents[AES_SALT_LENGTH],SHA_HASH_LENGTH);
			
			// Extract profile hash
			if((*data)->datahash != NULL) g_free((*data)->datahash);
			(*data)->datahash = (guchar*)g_malloc0(sizeof(guchar*) * (SHA_HASH_LENGTH + 1));
			g_memmove((*data)->datahash,&contents[AES_SALT_LENGTH + SHA_HASH_LENGTH],SHA_HASH_LENGTH);
			
			// Extract encrypted profile
			if((*data)->data != NULL) g_free((*data)->data);
			(*data)->datalength = length - SHA_HASH_LENGTH - SHA_HASH_LENGTH - AES_SALT_LENGTH;
			(*data)->data = (guchar*)g_malloc0(sizeof(guchar*) * ((*data)->datalength));
			g_memmove((*data)->data,&contents[AES_SALT_LENGTH + SHA_HASH_LENGTH + SHA_HASH_LENGTH],(*data)->datalength);
			
			rval = READ_OK;
		}
	}
	else
	{
		rval = READ_FAIL_FILE_NOT_EXIST;
	}
  
	if(contents != NULL) g_free(contents);
	if(filename != NULL) g_free(filename);

	return rval;
}

// TODO list profiles on disk
// TODO load profiles on disk to jammo_profile_container*
// TODO search profile by username & password hash
// TODO process profile request
// TODO send profile to user (use jammo_profile_container*
