/*
 * gems_service_profile.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */
 
#include <glib.h>
#include <string.h>
#include "gems_service_profile.h"
#include "gems_message_functions.h"
#include "gems_security.h"
#include "gems_teacher_server_utils.h"
#include "communication.h"
#include "../cem/cem.h"

gems_service_profile* gems_service_profile_init()
{
	gems_service_profile* data = (gems_service_profile*)g_malloc(sizeof(gems_service_profile));
	data->enabled = FALSE;
	data->port = 0;
	return data;
}

void gems_service_profile_cleanup()
{
	gems_service_profile* data = gems_get_data()->service_profile;
	gems_service_profile_cleanup_lists();
	g_free(data);
	data = NULL;
}

void gems_service_profile_cleanup_lists()
{
	// TODO clear what?
	return;
}

gboolean gems_service_profile_process_request(gems_connection* element)
{
	gboolean success = TRUE;
	gems_message* profilemsg = NULL;
	guint16 command = gems_connection_get_16(element,(sizeof(guint16)+sizeof(guint32)));
	guint position = 0;
	gchar* logmsg = NULL;
	
	switch(command)
	{
		case GET_PROFILE:
		
			if(gems_connection_get_16(element,(sizeof(guint16)+sizeof(guint32)+sizeof(guint16))) == GET_PROFILE_PUB)
			{
				profilemsg = gems_create_message_service_profile(PROFILE_PUB);
				
				if(profilemsg != NULL)
				{					
					// Send
					if(gems_communication_write_encrypted_data(JAMMO_PACKET_PRIVATE,element,profilemsg) == FALSE)
					{
						logmsg = g_strdup_printf("gems_service_profile_process_request: cannot write PROFILE_PUB");
						cem_add_to_log(logmsg,J_LOG_ERROR);
						g_free(logmsg);
					}
					// Free message
					gems_clear_message(profilemsg);
				}
			}
			else if(gems_connection_get_16(element,(sizeof(guint16)+sizeof(guint32)+sizeof(guint16))) == GET_PROFILE_FULL)
			{
				gems_teacher_connection* tc = gems_get_teacher_connection();
	
				// Connection to teacher exists
				if(tc)
				{
					// Is connected
					if(ph_c_connection_is_connected(tc->connection->connection) == TRUE)
					{
						// Message is coming from teacher
						if(ph_c_connection_get_device_checksum(tc->connection->connection) == ph_c_connection_get_device_checksum(element->connection))
						{
							// TODO check that is authenticated as real teacher!
							
							profilemsg = gems_create_message_service_profile(PROFILE_FULL);
							
							if(profilemsg)
							{
								// Send
								if(gems_communication_write_encrypted_data(JAMMO_PACKET_PRIVATE,element,profilemsg) == FALSE)
									cem_add_to_log("gems_service_profile_process_request: cannot write PROFILE_FULL to teacher.",J_LOG_ERROR);
					
								// Free message
								gems_clear_message(profilemsg);
							}
						}
						else
						{
							cem_add_to_log("gems_service_profile_process_request: PROFILE_FULL request from other than teacher, ignored",J_LOG_NETWORK_DEBUG);
							success = FALSE;
						}
					}
					else
					{
						cem_add_to_log("gems_service_profile_process_request: PROFILE_FULL request received, no teacher connected, ignored",J_LOG_NETWORK_DEBUG);
						success = FALSE;
					}
				}
				else
				{
					cem_add_to_log("gems_service_profile_process_request: PROFILE_FULL request received, no teacher is set, ignored",J_LOG_NETWORK_DEBUG);
					success = FALSE;
				}
				
				success = TRUE;
			}
			else
			{
				logmsg = g_strdup_printf("gems_service_profile_process_request: requesting other than PROFILE_PUB/PROFILE_FULL");
				cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
				g_free(logmsg);
			}
			break;
			
		case PROFILE_PUB:
			// stop timer
			g_timer_stop(element->profile_request_timer);
			
			position = sizeof(guint16) + sizeof(guint32) + sizeof(guint16);
			// Username
			gchar* uname = gems_connection_get_char(element,position);
			position = position + strlen(uname) + 1; // username is followed by '\0'
			
			// User id
			guint32 uid = gems_connection_get_32(element,position);
			position = position + sizeof(guint32);
			
			// age
			guint16 age = gems_connection_get_16(element,position);
			position = position + sizeof(guint16);
			
			// avatar id
			guint32 aid = gems_connection_get_32(element,position);
			
			// Already created?
			if(element->profile == NULL) element->profile = gems_new_peer_profile(uname,uid,age,aid);
			else gems_peer_profile_set_values(element->profile,uname,uid,age,aid);
			
			logmsg = g_strdup_printf("gems_service_profile_process_request: Got profile:\n\tId:\t%u\n\tName:\t%s\n\tAge:\t%d\n\tAvtr:\t%u",
				element->profile->id,element->profile->username,element->profile->age,element->profile->avatarid);
			cem_add_to_log(logmsg,J_LOG_NETWORK_DEBUG);
			g_free(logmsg);
			
			success = TRUE;
			break;
		//case PROFILE_FULL:
		default:
			success = FALSE;
			break;
	}
	
	return success;
}

gboolean gems_service_profile_process_profile_upload(gems_connection* element)
{
	gboolean success = TRUE;
	
	gems_teacher_connection* tc = gems_get_teacher_connection();
	
	// Connection to teacher exists
	if(tc)
	{
		// Is connected
		if(ph_c_connection_is_connected(tc->connection->connection) == TRUE)
		{
			// Message is coming from teacher
			if(ph_c_connection_get_device_checksum(tc->connection->connection) == ph_c_connection_get_device_checksum(element->connection))
			{
				// TODO check that is authenticated as real teacher!
				
				// Whole msg length
				guint position = sizeof(guint16);
				guint32 msglen = gems_connection_get_32(element,position);
				
				// Username length
				position = position + sizeof(guint32)+sizeof(guint16);
				guint16 unamelen = gems_connection_get_16(element,position);
				
				// Username
				position = position + sizeof(guint16);
				gchar* username = gems_connection_get_data(element,position,unamelen);
				
				// Profile length and profile
				position = position + unamelen;
				guint profilelen = msglen - position;
				gchar* profile = gems_connection_get_data(element,position,profilelen);
				
				// if logged in with same username, do not allow!
				if(gems_profile_manager_is_authenticated() && g_strcmp0(username,gems_profile_manager_get_username(NULL)) == 0)
				{
					cem_add_to_log("Got profile from teacher for user who has already logged in, discarding profile",J_LOG_DEBUG);
				}
				else
				{
					// Write profile
					if(gems_teacher_server_save_encrypted_profile(username, profile, profilelen))
					{
						cem_add_to_log("Wrote encrypted profile to disk",J_LOG_DEBUG);
						
						// Same profile already loaded
						if(gems_profile_manager_get_loaded_username() != NULL)
						{
							if(g_strcmp0(username,gems_profile_manager_get_loaded_username()) == 0)
							{
								cem_add_to_log("Got profile from teacher for user whose profile is already loaded, clearing loaded.",J_LOG_DEBUG);
								if(profilemanager_storage_reload_profile(gems_get_profile_manager())) cem_add_to_log("Profile reloaded",J_LOG_DEBUG);
								else cem_add_to_log("Cannot reload profile",J_LOG_DEBUG);
							}
						}
						
						// Check if this is a response for the request
						if(gems_is_waiting_profile_from_teacher())
						{
							gchar* loginname = gems_get_login_username();
							if(loginname)
							{
								// Got the profile for our request
								if(g_strcmp0(loginname,username))
								{
									// Make sure that timeout function is removed
									if(gems_get_data()->profile_request_timeout_sid != 0) g_source_remove(gems_get_data()->profile_request_timeout_sid);
									
									// Destroy timer and set to NULL
									if(gems_get_data()->teacher_profile_request_timer)
									{
										g_timer_destroy(gems_get_data()->teacher_profile_request_timer);
										gems_get_data()->teacher_profile_request_timer = NULL;
									}
									
									// Allow to continue with login
									gems_unset_waiting_profile_from_teacher();
								}
							}
						}
					}
					else cem_add_to_log("Could not write encrypted profile to disk",J_LOG_ERROR);
				}
				
				g_free(profile);
				g_free(username);
			}
			else success = FALSE;
		}
		else success = FALSE;
	}
	else success = FALSE;
	
	return success;
}
