/*
 * ProfileManager.h
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2011 Lappeenranta University of Technology
 *
 * Authors: Janne Parkkila
 * 					Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */
 
#ifndef __GEMS_PROFILEMANAGER_H_
#define __GEMS_PROFILEMANAGER_H_

#include "StorageAgent.h"
#ifdef __cplusplus
#include <iostream>
#include <string>
#include <glib.h>

using namespace std;

class ProfileManager {
public:
	
	// Constructor, creates a server
	// for sharing profile info over peerhood
	static ProfileManager* GetInstance();
	
	// Destructor
	~ProfileManager();
	
	// Login with certain userID
	gboolean Login(guint32 userId);
	// Logout
	gboolean Logout();
	// Show user details
	string ViewDetails();
	// Get single detail
	string ViewDetail(string& detail);
	// Edit own details
	gboolean EditDetails(string& detail, string& newValue);
	// Push another user to service
	gboolean ForceUser(guint32 userID);
	// Get id
	guint32 GetId();
	
	gboolean loadEncryptedProfile(gchar* _username);
	gboolean saveEncryptedProfile();
	gboolean changeLoadedUser(gchar* _username);
	gboolean deserializeProfile(gchar* _profile);
	gchar* serializeProfile();
	
	gchar* getLoadedUsername();
	guchar* getSalt();
	guchar* getPasswordHash();
	guchar* getProfileHash();
	guchar* getEncryptedProfile();
	guint getEncryptedProfileSize();
	
	gboolean setSalt(guchar* _salt);
	gboolean setPasswordHash(guchar* _password_hash);
	gboolean setEncryptedProfile(guchar* _profile_hash, guchar* _encrypted_profile, guint _e_profile_size);
	
	gboolean isSavingRequired();
	gboolean isAuthenticated();
	gboolean setAuthenticated();

private:
	
	// Constructor, creates a server
	// for sharing profile info over peerhood
	ProfileManager();

	static ProfileManager* iInstance;
	StorageAgent *iStorageAgent;
	//ProcessAgent *iProcessAgent;
	guint32 iUser;
	gboolean iSavingRequired;
};

#else
	typedef
		struct ProfileManager
			ProfileManager;
#endif // __cplusplus

#ifdef __cplusplus
extern "C" {
#endif

void log_wrap(char* msg, int info);

/*//////////////////////////////////////////////////////
||||||||||| DO NOT USE THESE DIRECTLY !!!! |||||||||||||
||||||| USE FUNCTIONS IN gems_profile_manager.c  |||||||
////////////////////////////////////////////////////////*/

// ---------------- C wrappers -------------------------//

/* Create new profile manager interface
 *
 * Wrapper for ProfileManager::ProfileManager() 
 *
 * Returns: a pointer to struct ProfileManager
 *
 * Use the struct only through functions described here!
 */
ProfileManager* profilemanager_new_manager();


/* Delete profile manager
 *
 * Wrapper for ProfileManager::~ProfileManager() 
 *
 * Params:
 *  ProfileManager* _aManager - object to be destroyed
 */ 
void profilemanager_delete_manager(ProfileManager* _aManager);


/* Login to ProfileManager
 *
 * Wrapper for bool ProfileManager::Login(int)
 *
 * Params:
 *  ProfileManager* _aManager - profile manager to use
 *  int _userId - id of the user
 *
 * Returns: int, 0 if false, 1 if true
 */
gboolean profilemanager_login(ProfileManager* _aManager, guint32 _userId);


/* Login to ProfileManager
 *
 * Wrapper for bool ProfileManager::Logout()
 *
 * Params:
 *  ProfileManager* _aManager - profile manager to use
 *
 * Returns: int, 0 if false, 1 if true
 */
gboolean profilemanager_logout(ProfileManager* _aManager);


/* View a single detail
 *
 * Wrapper for string ProfileManager::ViewDetail(string)
 * 
 * Params:
 *  ProfileManager* _aManager - profile manager to use
 *  char* _parameter - parameter to retrieve
 *
 * Returns: char* containing the value, or empty if not found
 */
const gchar* profilemanager_view_info(ProfileManager* _aManager,
	gchar* _parameter);


/* Edit own detail
 * 
 * Wrapper for bool ProfileManager::EditDetails(string, string);
 *
 * Params:
 *  ProfileManager* _aManager - profile manager to use
 *  char* _detail - Detail to be changed
 *  char* _value - Value for the detail to be changed
 *
 * Returns: int, 0 - detail was not found, 1 - detail was found (val. changed)
 */
gboolean profilemanager_edit_info(ProfileManager* _aManager, gchar* _parameter,
	gchar* _value);

guint32 profilemanager_get_userid(ProfileManager* _aManager);

/* Get own username */
const gchar* profilemanager_get_username(ProfileManager* _aManager);

/* Get firstname */
const gchar* profilemanager_get_firstname(ProfileManager* _aManager);

/* Get lastname */
const gchar* profilemanager_get_lastname(ProfileManager* _aManager);

/* Get age */
guint16 profilemanager_get_age(ProfileManager* _aManager);

/* Get points */
guint32 profilemanager_get_points(ProfileManager* _aManager);

/* Get avatar id */
guint32 profilemanager_get_avatarid(ProfileManager* _aManager);

/* Set own username */
gboolean profilemanager_set_username(ProfileManager* _aManager, gchar* _value);

/* Set firstname */
gboolean profilemanager_set_firstname(ProfileManager* _aManager, gchar* _value);

/* Set lastname */
gboolean profilemanager_set_lastname(ProfileManager* _aManager, gchar* _value);

/* Set age - limit is 120 */
gboolean profilemanager_set_age(ProfileManager* _aManager, guint16 _value);

/* Get avatar id */
gboolean profilemanager_set_avatarid(ProfileManager* _aManager, guint32 _value);

/* Add points */
gboolean profilemanager_add_points(ProfileManager* _aManager, guint32 _value);

/* Remove points */
gboolean profilemanager_remove_points(ProfileManager* _aManager, guint32 _value);

/* Reset points (set to zero) */
void profilemanager_reset_points(ProfileManager* _aManager);

/* For encrypted profile data */

gboolean profilemanager_storage_load_profile(ProfileManager* _aManager, gchar* _username);
gboolean profilemanager_storage_save_profile(ProfileManager* _aManager);
gboolean profilemanager_storage_change_profile(ProfileManager* _aManager, gchar* _username);
guchar* profilemanager_storage_get_password_salt(ProfileManager* _aManager);
guchar* profilemanager_storage_get_password_hash(ProfileManager* _aManager);
guchar* profilemanager_storage_get_profile_hash(ProfileManager* _aManager);
guchar* profilemanager_storage_get_encrypted_profile(ProfileManager* _aManager);
guint profilemanager_storage_get_encrypted_profile_size(ProfileManager* _aManager);
gboolean profilemanager_storage_set_password_salt(ProfileManager* _aManager, guchar* _salt);
gboolean profilemanager_storage_set_password_hash(ProfileManager* _aManager, guchar* _hash);
gboolean profilemanager_storage_set_encrypted_profile(ProfileManager* _aManager, guchar* _profile_hash, guchar* _encrypted_profile, guint _e_profile_size);

gboolean profilemanager_storage_deserialize_profile(ProfileManager* _aManager, guchar* _profile);
gchar* profilemanager_storage_serialize_profile(ProfileManager* _aManager);

gboolean profilemanager_set_authenticated(ProfileManager* _aManager);
gboolean profilemanager_is_authenticated(ProfileManager* _aManager);

gboolean profilemanager_is_saving_required(ProfileManager* _aManager);
gchar* profilemanager_get_loaded_username(ProfileManager* _aManager);
#ifdef __cplusplus
}
#endif

#endif /*__GEMS_PROFILEMANAGER_H_ */

