/*
 * ProfileManager.cc
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2011 Lappeenranta University of Technology
 *
 * Authors: Janne Parkkila
 * 					Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */
 
#include <iostream>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include <sstream>
#include <cstdlib>
#include "ProfileManager.h"
#include "gems_definitions.h"

extern "C" {
	#include "../cem/cem.h"
}

using namespace std;

ProfileManager* ProfileManager::iInstance = NULL;

ProfileManager* ProfileManager::GetInstance() 
{
	// No instance->create
	if (iInstance == NULL) iInstance = new ProfileManager;
	
	return iInstance;
}

// Constructor
ProfileManager::ProfileManager()
{
	iSavingRequired = FALSE;
	iStorageAgent = new StorageAgent();
	// process agent not yet used.
	//iProcessAgent = NULL;
}

// Destructor
ProfileManager::~ProfileManager()
{
	if(iStorageAgent != NULL)
	{
		delete iStorageAgent;
		iStorageAgent = NULL;
	}
	/*if(iProcessAgent != NULL)
	{
		delete iProcessAgent;
		iProcessAgent = NULL;
	}*/
}

// Login, calls StorageAgent to retrieve the user
// profile data
gboolean ProfileManager::Login(guint32 userID)
{
	gchar* logmsg = g_strdup_printf("ProfileManager::Login() : attempted to use deprecated function!");
	log_wrap(logmsg, J_LOG_ERROR);
	g_free(logmsg);
	return FALSE;
	// Store the id
	//iUser = userID;
	
	//string msg = "User Profile ";
	// C++ type int to string conversion
	//stringstream user;
	//user << userID;
	//msg.append(user.str());
	
	// If profile can be retrieved correctly,
	// create a server for displaying the profile
	// to other users
	/*if (iStorageAgent->Retrieve(iUser) == true)
	{
		msg.append(" succesfully retrieved");
		log_wrap((char*)msg.c_str(),J_LOG_INFO);
		return true;
	}
	else
	{
		msg.append(" cannot be found.");
		log_wrap((char*)msg.c_str(),J_LOG_INFO);
		return false;
	}*/
	
}

// Logout
gboolean ProfileManager::Logout()
{
	// Clear the user from the memory
	iUser = 0;
	// Clear data (sets authenticated to FALSE)
	iStorageAgent->clearDetails();
	// Clear encrypted data, should be saved before logout
	iStorageAgent->clearEncryptedData();

	// Not logged in, cannot log out but clear the data just to avoid leaking
	if(!isAuthenticated()) return FALSE;
	else return TRUE;
}

string ProfileManager::ViewDetails()
{
	// check if authenticated
	if(isAuthenticated()) return iStorageAgent->getDetails();
	else return "";
}

string ProfileManager::ViewDetail(string& detail)
{
	if(isAuthenticated()) return iStorageAgent->getDetail(detail);
	else return "";
}

// Edit user details. Takes the name of the detail and the new Value
// as a string
gboolean ProfileManager::EditDetails(string& detail, string& newValue)
{
	// Cannot set details if not authenticated
	if(!isAuthenticated()) return FALSE;
	
	// If the detail changing operation is failure, function return false
	if(!iStorageAgent->setDetails(detail, newValue)) return FALSE;
	// Changed, requires saving
	iSavingRequired = TRUE;
	return TRUE;
}

// This function is used to Force another user in to the
// profile manager. Requires admin level in the profile
gboolean ProfileManager::ForceUser(guint32 userID)
{
	string auth;
	// Get detail authLevel from the profile
	// if it is admin, the user is allowed to force
	// another user to the machine
	string param = string("authLevel");
	auth = iStorageAgent->getDetail(param);
	if (auth == "admin")
	{
		// Logout the admin and login wit the
		// wanted userID
		if (Logout() == true)
		{
			Login(userID);
			return TRUE;
		}
		else return FALSE;
	}
	else return FALSE;
}

guint32 ProfileManager::GetId()
{
	return iUser;
}

gboolean ProfileManager::loadEncryptedProfile(gchar* _username)
{
	if(iStorageAgent->isProfileLoaded()) return TRUE; // TODO check if user is the same as previously loaded!
	
	switch(iStorageAgent->readProfile(_username))
	{
		case 0:
			return TRUE;
		case 1:
		case 2:
		default:
			return FALSE;
	}
}

gboolean ProfileManager::saveEncryptedProfile()
{
	string param = string(PROF_PARAM_USERNAME);
	gchar* username = (gchar*)ViewDetail(param).c_str();

	if(username == NULL) return FALSE;
	
		switch(iStorageAgent->writeProfile(username))
		{
			// Errors accessing file
			case -1:
				iSavingRequired = TRUE;
				return FALSE;
			// OK
			case 0:
				iSavingRequired = FALSE;
				return TRUE;
			
			// Salt missing
			case 2:
			// Password hash missing
			case 3:
			// Profile hash missing
			case 4:
			// encrypted profile missing
			case 5:
			// encrypted profile size missing
			case 6:
			default:
				iSavingRequired = TRUE;
				return FALSE;
		}
}

gboolean ProfileManager::changeLoadedUser(gchar* _username)
{
	// clear previous user data
	iStorageAgent->clearEncryptedData();
	iStorageAgent->clearDetails();
	// Load new
	return loadEncryptedProfile(_username);
}

gboolean ProfileManager::deserializeProfile(gchar* _profile)
{
	gboolean rval = iStorageAgent->deserializeProfile(_profile);
	
	if(rval)
	{
		string param = string(PROF_PARAM_USERID);
		iUser = (guint32)strtoul(iStorageAgent->getDetail(param).c_str(),NULL,10);
	}
	return rval;
}

gchar* ProfileManager::serializeProfile()
{
	return iStorageAgent->serializeProfile();
}

guchar* ProfileManager::getSalt()
{
	return iStorageAgent->getSalt();
}

guchar* ProfileManager::getPasswordHash()
{
	return iStorageAgent->getPasswordHash();
}

guchar* ProfileManager::getProfileHash()
{
	return iStorageAgent->getProfileHash();
}

guchar* ProfileManager::getEncryptedProfile()
{
	return iStorageAgent->getEncryptedProfile();
}

guint ProfileManager::getEncryptedProfileSize()
{
	return iStorageAgent->getEncryptedProfileSize();
}

gboolean ProfileManager::setSalt(guchar* _salt)
{
	if(iStorageAgent->isAuthenticated()) return iStorageAgent->setSalt(_salt);
	else return FALSE;
}

gboolean ProfileManager::setPasswordHash(guchar* _password_hash)
{
	if(iStorageAgent->isAuthenticated()) return iStorageAgent->setPasswordHash(_password_hash);
	else return FALSE;
}

gboolean ProfileManager::setEncryptedProfile(guchar* _profile_hash, guchar* _encrypted_profile, guint _e_profile_size)
{
	if(iStorageAgent->isAuthenticated())
	{
		if(iStorageAgent->setEncryptedProfile(_profile_hash, _encrypted_profile, _e_profile_size))
		{
			iSavingRequired = FALSE;
			return TRUE;
		}
		else
		{
			iSavingRequired = TRUE;
			return FALSE;
		}
	}
	else return FALSE;
}

gboolean ProfileManager::isSavingRequired()
{
	return iSavingRequired;
}

gboolean ProfileManager::isAuthenticated()
{
	return iStorageAgent->isAuthenticated();
}

// TODO add checks for this
gboolean ProfileManager::setAuthenticated()
{
	return iStorageAgent->setAuthenticated();
}
	

void log_wrap(char* msg, int info)
{
	cem_add_to_log(msg,info);
	//printf("message: %s (%d)\n",msg,info);
}

/*//////////////////////////////////////////////////////
||||||||||| DO NOT USE THESE DIRECTLY !!!! |||||||||||||
||||||| USE FUNCTIONS IN gems_profile_manager.c  |||||||
////////////////////////////////////////////////////////*/

// New manager object
ProfileManager* profilemanager_new_manager()
{
	return ProfileManager::GetInstance();
}

// Delete manager object
void profilemanager_delete_manager(ProfileManager* _aManager)
{
	delete _aManager;
}

// Login to manager (no auth checking yet)
gboolean profilemanager_login(ProfileManager* _aManager, guint32 _userId)
{
	return _aManager->Login(_userId);
}

// Logout from manager
gboolean profilemanager_logout(ProfileManager* _aManager)
{
	return _aManager->Logout();
}

// View one information
const gchar* profilemanager_view_info(ProfileManager* _aManager, gchar* _parameter)
{
	string param = string(_parameter);
	return (_aManager->ViewDetail(param)).c_str();
}

// Edit own info
gboolean profilemanager_edit_info(ProfileManager* _aManager, gchar* _parameter, gchar* _value)
{
	string param = string(_parameter);
	string value = string(_value);
	return _aManager->EditDetails(param,value);
}

guint32 profilemanager_get_userid(ProfileManager* _aManager)
{
	return _aManager->GetId();
}

// Get own username
const gchar* profilemanager_get_username(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_USERNAME);
	return (_aManager->ViewDetail(param)).c_str();
}

// Get first name
const gchar* profilemanager_get_firstname(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_FIRSTNAME);
	return (_aManager->ViewDetail(param)).c_str();
}

// Get last name
const gchar* profilemanager_get_lastname(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_LASTNAME);
	return (_aManager->ViewDetail(param)).c_str();
}

// Get age
guint16 profilemanager_get_age(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_AGE);
	return atoi((_aManager->ViewDetail(param)).c_str());
}

guint32 profilemanager_get_points(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_POINTS);
	return (guint32)strtoul((_aManager->ViewDetail(param)).c_str(),NULL,10);
}

/* Get avatar id */
guint32 profilemanager_get_avatarid(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_AVATARID);
	return (guint32)strtoul((_aManager->ViewDetail(param)).c_str(),NULL,10);
}

gboolean profilemanager_set_username(ProfileManager* _aManager, gchar* _value)
{
	string param = string(PROF_PARAM_USERNAME);
	string value = string(_value);
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_set_firstname(ProfileManager* _aManager, gchar* _value)
{
	string param = string(PROF_PARAM_FIRSTNAME);
	string value = string(_value);
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_set_lastname(ProfileManager* _aManager, gchar* _value)
{
	string param = string(PROF_PARAM_LASTNAME);
	string value = string(_value);
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_set_age(ProfileManager* _aManager, guint16 _value)
{
	stringstream strvalue;
	string param = string(PROF_PARAM_AGE);
	
	if(_value > 120) return FALSE;
	
	strvalue << _value;
	string value = strvalue.str();
	
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_set_avatarid(ProfileManager* _aManager, guint32 _value)
{
	string param = string(PROF_PARAM_AVATARID);
	
	stringstream strvalue;
	strvalue << _value;
	
	string value = strvalue.str();
	
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_add_points(ProfileManager* _aManager, guint32 _value)
{
	string param = string(PROF_PARAM_POINTS);
	guint32 points = (guint32)strtoul((_aManager->ViewDetail(param)).c_str(),NULL,10);
	
	points += _value;
	
	stringstream strvalue;
	strvalue << points;
	
	string value = strvalue.str();
	
	return _aManager->EditDetails(param,value);
}

gboolean profilemanager_remove_points(ProfileManager* _aManager, guint32 _value)
{
	string param = string(PROF_PARAM_POINTS);
	guint32 points = (guint32)strtoul((_aManager->ViewDetail(param)).c_str(),NULL,10);
	
	if((points - _value) < 0) points = 0;
	else points = points - _value;
	
	stringstream strvalue;
	strvalue << points;
	
	string value = strvalue.str();
	
	return _aManager->EditDetails(param,value);	
}

void profilemanager_reset_points(ProfileManager* _aManager)
{
	string param = string(PROF_PARAM_POINTS);
	string value = "0";
	_aManager->EditDetails(param,value);
}

gboolean profilemanager_storage_load_profile(ProfileManager* _aManager, gchar* _username)
{
	return _aManager->loadEncryptedProfile(_username);
}

gboolean profilemanager_storage_save_profile(ProfileManager* _aManager)
{
	return _aManager->saveEncryptedProfile();
}

gboolean profilemanager_storage_change_profile(ProfileManager* _aManager, gchar* _username)
{
	return _aManager->changeLoadedUser(_username);
}

guchar* profilemanager_storage_get_password_salt(ProfileManager* _aManager)
{
	if(_aManager == NULL) return NULL;
	return _aManager->getSalt();
}

guchar* profilemanager_storage_get_password_hash(ProfileManager* _aManager)
{
	if(_aManager == NULL) return NULL;
	return _aManager->getPasswordHash();
}

guchar* profilemanager_storage_get_profile_hash(ProfileManager* _aManager)
{
	if(_aManager == NULL) return NULL;
	return _aManager->getProfileHash();
}

guchar* profilemanager_storage_get_encrypted_profile(ProfileManager* _aManager)
{
	if(_aManager == NULL) return NULL;
	return _aManager->getEncryptedProfile();
}

guint profilemanager_storage_get_encrypted_profile_size(ProfileManager* _aManager)
{
	if(_aManager == NULL) return 0;
	return _aManager->getEncryptedProfileSize();
}

gboolean profilemanager_storage_set_password_salt(ProfileManager* _aManager, guchar* _salt)
{
	return _aManager->setSalt(_salt);
}

gboolean profilemanager_storage_set_password_hash(ProfileManager* _aManager, guchar* _hash)
{
	return _aManager->setPasswordHash(_hash);
}

gboolean profilemanager_storage_set_encrypted_profile(ProfileManager* _aManager, guchar* _profile_hash, guchar* _encrypted_profile, guint _e_profile_size)
{
	return _aManager->setEncryptedProfile(_profile_hash, _encrypted_profile, _e_profile_size);
}

gboolean profilemanager_storage_deserialize_profile(ProfileManager* _aManager, guchar* _profile)
{
	return _aManager->deserializeProfile((gchar*)_profile);
}

gchar* profilemanager_storage_serialize_profile(ProfileManager* _aManager)
{
	return _aManager->serializeProfile();
}

gboolean profilemanager_set_authenticated(ProfileManager* _aManager)
{
	return _aManager->setAuthenticated();
}

gboolean profilemanager_is_authenticated(ProfileManager* _aManager)
{
	return _aManager->isAuthenticated();
}
gboolean profilemanager_is_saving_required(ProfileManager* _aManager)
{
	return _aManager->isSavingRequired();
}

