/*
 * gems_profile_manager.c
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu, Lappeenranta University of Technology
 *
 * Authors: Jussi Laakkonen <jussi.laakkonen@lut.fi>
 */

#include "gems_profile_manager.h"
#include "gems_security.h"
#include "gems.h"
#include "../cem/cem.h"

#include "gems_teacher_server_utils.h"

// Login
gboolean gems_profile_manager_login(guint32 userId, gchar* username, gchar* passworddata)
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return FALSE;
	
	return profilemanager_login(pm,userId);
}

// Login with pen gesture
gboolean gems_profile_manager_login_pen_gesture(guint32 userId, gchar* username, gchar* passworddata)
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return FALSE;
	
	return profilemanager_login(pm,userId);
}

//Set password
gboolean gems_profile_manager_set_password(guint32 userId, gchar* username, gchar* passworddata)
{

	return TRUE;
}

//Set pen gesture password
gboolean gems_profile_manager_set_password_pen_gesture(guint32 userId, gchar* username, gchar* passworddata)
{

	return TRUE;
}



gint gems_profile_manager_authenticate_default_user(gchar* passworddata)
{
	return gems_profile_manager_authenticate_user("defaultuser",passworddata);
}

gint gems_profile_manager_authenticate_user(gchar* username, gchar* passworddata)
{
	guint passlength = 0;
	guint proflength = 0;
	guint decprofhashlength = 0;
	guchar* passwd = NULL;
	guchar* pwhash = NULL;
	guchar* decrypted = NULL;
	guchar* decprofhash = NULL;
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return FALSE;
	
	if(username == NULL || passworddata == NULL)
	{
		cem_add_to_log("Authentication failure, username or password not given", J_LOG_ERROR);
		return LOGIN_INVALID_PARAMS;
	}
	
	// Try to load profile
	if(!profilemanager_storage_load_profile(pm, username))
	{
		cem_add_to_log("Authentication failure, profile not found.", J_LOG_ERROR);
		return LOGIN_NO_PROFILE;
	}
	else cem_add_to_log("Profile loaded", J_LOG_DEBUG);
	
	// Create password from data
	if((passwd = gems_security_create_password(passworddata,profilemanager_storage_get_password_salt(pm))) == NULL)
	{
		cem_add_to_log("Authentication failure, given password or salt is NULL.", J_LOG_ERROR);
		return LOGIN_CANNOT_INIT_PASSWD;
	}
	
	// Initialize security
	if(!gems_security_init_security(passwd,profilemanager_storage_get_password_salt(pm)))
	{
		cem_add_to_log("Authentication methods not available, couldn't initialize security.", J_LOG_ERROR);
		if(passwd != NULL) g_free(passwd);
		gems_security_clear_security();
		return LOGIN_CANNOT_INIT_SECURITY;
	}
	else cem_add_to_log("Security contexts initialized", J_LOG_DEBUG);

	// hash password
	passlength = PASSLEN;
	pwhash = gems_security_calculate_hash(passwd,&passlength);
	if(passlength != SHA_HASH_LENGTH) cem_add_to_log("Authentication might fail, password hash length is not 256 bits (32 bytes)!", J_LOG_ERROR);

	// verify passwords
	if(!gems_security_verify_hash(pwhash,profilemanager_storage_get_password_hash(pm)))
	{
		cem_add_to_log("Authentication failure: invalid password", J_LOG_INFO);
		if(passwd != NULL) g_free(passwd);
		if(pwhash != NULL) g_free(pwhash);
		gems_security_clear_security();
		return LOGIN_INVALID_PASSWORD;
	}
	else cem_add_to_log("Password ok", J_LOG_DEBUG);
	
	// Decrypt profile
	proflength = profilemanager_storage_get_encrypted_profile_size(pm);
	
	decrypted = gems_security_decrypt_data(profilemanager_storage_get_encrypted_profile(pm), &proflength);
	
	// Calculate hash from profile
	decprofhashlength = proflength;
	decprofhash = gems_security_calculate_hash(decrypted,&decprofhashlength);
	if(decprofhashlength != SHA_HASH_LENGTH) cem_add_to_log("Authentication might fail, profile hash length is not 256 bits (32 bytes)!", J_LOG_ERROR);
	
	// verify profile hashes
	if(!gems_security_verify_hash(decprofhash,profilemanager_storage_get_profile_hash(pm)))
	{
		cem_add_to_log("Authentication failure: decrypted profile content differs from stored!", J_LOG_ERROR);
		if(passwd != NULL) g_free(passwd);
		if(pwhash != NULL) g_free(pwhash);
		if(decrypted != NULL) g_free(decrypted);
		if(decprofhash != NULL) g_free(decprofhash);
		gems_security_clear_security();
		return LOGIN_PROFILE_HASH_FAIL;
	}
	else cem_add_to_log("Profile verified!", J_LOG_DEBUG);
	
	// parse profile data to manager
	if(!profilemanager_storage_deserialize_profile(pm,decrypted))
	{
		cem_add_to_log("Unable to set profile data, data is invalid.",J_LOG_ERROR);
		if(passwd != NULL) g_free(passwd);
		if(pwhash != NULL) g_free(pwhash);
		if(decrypted != NULL) g_free(decrypted);
		if(decprofhash != NULL) g_free(decprofhash);
		gems_security_clear_security();
		return LOGIN_PROFILE_CORRUPT;
	}
	
	// Set to authenticated
	profilemanager_set_authenticated(pm);
	
	if(passwd != NULL) g_free(passwd);
	if(pwhash != NULL) g_free(pwhash);
	if(decrypted != NULL) g_free(decrypted);
	if(decprofhash != NULL) g_free(decprofhash);
	
	cem_add_to_log("Authentication success!", J_LOG_DEBUG);
	
	return LOGIN_OK;
}

gboolean gems_profile_manager_change_password(gchar* passworddata)
{
	gboolean rval = TRUE;
	ProfileManager* pm = gems_get_profile_manager();
	
	// Do not allow password change if not authenticated
	if(profilemanager_is_authenticated(pm))
	{
		// Create salt and password
		guchar* salt = gems_security_create_password_salt();
		guchar* passwd = gems_security_create_password(passworddata,salt);
		gems_security_change_password(passwd,salt);
		
		// Hash password
		guint hashlen = PASSLEN;
		guchar* passhash = gems_security_calculate_hash(passwd,&hashlen);
		if(hashlen != SHA_HASH_LENGTH) cem_add_to_log("Next authentication might fail, password hash length is not 256 bits (32 bytes)!", J_LOG_ERROR);
		
		// Set new salt and password
		if(profilemanager_storage_set_password_salt(pm,salt) && profilemanager_storage_set_password_hash(pm,passhash))
		{
			cem_add_to_log("Password change: success.", J_LOG_DEBUG);
		
			// Serialize profile
			guchar* profile = (guchar*)profilemanager_storage_serialize_profile(pm);
			guint profilelength = strlen((gchar*)profile)+1;
			
			// Hash profile
			guint hashlength = profilelength;
			guchar* profilehash = gems_security_calculate_hash(profile,&hashlength);
			if(hashlength != SHA_HASH_LENGTH) cem_add_to_log("Next authentication might fail, profile hash length is not 256 bits (32 bytes)!", J_LOG_ERROR);
			
			// Encrypt profile
			guchar* encrypted = gems_security_encrypt_data(profile,&profilelength);
			guint enc_length = profilelength;
			
			// Decrypt profile
			guchar* decrypted = gems_security_decrypt_data(encrypted,&enc_length);
			hashlength = enc_length;
			guchar* decryptedhash = gems_security_calculate_hash(decrypted,&hashlength);
			if(hashlength != SHA_HASH_LENGTH) cem_add_to_log("Next uthentication might fail, decrypted profile hash length is not 256 bits (32 bytes)!", J_LOG_ERROR);
		
			// Verify that the encryption and decryption works
			if(gems_security_verify_hash(profilehash,decryptedhash))
			{
				// Set profile (encrypted with new password)
				if(profilemanager_storage_set_encrypted_profile(pm,profilehash,encrypted,profilelength))
				{
					// Try to save profile
					if(profilemanager_storage_save_profile(pm))	cem_add_to_log("New encrypted profile saved.", J_LOG_DEBUG);
					else
					{
						rval = FALSE;
						cem_add_to_log("New encrypted profile not saved.", J_LOG_ERROR);
					}
				}
			}
			else
			{
				rval = FALSE;
				cem_add_to_log("Errors while encrypting profile with new password!", J_LOG_ERROR);
			}
			
			if(profile != NULL) g_free(profile);
			if(profilehash != NULL) g_free(profilehash);
			if(encrypted != NULL) g_free(encrypted);
			if(decrypted != NULL) g_free(decrypted);
			if(decryptedhash != NULL) g_free(decryptedhash);
		}
		else
		{
			rval = FALSE;
			cem_add_to_log("Password not changed.",J_LOG_ERROR);
		}
		
		if(salt != NULL) g_free(salt);
		if(passwd != NULL) g_free(passwd);
		if(passhash != NULL) g_free(passhash);
	}
	// Not authenticated
	else
	{
		rval = FALSE;
		cem_add_to_log("User not authenticated, password will not be changed!", J_LOG_DEBUG);
	}
	
	return rval;
}

gboolean gems_profile_manager_create_default_user_profile(gchar* passworddata, guint16 age)
{
	gboolean rval = TRUE;
	GRand* grand = g_rand_new_with_seed(time(NULL));
	
	// Create profile with default informationm, except id (random), password (given) and age (given)
	jammo_profile_container* profile = gems_teacher_server_create_profile(g_rand_int(grand),"defaultuser",passworddata,"none","FirstName","LastName",age,0);

	if(profile)
	{
		if(gems_teacher_server_write_profile(profile) != WRITE_OK)
		{
			cem_add_to_log("Cannot create profile for user \"defaultuser\"", J_LOG_DEBUG);
			rval = FALSE;
		}
		else cem_add_to_log("Created profile for user \"defaultuser\"", J_LOG_DEBUG);
		gems_teacher_server_clear_profile_container(profile);
	}
	else rval = FALSE;
	
	g_free(grand);
	return rval;
}

// Logout
gboolean gems_profile_manager_logout()
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return FALSE;
	
	/* TODO add checks for each call (for NULL's) */
	 
	// Is it necessary to save profile
	if(profilemanager_is_authenticated(pm))
	{
		if(profilemanager_is_saving_required(pm))
		{
			// Get all profile information to one line
			guchar* profile = (guchar*)profilemanager_storage_serialize_profile(pm);
			guint profilelength = strlen((gchar*)profile)+1;
			guint hashlength = profilelength;
		
			// Calculate hash
			guchar* profilehash = gems_security_calculate_hash(profile,&hashlength);
			if(hashlength != SHA_HASH_LENGTH) cem_add_to_log("Next authentication might fail, profile hash length is not 256 bits (32 bytes)!", J_LOG_ERROR);
		
			// Encrypt profile
			guchar* encrypted = gems_security_encrypt_data(profile,&profilelength);
		
			guint enc_length = profilelength;
		
			// Decrypt the encrypted
			guchar* decrypted = gems_security_decrypt_data(encrypted,&enc_length);
			hashlength = enc_length;
			guchar* decryptedhash = gems_security_calculate_hash(decrypted,&hashlength);
			if(hashlength != SHA_HASH_LENGTH) cem_add_to_log("Next authentication might fail, decrypted profile hash length is not 256 bits (32 bytes)!", J_LOG_ERROR);
		
			// Verify that the profile can be decrypted with same password
			if(gems_security_verify_hash(profilehash,decryptedhash))
			{
				// Save the profile to file
				if(profilemanager_storage_set_encrypted_profile(pm,profilehash,encrypted,profilelength))
				{
					if(profilemanager_storage_save_profile(pm))
					{
						cem_add_to_log("Profile saved.", J_LOG_DEBUG);
					}
					else
					{
						cem_add_to_log("Profile not saved.", J_LOG_DEBUG);
					}
				}
			}
			else cem_add_to_log("Problems with encrypting new profile, hashes do not match. Profile not saved",J_LOG_ERROR);
		
			if(profile != NULL) g_free(profile);
			if(profilehash != NULL) g_free(profilehash);
			if(encrypted != NULL) g_free(encrypted);
			if(decrypted != NULL) g_free(decrypted);
			if(decryptedhash != NULL) g_free(decryptedhash);
		}
		else cem_add_to_log("No changes in profile, profile not saved.", J_LOG_DEBUG);
	}
	else cem_add_to_log("Not authenticated, nothing done for logout.", J_LOG_DEBUG);
	
	gems_security_clear_security();
	
	return profilemanager_logout(pm);
}

/* PROFILE MANAGER API*/

/* Get userid */
guint32 gems_profile_manager_get_userid(gems_peer_profile* profile)
{
	if(profile == NULL)
	{
		ProfileManager* pm = gems_get_data()->pm;
		if(pm == NULL) return 0;
		return profilemanager_get_userid(pm);
	}
	else return profile->id;
}

/* Get own username */
const gchar* gems_profile_manager_get_username(gems_peer_profile* profile)
{
	if(profile == NULL)
	{
		ProfileManager* pm = gems_get_data()->pm;
		if(pm == NULL) return NULL;
		return profilemanager_get_username(pm);
	}
	else return profile->username;
}

/* Get firstname */
const gchar* gems_profile_manager_get_firstname()
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return NULL;
	return profilemanager_get_firstname(pm);
}

/* Get lastname */
const gchar* gems_profile_manager_get_lastname()
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return NULL;
	return profilemanager_get_lastname(pm);
}

/* Get age */
guint16 gems_profile_manager_get_age(gems_peer_profile* profile)
{
	if(profile == NULL)
	{
		ProfileManager* pm = gems_get_data()->pm;
		if(pm == NULL) return 0;
		return profilemanager_get_age(pm);
	}
	else return profile->age;
}

guint32 gems_profile_manager_get_avatar_id(gems_peer_profile* profile)
{
	if(profile == NULL)
	{
		ProfileManager* pm = gems_get_data()->pm;
		if(pm == NULL) return 0;
		return profilemanager_get_avatarid(pm);
	}
	else return profile->avatarid;
}

gboolean gems_profile_manager_set_avatar_id(guint32 avatarid)
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return FALSE;
	return profilemanager_set_avatarid(pm,avatarid);
}

/* Get points */
guint32 gems_profile_manager_get_points()
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return 0;
	return profilemanager_get_points(pm);
}

/* Add points */
gboolean gems_profile_manager_add_points(guint32 _value)
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return FALSE;
	return profilemanager_add_points(pm, _value);
}

/* Remove points */
gboolean gems_profile_manager_remove_points(guint32 _value)
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return FALSE;
	return profilemanager_remove_points(pm, _value);
}

/* Reset points (set to zero) */
void gems_profile_manager_reset_points()
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm != NULL) profilemanager_reset_points(pm);
}

gems_peer_profile* gems_profile_get_profile_of_user(guint32 id)
{
	if(gems_get_data()->communication == NULL) return NULL; // Communication not initialized
	gems_connection* element = gems_communication_get_connection_with_userid(gems_get_data()->communication->connections,id);
	
	if(element == NULL) return NULL;
	else return element->profile; // If not set it is still NULL
}

gboolean gems_profile_manager_is_authenticated()
{
	ProfileManager* pm = gems_get_data()->pm;
	if(pm == NULL) return FALSE;
	
	return profilemanager_is_authenticated(pm);
}
