/**sequencer_loop.c is part of JamMo.
 License: GPLv2, read more from COPYING

 This file is for clutter based gui.
 This is part of the sequencer.
 */
#include <glib-object.h>
#include <clutter/clutter.h>
#include <string.h>
#include <stdlib.h>
#include <tangle.h>

#include "sequencer_loop.h"
#include "sequencer.h"
#include "../jammo-sample-button.h"
#include "../jammo.h" //get_id

#include "../../cem/cem.h"

#include "../../configure.h"
//Static pointers for loop rolls
static ClutterActor* rhytmical_sample_looper;
static ClutterActor* melodical_sample_looper;
static ClutterActor* harmonical_sample_looper;
static ClutterActor* effect_sample_looper;


typedef struct {
	uint r;
	uint m;
	uint h;
	uint e;
} quad;

static int number_of_cannot_fail_combos = 11; //TODO: calculate dynamically
static quad cannot_fail_lists[] = {
	{ 1,0,0,-1    },
	{ 0,1,1,0     },
	{ 2,2,2,-1    },
	{ 3,3,3,-1    },
	{ 6,8,-1,1    },
	{ 7,4,4,-1    },
	{ 3,5,5,2     },
	{ 5,7,6,-1    },
	{ 8,8,7,-1    },
	{ 4,6,9,4     },
	{ 9,9,8,-1    }
};

gboolean sequencer_lock_pressed (TangleButton* tanglebutton, gpointer data){

	const gchar* name = clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton));
	gchar* message = g_strdup_printf("Wheel_game: Lock '%s' pressed",name);
	cem_add_to_log(message,J_LOG_USER_ACTION);
	g_free(message);

	ClutterActor* scroller = NULL;
	if (strncmp(name,"rhytm",5)==0)
		scroller = rhytmical_sample_looper;
	else if (strncmp(name,"melod",5)==0)
		scroller = melodical_sample_looper;
	else if (strncmp(name,"harmo",5)==0)
		scroller = harmonical_sample_looper;
	else if (strncmp(name,"effec",5)==0)
		scroller = effect_sample_looper;

	else {
		cem_add_to_log("can't detect which lock pressed",J_LOG_WARN);
		return FALSE;
	}

	ClutterAction* scroll_action = tangle_actor_get_action_by_type(scroller,TANGLE_TYPE_SCROLL_ACTION);
	gfloat threshold_x, threshold_y;

	if (scroller)  {
		tangle_scroll_action_get_thresholds(TANGLE_SCROLL_ACTION(scroll_action), &threshold_x,&threshold_y);
		if (threshold_y==10000.0)
			tangle_scroll_action_set_thresholds(TANGLE_SCROLL_ACTION(scroll_action), threshold_x,30.0);//TODO: use original default value
		else
			tangle_scroll_action_set_thresholds(TANGLE_SCROLL_ACTION(scroll_action), threshold_x,10000.0);
	}
	return TRUE;
}


/*
ScrollingActor doesn't have clampping. This will set it.
*/
static void fine_tune_wheel (ClutterActor* sample_looper) {
	if (sample_looper==NULL)
		return;

	ClutterAction* action;
	
	action = tangle_actor_get_action_by_type(sample_looper, TANGLE_TYPE_SCROLL_ACTION);
	g_signal_connect_swapped(action, "clamp-offset-y", G_CALLBACK(tangle_widget_clamp_child_boundaries), sample_looper);
}


#define ABS_F(x) ((x) < 0 ? -(x) : (x))
/*
Returns JammoSample closest to center of wheel. If wheel is rolling and two sample are equal
some of them are returned.
(This function can be used when wheel is running but there are no reason to do that).
Will return NULL in error cases.
Returns new just created sample, not actual sample from wheel.
*/
static JammoSample* get_centermost_sample_from_loop_roll(ClutterActor* wheel) {
	gfloat y;
	gfloat offset=0,not_used; //How much scroller is scrolled
	gfloat height;    //Height of the scroller (we take this from color_box behind samples)

	ClutterAction* scroll_action = tangle_actor_get_action_by_type(wheel,TANGLE_TYPE_SCROLL_ACTION);
	tangle_scroll_action_get_offsets(TANGLE_SCROLL_ACTION(scroll_action), &not_used,&offset);

	//Check height of first sample-button, suppose all have same height
	ClutterActor* first_actor = tangle_widget_get_nth_child(TANGLE_WIDGET(wheel),0);
	//If first is NULL, suppose they are all NULL.
	if (first_actor==NULL)
		return NULL;
	offset+= clutter_actor_get_height(first_actor);

	//Check height of the wheel
	height = clutter_actor_get_height(wheel);
	//printf("height = %f\n",height);

	//Check all sample button which one is closest to center (='offset')
	float best_value = height;
	JammoSampleButton* closest = NULL;

	GList* children;
	for (children = tangle_widget_get_children_readonly(TANGLE_WIDGET(wheel)); children; children = children->next) {
		if (!JAMMO_IS_SAMPLE_BUTTON(children->data)){
			//printf("this is not sample_button\n");  //Color_box triggers this at least
			continue;
		}

		y = clutter_actor_get_y (children->data);
		if (y==0) y+=height; //actor in first slot should be consider be in after very last (+height)

		if (ABS_F(y-offset) < best_value) {
			best_value = ABS_F(y-offset);
			closest = JAMMO_SAMPLE_BUTTON(children->data);
		}

	}
	if (closest==NULL) return NULL;
	return jammo_sample_new_from_existing(jammo_sample_button_get_sample(closest));
}


/***************************
Functions for json
**************************/


//Removing ad-hoc -sequncer when stopped
static void on_sequencer_stopped(JammoSequencer* sequencer, gpointer data) {
	//printf("stopped\n");
	g_object_unref(sequencer);
}

/*
 * plays all samples from centerline of loop rolls
 */
void fullsequencer_loop_arrow_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	cem_add_to_log("fullsequencer_play_all_loops-clicked",J_LOG_USER_ACTION);

	//get_centermost_sample_from_loop_roll can return NULL (in error cases)
	JammoSample* r = get_centermost_sample_from_loop_roll(rhytmical_sample_looper);
	JammoSample* m = get_centermost_sample_from_loop_roll(melodical_sample_looper);
	JammoSample* h = get_centermost_sample_from_loop_roll(harmonical_sample_looper);
	JammoSample* e = get_centermost_sample_from_loop_roll(effect_sample_looper);

	//Make ad-hoc-sequencer for samples
	JammoSequencer* ad_hoc_sequencer = jammo_sequencer_new();
	jammo_sequencer_set_tempo(ad_hoc_sequencer,jammo_sequencer_get_tempo(JAMMO_SEQUENCER(jammo_get_object_by_id("fullsequencer-the-sequencer"))));
	jammo_sequencer_set_pitch(ad_hoc_sequencer,jammo_sequencer_get_pitch(JAMMO_SEQUENCER(jammo_get_object_by_id("fullsequencer-the-sequencer"))));

	if (r!=NULL){
		JammoEditingTrack* track1 = jammo_editing_track_new();
		jammo_sequencer_add_track(ad_hoc_sequencer,JAMMO_TRACK(track1));
		jammo_editing_track_add_sample(track1,r,0);
	}
	if (m!=NULL) {
		JammoEditingTrack* track2 = jammo_editing_track_new();
		jammo_sequencer_add_track(ad_hoc_sequencer,JAMMO_TRACK(track2));
		jammo_editing_track_add_sample(track2,m,0);
	}
	if (h!=NULL) {
		JammoEditingTrack* track3 = jammo_editing_track_new();
		jammo_sequencer_add_track(ad_hoc_sequencer,JAMMO_TRACK(track3));
		jammo_editing_track_add_sample(track3,h,0);
	}
	if (e!=NULL) {
		JammoEditingTrack* track4 = jammo_editing_track_new();
		jammo_sequencer_add_track(ad_hoc_sequencer,JAMMO_TRACK(track4));
		jammo_editing_track_add_sample(track4,e,0);
	}

	g_signal_connect(ad_hoc_sequencer, "stopped", G_CALLBACK(on_sequencer_stopped), NULL);
	jammo_sequencer_play(ad_hoc_sequencer);
}


/*
amount=how many slots.
If loop is already rolling, this will abort it and will make own.
Loop will still snap to grid always.
*/
static void rotate_loop_for_amount(ClutterActor* wheel, gint amount) {
	ClutterAction* scroll_action = tangle_actor_get_action_by_type(wheel,TANGLE_TYPE_SCROLL_ACTION);

	//Check if lock is pressed
	gfloat not_used,threshold_y;
	tangle_scroll_action_get_thresholds(TANGLE_SCROLL_ACTION(scroll_action), &not_used,&threshold_y);

	//printf("threshold_y '%f'\n",threshold_y);
	if (threshold_y==10000.0) //means rolling is disabled for this wheel
		return;
	gfloat offset;
	tangle_scroll_action_get_offsets(TANGLE_SCROLL_ACTION(scroll_action), &not_used,&offset);
	//printf("offset '%f'\n",offset);

	int current_slot = (int)(offset)/92; //round to lower slot, if rolling already
	gfloat target = (current_slot+amount) * 92.0;
	//printf("current_slot = %d, target pixel='%f'\n",current_slot,target);
	tangle_object_animate(G_OBJECT(scroll_action),  CLUTTER_EASE_IN_OUT_QUAD, 550, "offset-y", target, NULL);
}


/*
Will rotate wheel so target_slot is center of wheel.
slot means index on list.
-1 means 'do not rotate this at all'
*/
static void rotate_to_place(ClutterActor* wheel, gint target_slot) {
	if (target_slot==-1)
		return;
	ClutterAction* scroll_action = tangle_actor_get_action_by_type(wheel,TANGLE_TYPE_SCROLL_ACTION);

	//printf("current_slot = %d, target pixel='%f'\n",current_slot,target);
	tangle_object_animate(G_OBJECT(scroll_action),  CLUTTER_EASE_IN_OUT_QUAD, 750, "offset-y", target_slot*92.0, NULL);
}


/*
cannot_fail_lists is index for corresponding lists.
*/
void fullsequencer_loop_cannot_fail_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	cem_add_to_log("fullsequencer_loop_cannot_fail_clicked-clicked",J_LOG_USER_ACTION);

	GRand *random_gen = g_rand_new();

	gint index_for_combo = g_rand_int_range(random_gen, 0, number_of_cannot_fail_combos-1);
	quad combo = cannot_fail_lists[index_for_combo];
	printf("CannotFailCombo[%d]: %d,%d,%d,%d\n", index_for_combo,combo.r, combo.m, combo.h, combo.e);

	rotate_to_place(rhytmical_sample_looper, combo.r);
	rotate_to_place(melodical_sample_looper, combo.m);
	rotate_to_place(harmonical_sample_looper, combo.h);
	rotate_to_place(effect_sample_looper, combo.e);
}


//There will be some logic
void fullsequencer_loop_crazy_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	cem_add_to_log("fullsequencer_loop_crazy_clicked-clicked",J_LOG_USER_ACTION);
	GRand *random_gen = g_rand_new();

	gint rot_rhyt = g_rand_int_range(random_gen, -30, 30);
	gint rot_melo = g_rand_int_range(random_gen, -30, 30);
	gint rot_harm = g_rand_int_range(random_gen, -30, 30);
	gint rot_effe = g_rand_int_range(random_gen, -30, 30);

	rotate_loop_for_amount(rhytmical_sample_looper, rot_rhyt);
	rotate_loop_for_amount(melodical_sample_looper, rot_melo);
	rotate_loop_for_amount(harmonical_sample_looper, rot_harm);
	rotate_loop_for_amount(effect_sample_looper, rot_effe);

	g_rand_free(random_gen);
}


//////////////////////////////////////


/*
DEPRECATED
Bottom of the loop-view there are dropping-area, and when samplebutton is dragged here, view is changed.
*/
gboolean sequencer_drag_begin(TangleDropAction* drop_action, TangleDragAction* drag_action){
	cem_add_to_log("jammo_sample_button dragged from wheel to bottom of the screen",J_LOG_USER_ACTION);
	sequencer_change_to_sequencer_view(NULL,NULL);
	
	return FALSE;
}

static void sample_button_drag_begin(JammoSampleButton* sample_button, gpointer data) {
	cem_add_to_log("jammo_sample_button from wheel drag-begin",J_LOG_USER_ACTION);
	sequencer_change_to_sequencer_view(NULL,NULL);
}

static int length_of_rhythms=0;
static int length_of_melodics=0;
static int length_of_harmonics=0;
static int length_of_effects=0;

typedef struct {
	const gchar* image;
	const gchar* sound;
} samplebutton_tuple;

static samplebutton_tuple rhythms[] = {
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_polka01_$t_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop07_$t_44_2.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_8beat02_$t_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/darabouka.png","wheel_game/Rh_darabouka03_$t_44_1.ogg"},  //Keep index. used for cannot-fail
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_8beat01_$t_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_rockbeat01_$t_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_dancebeat01_$t_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_disco01_$t_44_1.ogg"},  //Keep index. used for cannot-fail
	{"wheel_game/cahon.png","wheel_game/Rh_cajon06_$t_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_elbeat02_$t_44_2.ogg"},  //Keep index. used for cannot-fail
	{"wheel_game/darabouka.png","wheel_game/Rh_conga03_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_16beat01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_16beat02_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_8beat03_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_8beat04_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_bajiaogu01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_balladbeat01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_bossanova01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_claps01_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_claps02_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_claves01_$t_44_2.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_conga01_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_conga02_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_conga04_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_conga05_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_cowbell01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_crashbeat01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_drbeat01_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_drbeat02_$t_34_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_elbeat03_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_elbeat04_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_elbeat05_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_elbeat06_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_FolkMetalDrums1_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_FolkMetalDrums2_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_FolkMetalDrums3_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_FolkMetalDrums4_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_ghatom01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_ghatom02_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_gourd01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_GrimeDrums1_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_GrimeDrums2_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hihat01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_HiLifeCowBell_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_HiLifeDrums1_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_HiLifeDrums2_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_HiLifeDrums3_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop02_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop03_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop04_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop05_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop06_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop08_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop09_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_hiphop10_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_HipHopBalladDrumFill_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_HipHopBalladDrums_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_kanjara01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_kendong01_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_LatinDrums1_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_LatinDrums2_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_LatinDrums3_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_LatinDrums4_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_LatinPerc1_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_LatinTamb1_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_piatti01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_ridebeat01_$t_44_1.ogg"},
	{"wheel_game/shaker.png","wheel_game/Rh_shaker01_$t_44_1.ogg"},
	{"wheel_game/shaker.png","wheel_game/Rh_shaker02_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_shuffle01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_snare01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_snarefill02_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_snarefill_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_snareroll03_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_SocaDrumFills_$t_44_2.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_SocaDrums_$t_44_2.ogg"},
	{"wheel_game/tambourine.png","wheel_game/Rh_tambourine01_$t_44_1.ogg"},
	{"wheel_game/tambourine.png","wheel_game/Rh_tambourine02_$t_44_1.ogg"},
	{"wheel_game/tambourine.png","wheel_game/Rh_tambourine03_$t_34_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_tango01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_tango02_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_templeblock01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_tombeat01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_tombeat02_$t_34_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_tomfill01_$t_34_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_turkish01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_turkish02_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_twist01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_waltz01_$t_34_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_woodblock01_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_BalladTabla_$t_44_2.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_BalladTablaFill_$t_44_2.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon01_$t_44_1.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon02_$t_44_1.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon03_$t_44_1.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon04_$t_44_2.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon07_$t_44_1.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon08_$t_44_1.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon09_$t_34_1.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon10_$t_34_1.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon11_$t_44_1.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon12_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_darabouka01_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_darabouka02_$t_34_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_darabouka04_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_darabouka05_$t_34_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe01_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe02_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe03_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe04_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe05_$t_34_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe06_$t_34_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe07_$t_34_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe08_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe09_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe10_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe11_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_FunkyTabla_$t_44_2.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_tabla01_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_tabla02_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_tabla03_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_tabla04_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_tabla05_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_tabla06_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_tabla07_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_tabla08_$t_44_1.ogg"},
	{"wheel_game/tabla.png","wheel_game/Rh_tabla09_$t_44_1.ogg"},
	{NULL,NULL}
};


static samplebutton_tuple melodics[] = {
	{"wheel_game/strings.png", "wheel_game/Me_strings02_$t_$p_44_2.ogg"}, //Keep index. used for cannot-fail //TODO: tempo=130 missing!
	{"wheel_game/double_bass.png", "wheel_game/Me_doublebass01_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/cello.png", "wheel_game/Me_cello02_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/flute.png", "wheel_game/Me_flute05_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/organ.png", "wheel_game/Me_organ03_$t_$p_44_2.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/double_bass.png", "wheel_game/Me_ebass04_$t_$p_44_4.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/double_bass.png", "wheel_game/Me_ebass20_$t_$p_44_1.ogg"},//Keep index. used for cannot-fail
	{"wheel_game/double_bass.png", "wheel_game/Me_ebass08_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/saxophone.png", "wheel_game/Me_sax10_$t_$p_44_1.ogg"},//Keep index. used for cannot-fail
	{"wheel_game/synthesizer.png", "wheel_game/Me_delaysynth02_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthguitar02_$t_$p_44_2.ogg"},
	{"wheel_game/clavinet.png", "wheel_game/Me_clavinet01_$t_$p_44_1.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Me_FolkMetalAccordion1_$t_$p_44_2.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Me_FolkMetalAccordion2_$t_$p_44_1.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Me_FolkMetalAccordion3_$t_$p_44_2.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Me_FolkMetalAccordion4_$t_$p_44_2.ogg"},
	{"wheel_game/brass.png", "wheel_game/Me_SocaBrassRiff_$t_$p_44_2.ogg"},
	{"wheel_game/cello.png", "wheel_game/Me_cello01_$t_$p_44_2.ogg"},
	{"wheel_game/cello.png", "wheel_game/Me_cello05_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_doublebass02_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_doublebass03_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_doublebass04_$t_$p_44_2.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp01_$t_$p_34_2.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp02_$t_$p_44_2.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp03_$t_$p_44_2.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp04_$t_$p_34_2.ogg"},
	{"wheel_game/harp.png", "wheel_game/Me_harp05_$t_$p_34_2.ogg"},
	{"wheel_game/mandolin.png", "wheel_game/Me_mandolin02_$t_$p_34_4.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax01_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax02_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax03_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax04_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax05_$t_$p_34_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax06_$t_$p_34_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax07_$t_$p_44_1.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax08_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax09_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax11_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax12_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_ScarboroughFairSax_$t_$p_44_8.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_SiyaHambaSax1_$t_$p_44_4.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe01_$t_$p_44_2.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe02_$t_$p_44_1.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe03_$t_$p_44_2.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe04_$t_$p_44_1.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe05_$t_$p_34_2.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe06_$t_$p_34_1.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe07_$t_$p_34_2.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe08_$t_$p_34_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_LatinOrgan1_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_organ04_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_organ05_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_organ06_$t_$p_44_2.ogg"},
	{"wheel_game/vibraphone.png", "wheel_game/Me_vibraphone01_$t_$p_34_2.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone01_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone02_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone03_$t_$p_44_2.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone04_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone05_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone06_$t_$p_44_2.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone07_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone08_$t_$p_44_1.ogg"},
	{"wheel_game/trombone.png", "wheel_game/Me_trombone09_$t_$p_34_1.ogg"},
	{"wheel_game/glockenspiel.png", "wheel_game/Me_glockenspiel01_$t_$p_44_2.ogg"},
	{"wheel_game/electric_piano.png", "wheel_game/Me_rhodes01_$t_$p_44_1.ogg"}, //TODO: icon?
	{"wheel_game/strings.png", "wheel_game/Me_ScarboroughFairStrings_$t_$p_44_2.ogg"},
	{"wheel_game/strings.png", "wheel_game/Me_strings03_$t_$p_44_2.ogg"},
	{"wheel_game/strings.png", "wheel_game/Me_strings05_$t_$p_44_2.ogg"},
	{"wheel_game/strings.png", "wheel_game/Me_strings07_$t_$p_44_4.ogg"},
	{"wheel_game/strings.png", "wheel_game/Me_strings09_$t_$p_44_1.ogg"},
	{"wheel_game/strings.png", "wheel_game/Me_strings10_$t_$p_44_1.ogg"},
	{"wheel_game/piano.png", "wheel_game/Me_piano13_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Me_piano18_$t_$p_44_1.ogg"},
	{"wheel_game/piano.png", "wheel_game/Me_piano19_$t_$p_44_1.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute01_$t_$p_44_2.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute02_$t_$p_44_1.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute03_$t_$p_44_1.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute04_$t_$p_44_1.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute06_$t_$p_44_1.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute07_$t_$p_34_2.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute08_$t_$p_34_2.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute09_$t_$p_34_2.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute10_$t_$p_44_1.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute11_$t_$p_44_1.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute12_$t_$p_44_2.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute13_$t_$p_44_2.ogg"},
	{"wheel_game/flute.png", "wheel_game/Me_flute14_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass01_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass02_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass03_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass05_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass06_$t_$p_44_4.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass07_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass09_$t_$p_44_1.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass10_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass11_$t_$p_44_1.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass12_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass13_$t_$p_44_1.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass14_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass15_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass16_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass17_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass18_$t_$p_44_1.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass19_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass21_$t_$p_44_1.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass22_$t_$p_44_1.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass23_$t_$p_44_1.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass24_$t_$p_44_2.ogg"},
	{"wheel_game/electric_bass.png", "wheel_game/Me_ebass25_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_FolkMetalBass1_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_FolkMetalBass2_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_HiLifeBass1_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_HiLifeBass2_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_HiLifeBass3_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_HiLifeBass4_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_LatinBass1_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_LatinBass2_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_LatinBass3_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Me_eguitar05_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Me_eguitar06_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Me_HiLifeGuitar1_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Me_HiLifeGuitar2_$t_$p_44_1.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Me_HiLifeGuitar3_$t_$p_44_1.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Me_HiLifeGuitar4_$t_$p_44_1.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Me_LatinGuitar1_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Me_LatinGuitar2_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_GrimeSynthBass1_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_GrimeSynthBass2_$t_$p_44_1.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_HipHopBalladBassLine1_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_HipHopBalladBassLine2_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_HipHopBalladBassLine3_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_HipHopBalladBassLine4_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_SocaBassLine1_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_SocaBassLine2_$t_$p_44_2.ogg"},
	{"wheel_game/double_bass.png", "wheel_game/Me_SocaBassLine3_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_delaysynth01_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_hiphopbass01_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_hiphopbass02_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_hiphopbass03_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_hiphopbass04_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_hiphopbass05_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthbass01_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthbass02_$t_$p_44_4.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthbass03_$t_$p_44_4.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthbell01_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthbell02_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthbell03_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthbrass_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthguitar01_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthguitar03_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthlead01_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthlead02_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthriff_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_SynthScale_$t_$p_44_2.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin01_$t_$p_44_2.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin02_$t_$p_44_1.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin03_$t_$p_44_1.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin04_$t_$p_44_1.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin05_$t_$p_44_2.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin06_$t_$p_44_2.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin07_$t_$p_44_1.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin08_$t_$p_44_1.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin09_$t_$p_44_2.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin10_$t_$p_44_2.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin14_$t_$p_34_1.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin15_$t_$p_34_1.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin16_$t_$p_34_2.ogg"},
	{"wheel_game/cello.png", "wheel_game/Me_cello06_$t_$p_44_2.ogg"},
	{"wheel_game/cello.png", "wheel_game/Me_cello07_$t_$p_44_2.ogg"},
	{"wheel_game/cello.png", "wheel_game/Me_cello08_$t_$p_44_1.ogg"},
	{NULL,NULL}
};

static samplebutton_tuple harmonics[] = {
	{"wheel_game/piano.png", "wheel_game/Ha_piano01_$t_$p_44_2.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/accordion.png", "wheel_game/Ha_accordion01_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar01_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/strings.png", "wheel_game/Ha_strings04_$t_$p_44_2.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/organ.png", "wheel_game/Ha_organ01_$t_$p_44_2.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/piano.png", "wheel_game/Ha_piano04_$t_$p_44_4.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/cello.png", "wheel_game/Ha_cello03_$t_$p_44_1.ogg"},  //Keep index. used for cannot-fail
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar11_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/piano.png", "wheel_game/Ha_piano15_$t_$p_44_2.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar03_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/electric_piano.png", "wheel_game/Ha_HiLifeElecPiano_$t_$p_44_2.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Ha_accordion02_$t_$p_44_1.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Ha_SocaAccordionRiff1_$t_$p_44_2.ogg"},
	{"wheel_game/accordion.png", "wheel_game/Ha_SocaAccordionRiff2_$t_$p_44_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar02_$t_$p_44_1.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar04_$t_$p_44_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar05_$t_$p_44_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar08_$t_$p_34_2.ogg"},
	{"wheel_game/banjo.png", "wheel_game/Ha_banjo01_$t_$p_44_1.ogg"},
	{"wheel_game/banjo.png", "wheel_game/Ha_banjo02_$t_$p_44_1.ogg"},
	{"wheel_game/cello.png", "wheel_game/Ha_cello04_$t_$p_44_1.ogg"},
	{"wheel_game/cello.png", "wheel_game/Ha_cello09_$t_$p_44_1.ogg"},
	{"wheel_game/mandolin.png", "wheel_game/Ha_mandolin01_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ02_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ07_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ08_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ09_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ10_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ11_$t_$p_44_2.ogg"},
	{"wheel_game/organ.png", "wheel_game/Ha_organ12_$t_$p_44_2.ogg"},
	{"wheel_game/trumpet.png", "wheel_game/Ha_trumpet01_$t_$p_44_2.ogg"},
	{"wheel_game/strings.png", "wheel_game/Ha_strings01_$t_$p_44_2.ogg"},
	{"wheel_game/strings.png", "wheel_game/Ha_strings06_$t_$p_44_4.ogg"},
	{"wheel_game/strings.png", "wheel_game/Ha_strings08_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_DramaticPiano1_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_DramaticPiano2_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_MontunoPiano1_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_MontunoPiano2_$t_$p_44_1.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano02_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano03_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano05_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano06_$t_$p_44_4.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano07_$t_$p_44.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano08_$t_$p_44_4.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano09_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano10_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano11_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano12_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano14_$t_$p_44_4.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano16_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_piano17_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_SocaPianoRiff1_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_SocaPianoRiff2_$t_$p_44_2.ogg"},
	{"wheel_game/piano.png", "wheel_game/Ha_SocaPianoRiff3_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Ha_eguitar01_$t_$p_44_1.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Ha_eguitar02_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Ha_eguitar03_$t_$p_44_4.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Ha_eguitar04_$t_$p_44_4.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Ha_eguitar07_$t_$p_44_1.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Ha_eguitar08_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Ha_FolkMetalPickedGuitar1_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Ha_FolkMetalPickedGuitar2_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Ha_FolkMetalThrashGuitar1_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Ha_GrimeSynth1_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Ha_GrimeSynth2_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Ha_synth01_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Ha_synth02_$t_$p_44_4.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Ha_synthbrass_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Ha_synthstrings01_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Ha_synthstrings02_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Ha_WashChords1_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Ha_WashChords2_$t_$p_44_2.ogg"},
	{"wheel_game/violin.png","wheel_game/Ha_violin11_$t_$p_44_1.ogg"},
	{"wheel_game/violin.png","wheel_game/Ha_violin12_$t_$p_44_1.ogg"},
	{"wheel_game/violin.png","wheel_game/Ha_violin13_$t_$p_34_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar06_$t_$p_44_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar07_$t_$p_44_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar09_$t_$p_34_2.ogg"},
	{"wheel_game/acoustic_guitar.png", "wheel_game/Ha_acguitar10_$t_$p_34_2.ogg"},
	{NULL,NULL}
};





static samplebutton_tuple effects[] = {
	{"wheel_game/tambourine.png", "wheel_game/Fx_tambourine04_$t_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/soundscape.png", "wheel_game/Fx_swoosh_$t_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/fx.png", "wheel_game/Fx_scratch_$t_44_1.ogg"}, //Keep index. used for cannot-fail //TODO:icon
	{"wheel_game/synthpad.png", "wheel_game/Fx_pulsesynth_$t_$p_44_1.ogg"}, //Keep index. used for cannot-fail
	{"wheel_game/fx.png", "wheel_game/Fx_robot_$t_$p_$p_44_2.ogg"}, //TODO:icon
	{"wheel_game/electric_piano.png", "wheel_game/Fx_synth01_$t_44_1.ogg"},
	{"wheel_game/electric_piano.png", "wheel_game/Fx_synth02_$t_44_1.ogg"},
	{"wheel_game/fx.png", "wheel_game/Fx_antenna_$t_44_2.ogg"},
	{"wheel_game/fx.png", "wheel_game/Fx_electric_$t_44_1.ogg"},
	{"wheel_game/flute.png", "wheel_game/Fx_FluteFill_$t_$p_44_2.ogg"},
	{"wheel_game/fx.png", "wheel_game/Fx_beatbox_$t_44_1.ogg"},       //TODO:icon
	{"wheel_game/fx.png", "wheel_game/Fx_Beeps_$t_$p_44_2.ogg"}, //TODO:icon
	{"wheel_game/fx.png", "wheel_game/Fx_chimes_$t_34_4.ogg"}, //TODO:icon
	{"wheel_game/fx.png", "wheel_game/Fx_glidesynth_$t_$p_44_4.ogg"}, //TODO:icon
	{"wheel_game/fx.png", "wheel_game/Fx_rubber_$t_44_1.ogg"}, //TODO:icon
	{"wheel_game/strings.png", "wheel_game/Fx_BollywoodStrings_$t_$p_44_2.ogg"},
	{"wheel_game/strings.png", "wheel_game/Fx_StringStabs_$t_$p_44_2.ogg"},
	{"wheel_game/electric_guitar.png", "wheel_game/Fx_FolkMetalThrashHarmonics_$t_$p_44_2.ogg"},
	{"wheel_game/darabouka.png", "wheel_game/Fx_Sarangi_$t_$p_44_1.ogg"},
	{"wheel_game/tambourine.png", "wheel_game/Fx_TampuraDrone_$t_$p_44_2.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Fx_SquelchySynth_$t_$p_44_2.ogg"},
	{"wheel_game/banana_vocal.png", "wheel_game/Fx_DayDahLightVocal1_$t_$p_44_2.ogg"},
	{"wheel_game/banana_vocal.png", "wheel_game/Fx_DayDahLightVocal2_$t_$p_44_2.ogg"},
	{"wheel_game/banana_vocal.png", "wheel_game/Fx_DayDahLightVocal3_$t_$p_44_2.ogg"},
	{"wheel_game/labamba_vocal.png", "wheel_game/Fx_LaBambaVocal1_$t_$p_44_4.ogg"},
	{"wheel_game/labamba_vocal.png", "wheel_game/Fx_LaBambaVocal2_$t_$p_44_2.ogg"},
	{"wheel_game/hamba_vocal.png", "wheel_game/Fx_SiyaHambaVocal1_$t_$p_44_4.ogg"},
	{"wheel_game/hamba_vocal.png", "wheel_game/Fx_SiyaHambaVocal2_$t_$p_44_4.ogg"},
	{"wheel_game/hamba_vocal.png", "wheel_game/Fx_SiyaHambaVocal3_$t_$p_44_4.ogg"},
	{"wheel_game/synthpad.png", "wheel_game/Fx_filterpulse_$t_44_2.ogg"},
	{"wheel_game/synthpad.png", "wheel_game/Fx_synthpulse_$t_44_1.ogg"},
	{NULL,NULL}
};




//These tables are subsets of whole samplebuttons and used on level1-orientation game.
static samplebutton_tuple level1_rhythms[] = {
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_8beat01_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_16beat01_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_conga04_$t_44_1.ogg"},
	{"wheel_game/shaker.png","wheel_game/Rh_shaker02_$t_44_1.ogg"},
	{"wheel_game/tambourine.png","wheel_game/Rh_tambourine02_$t_44_1.ogg"},
	{"wheel_game/acoustic_drumkit.png","wheel_game/Rh_tombeat01_$t_44_1.ogg"},
	{"wheel_game/cahon.png","wheel_game/Rh_cajon08_$t_44_1.ogg"},
	{"wheel_game/darabouka.png","wheel_game/Rh_djembe01_$t_44_1.ogg"},
	{NULL,NULL}
};

static samplebutton_tuple level1_melodics[] = {
	{"wheel_game/electric_piano.png", "wheel_game/Me_rhodes01_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthbell01_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthguitar01_$t_$p_44_1.ogg"},
	{"wheel_game/synthesizer.png", "wheel_game/Me_synthriff_$t_$p_44_2.ogg"},
	{"wheel_game/saxophone.png", "wheel_game/Me_sax02_$t_$p_44_1.ogg"},
	{"wheel_game/organ.png", "wheel_game/Me_organ05_$t_$p_44_2.ogg"},
	{"wheel_game/violin.png","wheel_game/Me_violin09_$t_$p_44_2.ogg"},
	{"wheel_game/oboe.png", "wheel_game/Me_oboe04_$t_$p_44_1.ogg"},
	{NULL,NULL}
};


/*
loop_type:
1000=rhythms
2000=melodics
3000=harmonics
4000=effects

returns number of this type of sample_buttons
*/
static int add_sample_buttons(ClutterActor* looper, samplebutton_tuple table[], int loop_type) {
	samplebutton_tuple new_button=table[0];
	int counter=0;
	if (looper==NULL)
		return counter;

	while(new_button.image){
		ClutterActor* sample_button;
		gchar* full_image = tangle_lookup_filename(new_button.image);
		gchar* full_sound = g_strdup_printf("%s", new_button.sound);
		//printf("new_sample: '%s', '%s'\n",full_image,full_sound);
		sample_button = jammo_sample_button_new(); //TODO: jammo_sample_button_new_with_type
		g_object_set(sample_button,"sequencer", jammo_get_object_by_id("fullsequencer-the-sequencer"),NULL);
		//printf("DEBUG: looking image: '%s' and audio: '%s'\n",full_image,full_sound);
		jammo_sample_button_set_image_filename(JAMMO_SAMPLE_BUTTON(sample_button),full_image);
		jammo_sample_button_set_sample_filename(JAMMO_SAMPLE_BUTTON(sample_button), full_sound);

		int loop_id = counter+loop_type;

		g_object_set(sample_button,"drag-threshold-y",10000,NULL); //No dragging by vertical-axis (it is for scrolling wheel)
		g_signal_connect(sample_button,"drag-begin", G_CALLBACK(sample_button_drag_begin),NULL);


		jammo_sample_button_set_loop_id(JAMMO_SAMPLE_BUTTON(sample_button), loop_id);
		g_free(full_image);
		g_free(full_sound);

		tangle_widget_add(TANGLE_WIDGET(looper),sample_button,NULL);
		counter++;
		new_button=table[counter];
	}
	return counter;
}



void sequencer_loop_tune_wheels(int level_number) {
	//Static controls-for-edit1$mute-button
	ClutterActor* container = jammo_get_actor_by_id("fullsequencer_four_wheels");
	if (container==NULL)
		return;
	//It is: jammo_get_actor_by_id("fullsequencer_four_wheels$0"), but can't find!
	rhytmical_sample_looper  = tangle_widget_get_nth_child(TANGLE_WIDGET(container),0);
	melodical_sample_looper  = tangle_widget_get_nth_child(TANGLE_WIDGET(container),1);
	harmonical_sample_looper = tangle_widget_get_nth_child(TANGLE_WIDGET(container),2);
	effect_sample_looper     = tangle_widget_get_nth_child(TANGLE_WIDGET(container),3);

	if (level_number==1) { //
		length_of_rhythms = add_sample_buttons(rhytmical_sample_looper,level1_rhythms,1000);
		length_of_melodics = add_sample_buttons(melodical_sample_looper,level1_melodics,2000);
	}
	else {
		length_of_rhythms = add_sample_buttons(rhytmical_sample_looper,rhythms,1000);
		length_of_melodics = add_sample_buttons(melodical_sample_looper,melodics,2000);
		length_of_harmonics = add_sample_buttons(harmonical_sample_looper,harmonics,3000);
		length_of_effects = add_sample_buttons(effect_sample_looper,effects,4000);
	}

	fine_tune_wheel(rhytmical_sample_looper);
	fine_tune_wheel(melodical_sample_looper);
	fine_tune_wheel(harmonical_sample_looper);
	fine_tune_wheel(effect_sample_looper);
}



/*
Each jammo-sample-button has loop-id.
First digit tells type of loop.

Will return NULL, if anything goes wrong.

*/
ClutterActor* sequencer_loop_give_sample_button_for_this_id(guint id){
	samplebutton_tuple* asked_table;
	int length_of_asked;
	guint id_; //Id without first digit
	//gchar* type;  //TODO: use this
	if (id>=1000 && id <2000){
		asked_table=rhythms;
		length_of_asked=length_of_rhythms;
		id_=id-1000;
	} else if (id>=2000 && id <3000) {
		asked_table=melodics;
		length_of_asked=length_of_melodics;
		id_=id-2000;
	} else if (id>=3000 && id <4000) {
		asked_table=harmonics;
		length_of_asked=length_of_harmonics;
		id_=id-3000;
	}	else if (id>=4000 && id <5000) {
		asked_table=effects;
		length_of_asked=length_of_effects;
		id_=id-4000;
	} else {
		return NULL;
	}

	if (id_>length_of_asked)
		return NULL;

	ClutterActor* sample_button;
	samplebutton_tuple new_button=asked_table[id_];
	gchar* full_image = tangle_lookup_filename(new_button.image);
	gchar* full_sound = g_strdup_printf("%s/%s", DATA_DIR,new_button.sound);
	//printf("new_sample: '%s', '%s'\n",full_image,full_sound);
	sample_button = jammo_sample_button_new(); //TODO: jammo_sample_button_new_with_type
	//g_object_set(sample_button,"sequencer", jammo_get_object_by_id("fullsequencer-the-sequencer"),NULL);
	jammo_sample_button_set_image_filename(JAMMO_SAMPLE_BUTTON(sample_button),full_image);
	jammo_sample_button_set_sample_filename(JAMMO_SAMPLE_BUTTON(sample_button), full_sound);

	jammo_sample_button_set_loop_id(JAMMO_SAMPLE_BUTTON(sample_button), id);
	g_free(full_image);
	g_free(full_sound);

	return sample_button;
}
