#include "settingsdlg.h"
#include "advsettingsdlg.h"
#include "selectremotedlg.h"
#include "aboutdlg.h"

#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QWidget>
#include <QDialog>
#include <QPushButton>
#include <QLabel>
#include <QDebug>
#include <QNetworkConfiguration>
#include <QShowEvent>

SettingsDlg::SettingsDlg(QWidget *parent)
    : QDialog(parent)
    , m_busy(true)
    , m_netConfMan(NULL)
{
    QSettings settings(this);
    m_layout = new QVBoxLayout(this);
    m_btnLayout = new QHBoxLayout();
    m_remoteNameLayout = new QHBoxLayout();
    
    m_advSettingsBtn = new QPushButton(tr("Advanced"), this);
    m_selectRemoteBtn = new QPushButton(tr("Select remote"), this);
    m_aboutBtn = new QPushButton(tr("About"), this);
    m_rateUpBtn = new QPushButton(
            QIcon(settings.value("symbolPath",
                "/usr/share/irwi/symbols/").
                toString() + "symbol_thumbs_up.png"),
            "", this);
    m_rateDownBtn = new QPushButton(
            QIcon(settings.value("symbolPath",
                "/usr/share/irwi/symbols/").
                toString() + "symbol_thumbs_down.png"),
            "", this);
    m_rateUpBtn->setMaximumSize(72, 72);
    m_rateDownBtn->setMaximumSize(72, 72);

    m_btnLayout->addWidget(m_advSettingsBtn);
    m_btnLayout->addWidget(m_selectRemoteBtn);
    m_btnLayout->addWidget(m_aboutBtn);

    m_remoteNameLabel = new QLabel(this);
    m_ratingLabel = new QLabel(this);
    m_remoteNameLayout->addWidget(m_remoteNameLabel);
    m_remoteNameLayout->addWidget(m_ratingLabel);
    m_remoteNameLayout->addWidget(m_rateUpBtn);
    m_remoteNameLayout->addWidget(m_rateDownBtn);

    connect(m_advSettingsBtn, SIGNAL(clicked()),
            this, SLOT(showAdvSettingsDlg()));
    connect(m_selectRemoteBtn, SIGNAL(clicked()),
            this, SLOT(showSelectRemoteDlg()));
    connect(m_aboutBtn, SIGNAL(clicked()),
            this, SLOT(showAboutDlg()));
    connect(m_rateUpBtn, SIGNAL(clicked()),
            this, SLOT(rateUpClicked()));
    connect(m_rateDownBtn, SIGNAL(clicked()),
            this, SLOT(rateDownClicked()));
 
    m_layout->addLayout(m_remoteNameLayout);
    m_layout->addLayout(m_btnLayout);
    this->setLayout(m_layout);

    QString remoteName = settings.value("remoteName", "").toString();
    if (remoteName == "") {
        m_remoteNameLabel->setText(tr("No remote selected"));
    } else {
        // Create remote by name and update it's info if online
        m_remote = Remote(remoteName);
        connect(&m_remote, SIGNAL(infoUpdated()),
                this, SLOT(updateRemoteInfo()));
        m_remoteNameLabel->setText(settings.value("remoteMfg", "").toString()
                + " " + remoteName);
    }
    m_netConfMan = new QTM_PREPEND_NAMESPACE(
            QNetworkConfigurationManager)(this);
    connect(m_netConfMan, SIGNAL(updateCompleted()),
            this, SLOT(onNetworkStatusUpdate()));
    m_netConfMan->updateConfigurations();
    enableRateBtns(false);
}

SettingsDlg::~SettingsDlg()
{
    delete m_advSettingsBtn;
    delete m_selectRemoteBtn;
    delete m_rateUpBtn;
    delete m_rateDownBtn;
    delete m_aboutBtn;
    delete m_remoteNameLabel;
    delete m_ratingLabel;
    delete m_btnLayout;
    delete m_remoteNameLayout;
    delete m_layout;
    delete m_netConfMan;
}

void SettingsDlg::setBusy(bool busy)
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, busy);
#endif
    m_busy = busy;
}

void SettingsDlg::showAdvSettingsDlg()
{
    AdvSettingsDlg dlg;
    dlg.exec();
    if (QSettings(this).value("remoteName", "").toString() == "") {
        m_remoteNameLabel->setText(tr("No remote selected"));
        m_ratingLabel->setText("");
        enableRateBtns(false);
    }
}

void SettingsDlg::showSelectRemoteDlg()
{
    SelectRemoteDlg dlg;
    connect(&dlg, SIGNAL(remoteChanged(Remote)),
            this, SLOT(setRemote(Remote)));
    if (dlg.exec() == QDialog::Rejected) {
        onNetworkStatusUpdate();
    }
}

void SettingsDlg::showAboutDlg()
{
    AboutDlg dlg;
    dlg.exec();
}

void SettingsDlg::setRemote(Remote r)
{
    m_remote = r;
    connect(&m_remote, SIGNAL(infoUpdated()),
            this, SLOT(updateRemoteInfo()));
    updateRemoteInfo();
    enableRateBtns();
}

void SettingsDlg::onNetworkStatusUpdate()
{
    if (m_netConfMan->isOnline() &&
        QSettings(this).value("remoteName", "").toString() != "") {
        setBusy();
        m_remote.updateInfo();
        enableRateBtns();
    } else if (!m_netConfMan->isOnline()) {
        m_ratingLabel->setText(tr("Offline"));
        setBusy(false);
    } else {
        setBusy(false);
    }
}

void SettingsDlg::updateRemoteInfo()
{
    setBusy(false);
    m_remoteNameLabel->setText(m_remote.mfg() + " " + m_remote.name());
    m_ratingLabel->setText(tr("Rating") + ": "
            + QString::number(m_remote.rating()));
}

void SettingsDlg::rateUpClicked()
{
    processRatingSent();
    m_remote.sendRating(Rating::Up);
}

void SettingsDlg::rateDownClicked()
{
    processRatingSent();
    m_remote.sendRating(Rating::Down);
}

void SettingsDlg::processRatingSent()
{
    setBusy();
    enableRateBtns(false);
}

void SettingsDlg::enableRateBtns(bool enable)
{
    m_rateUpBtn->setEnabled(enable);
    m_rateDownBtn->setEnabled(enable);
}

void SettingsDlg::showEvent(QShowEvent *event)
{
    setBusy(m_busy);
    QDialog::showEvent(event);
}



