#include "selectremotedlg.h"

#include "remote.h"
#include "onlinepollerthread.h"
#include "remotetable.h"

#include <QHBoxLayout>
#include <QLabel>
#include <QString>
#include <QListWidget>
#include <QListWidgetItem>
#include <QSettings>
#include <QMap>
#include <QList>
#include <QMessageBox>
#include <QPushButton>

SelectRemoteDlg::SelectRemoteDlg(QWidget *parent)
    : QDialog(parent)
{
    onlinePollerThread = NULL;
    this->setWindowTitle(tr("Select remote"));
    this->setMinimumHeight(320);

    layout = new QHBoxLayout(this);

    alphabetList = new QListWidget(this);
    alphabetList->setMaximumWidth(64);
    layout->addWidget(alphabetList);
    connect(alphabetList,
            SIGNAL(currentItemChanged(QListWidgetItem*, QListWidgetItem*)),
            this,
            SLOT(alphabetItemChanged(QListWidgetItem*, QListWidgetItem*)));
    
    mfgList = new QListWidget(this);
    mfgList->setMaximumWidth(192);
    layout->addWidget(mfgList);
    connect(mfgList,
            SIGNAL(currentItemChanged(QListWidgetItem*, QListWidgetItem*)),
            this,
            SLOT(mfgItemChanged(QListWidgetItem*, QListWidgetItem*)));

    modelList = new RemoteTable(this);
    layout->addWidget(modelList);

    downloadBtn = new QPushButton(tr("Download"), this);
    layout->addWidget(downloadBtn);
    connect(downloadBtn, SIGNAL(clicked()), 
            this, SLOT(downloadRemote()));

    this->setLayout(layout);
    connect(&remoteDBMgr, SIGNAL(dbReady(RemoteDB*)),
            this, SLOT(setDB(RemoteDB*)));
    connect(&remoteDBMgr, SIGNAL(downloadFailed(int)),
            this, SLOT(onDBError(int)));
}

SelectRemoteDlg::~SelectRemoteDlg()
{
    delete layout;
    if (onlinePollerThread != NULL) {
        delete onlinePollerThread;
        onlinePollerThread = NULL;
    }
}

void SelectRemoteDlg::setDB(RemoteDB *db)
{
    remoteDB = db;
    alphabetList->addItems(remoteDB->keys());
    setBusy(false);
}

void SelectRemoteDlg::setBusy(bool busy)
{
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, busy);
#endif
    setEnabled(!busy);
}

void SelectRemoteDlg::alphabetItemChanged(QListWidgetItem *current,
                                      QListWidgetItem * /*previous*/)
{
    mfgList->clear();
    modelList->removeAllRows();
    if (current) {
        mfgList->addItems((*remoteDB)[current->text()].keys());
    }
}

void SelectRemoteDlg::mfgItemChanged(QListWidgetItem *current,
                                     QListWidgetItem * /*previous*/)
{
    modelList->removeAllRows();
    if (current) {
        RemoteList remotes =
            (*remoteDB)[alphabetList->currentItem()->text()][current->text()];
        foreach(Remote *remote, remotes) {
            modelList->addItem(remote);
        }
    }
}

void SelectRemoteDlg::downloadRemote()
{
    Remote *currentModel = modelList->selected();
    if (currentModel) {
        setResult(QDialog::Accepted);
        setBusy();
        connect(currentModel, SIGNAL(saveFinished()),
                this, SLOT(close()));
        currentModel->saveToFile();

        emit remoteChanged(*currentModel);
    }
}

void SelectRemoteDlg::getDB()
{
    if (onlinePollerThread != NULL) {
        delete onlinePollerThread;
        onlinePollerThread = NULL;
    }
    remoteDBMgr.getDBAsync();
}

void SelectRemoteDlg::refreshDB()
{
    setBusy(true);
    if (onlinePollerThread != NULL) {
        delete onlinePollerThread;
        onlinePollerThread = NULL;
    }
    onlinePollerThread = new OnlinePollerThread();
    connect(onlinePollerThread, SIGNAL(online()),
            this, SLOT(getDB()));
    onlinePollerThread->start();
}

void SelectRemoteDlg::showEvent(QShowEvent *event)
{
    refreshDB();
    QDialog::showEvent(event);
}

void SelectRemoteDlg::onDBError(int error)
{
    this->setWindowTitle(tr("Network error") + " " + QString::number(error));
    setBusy(false);
    downloadBtn->setVisible(false);
}

