#!/bin/bash

HEADER='./src/api/irreco_backend_api.h'

# Make sure that this script can be executed from anywhere.
HEADER="$( dirname "$0" )/$HEADER"

backend_maker()
{
	if [ "$1" == "" ]; then
		echo "Usage: $0 OUTPUT_FILE"
		echo ""
		echo "This script reads example function implementations from"
		echo "irreco_backend_api.h header and creates a matching template"
		echo "source file based on them. Then someone can use this template"
		echo "to build an Irreco Backend."
		echo ""
		exit
	else
		OUTPUT_FILE="$1"
		if [ -f "$OUTPUT_FILE" ]; then
			if [ "$( stat -c %Y "$OUTPUT_FILE" )" -gt \
			     "$( stat -c %Y "$HEADER" )" ]; then
				echo "\"$OUTPUT_FILE\" is up to date."
				exit 0
			else
				rm -f "$OUTPUT_FILE"
			fi
		fi
	fi
	
	echo "/*"					>> "$OUTPUT_FILE"
	echo " * This file was autogenerated by"	>> "$OUTPUT_FILE"
	echo " *    $0"					>> "$OUTPUT_FILE"
	echo " * from "					>> "$OUTPUT_FILE"
	echo " *    $HEADER"				>> "$OUTPUT_FILE"
	echo " */"					>> "$OUTPUT_FILE"
	echo ""						>> "$OUTPUT_FILE"
	echo '#define IRRECO_DEBUG_PREFIX "MYBA"'	>> "$OUTPUT_FILE"
	echo ""						>> "$OUTPUT_FILE"
	echo "#include <gtk/gtk.h>"			>> "$OUTPUT_FILE"
	echo "#include <irreco_backend_api.h>"		>> "$OUTPUT_FILE"
	echo ""						>> "$OUTPUT_FILE"
	echo "typedef struct _MyBackend MyBackend;"	>> "$OUTPUT_FILE"
	echo "struct _MyBackend {"			>> "$OUTPUT_FILE"
	echo "	gint important_variable;"		>> "$OUTPUT_FILE"
	echo "	/** @todo Implement. */"		>> "$OUTPUT_FILE"
	echo "};"					>> "$OUTPUT_FILE"
	echo ""						>> "$OUTPUT_FILE"
	echo ""						>> "$OUTPUT_FILE"
	
	cat "$HEADER" \
	| grep -v '^/\*\*.*\*/' \
	| api_parser "$@"
}

api_parser()
{
	CODEPOS=0
	COMMENTPOS=0
	local STATE='find'
	while read LINE; do
		
		# State change logic wich applies from this line.
		case "$STATE" in
		'find')
			echo "$LINE" \
			| grep '^/\*\*' > /dev/null \
			&& STATE='comment'
			;;
		'comment')
			echo "$LINE" \
			| fgrep '@par Implement' > /dev/null \
			&& IGNORE_LINE=1
			
			echo "$LINE" \
			| fgrep '@code' > /dev/null && STATE='startcode' \
			&& CODEFOUND=1
			
			echo "$LINE" \
			| fgrep '@addtogroup' > /dev/null \
			&& IGNORE=1
			;;
		'startcode')
			STATE='code'
			;;
		'code')
			echo "$LINE" \
			| fgrep '@endcode' > /dev/null && STATE='endcode'
			;;
		'endcode')
			STATE='comment'
			;;
		esac
		
		# Print line with state.
		if [ "$IGNORE_LINE" == "1" ]; then
			echo "Ignored: $LINE"
			unset IGNORE_LINE
		else
			echo "$STATE: $LINE"
			
			# Store intresting parts to arrays.
			case "$STATE" in
				'code') CODE[$CODEPOS]="$LINE"; ((CODEPOS++));;
				'comment') COMMENT[$COMMENTPOS]="$LINE"; ((COMMENTPOS++));;
			esac
		fi
			
		# Detect ending of comment block.
		if [ "$STATE" == 'comment' ] && echo "$LINE" \
		   | grep '\*/' > /dev/null; then
			
			# Print function prototype if it was found.
			if [[ "$CODEFOUND" == "1" && "$IGNORE" != "1" ]]; then
				echo
				echo "Writing function to \"$OUTPUT_FILE\"."
				echo
				
				POS=0
				while [ "${#COMMENT[@]}" -gt $POS ]; do
					echo "${COMMENT[$POS]}" \
					| sed 's|^\*| *|' >> "$OUTPUT_FILE"
					((POS++))
				done
				
				POS=0
				while [ "${#CODE[@]}" -gt $POS ]; do
					echo "${CODE[$POS]}" \
					| sed -r -e 's|^\*[ ]{0,1}||' \
					-e 's|/_|/*|' -e 's|_/|*/|' \
					>> "$OUTPUT_FILE"
					((POS++))
				done
				echo >> "$OUTPUT_FILE"
			fi
			
			# Reset state machine.
			STATE='find'
			unset CODE
			unset COMMENT
			CODEPOS=0
			COMMENTPOS=0
			CODEFOUND=0
			IGNORE=0
		fi
	done
}

backend_maker "$@"
