/*
 *  Copyright (C) 2011 Javier S. Pedro
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <math.h>
#include <gtk/gtk.h>
#include <pulse/pulseaudio.h>
#include <pulse/glib-mainloop.h>

#include "widget.h"

HD_DEFINE_PLUGIN_MODULE(IroncopeWidget, ironcope_widget, HD_TYPE_HOME_PLUGIN_ITEM);

#define IRONCOPE_WIDGET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), IRONCOPE_TYPE_WIDGET, IroncopeWidgetPrivate))

#define WIDGET_HEIGHT 50
#define WIDGET_WIDTH 300

#define PAINT_TIMER 100 /* ms. */

#define AUDIO_RATE 48000

/* Must be > than the scope area width, as we use one recorded level per pixel */
#define LEVEL_RECORDS 250
#define SAMPLES_PER_RECORD (AUDIO_RATE / (PAINT_TIMER * 2))

struct _IroncopeWidgetPrivate {
	gboolean running;
	guint timer;
	gfloat circle1;
	gfloat *levelsL;
	gfloat *levelsR;
	gfloat levelL;
	gfloat levelR;
	guint samples;
	guint silences;
	pa_glib_mainloop *mainloop;
	pa_context *context;
	pa_stream *stream;
};

static inline gboolean is_visible(IroncopeWidget *self)
{
	gboolean is_on_current_desktop;
	g_object_get(G_OBJECT(self),
		"is-on-current-desktop", &is_on_current_desktop, NULL);
	return is_on_current_desktop;
}

static void ironcope_widget_repaint(IroncopeWidget *self)
{
	GtkWidget *widget = GTK_WIDGET(self);
	if (!widget->window) return;

	GdkRegion *region = gdk_drawable_get_clip_region(widget->window);
	if (!region) return;

	gdk_window_invalidate_region(widget->window, region, TRUE);
	gdk_region_destroy(region);
}

static gboolean timer(gpointer data)
{
	IroncopeWidget *self = IRONCOPE_WIDGET(data);

	ironcope_widget_repaint(self);

	return TRUE;
}

static void start_stop_anim(IroncopeWidget *self, gboolean start)
{
	IroncopeWidgetPrivate *priv = self->priv;
	if (start) {
		if (!priv->timer) {
			priv->timer = g_timeout_add(PAINT_TIMER, timer, self);
			g_message("Animation started");
		}
	} else {
		ironcope_widget_repaint(self);
		if (priv->timer) {
			g_source_remove(priv->timer);
			priv->timer = 0;
			g_message("Animation stopped");
		}
	}
}

static void stream_state_callback(pa_stream *s, void *userdata)
{
	IroncopeWidget *self = IRONCOPE_WIDGET(userdata);
	IroncopeWidgetPrivate *priv = self->priv;

	switch (pa_stream_get_state(s)) {
		case PA_STREAM_READY:
            g_message("Stream ready");
			if (!pa_stream_is_suspended(s)) start_stop_anim(self, TRUE);
			break;

		case PA_STREAM_FAILED:
			g_critical("Stream connection failed: %s",
				pa_strerror(pa_context_errno(priv->context)));
			break;

		case PA_STREAM_UNCONNECTED:
		case PA_STREAM_CREATING:
		case PA_STREAM_TERMINATED:
			break;
	}
}

static void stream_suspended_callback(pa_stream *s, void *userdata)
{
	IroncopeWidget *self = IRONCOPE_WIDGET(userdata);

	start_stop_anim(self, pa_stream_is_suspended(s) ? FALSE : TRUE);
}

static void stream_read_callback(pa_stream *s, size_t nbytes, void *userdata)
{
	IroncopeWidget *self = IRONCOPE_WIDGET(userdata);
	IroncopeWidgetPrivate *priv = self->priv;
	const void *p;


    if (pa_stream_peek(s, &p, &nbytes) < 0) {
        g_warning("pa_stream_peek() failed: %s",
			pa_strerror(pa_context_errno(priv->context)));
        return;
    }

	const float *d = p;
	gsize n = nbytes / sizeof(float);

	while (n > 0) {
		if (priv->samples++ > SAMPLES_PER_RECORD) {
			if (priv->levelL < 0.05 || priv->levelR < 0.05) {
				priv->silences++;
			} else {
				priv->silences = 0;
			}

#if LOGARITHMIC
			priv->levelsL[LEVEL_RECORDS-1] = log10(priv->levelL*9+1);
			priv->levelsR[LEVEL_RECORDS-1] = log10(priv->levelR*9+1);
#else
			priv->levelsL[LEVEL_RECORDS-1] = priv->levelL;
			priv->levelsR[LEVEL_RECORDS-1] = priv->levelR;
#endif

			memmove(&priv->levelsL[0], &priv->levelsL[1],
				(LEVEL_RECORDS-1)*sizeof(float));
			memmove(&priv->levelsR[0], &priv->levelsR[1],
				(LEVEL_RECORDS-1)*sizeof(float));

			priv->levelL = 0;
			priv->levelR = 0;
			priv->samples = 0;
		}

		float v = fabs(*d);
		if (v > priv->levelL) {
			priv->levelL = v;
		}
		d++; n--;

		v = fabs(*d);
		if (v > priv->levelR) {
			priv->levelR = v;
		}
		d++; n--;
	}

    pa_stream_drop(s);
}

static void start_stop_pa(IroncopeWidget *self, gboolean start)
{
	IroncopeWidgetPrivate *priv = self->priv;

	if (start) {
		if (!priv->stream) {
			pa_sample_spec nss;
			nss.format = PA_SAMPLE_FLOAT32;
			nss.rate = AUDIO_RATE;
			nss.channels = 2;

			priv->stream = pa_stream_new(priv->context, "Ironcope", &nss, NULL);
			pa_stream_set_state_callback(priv->stream,
				stream_state_callback, self);
			pa_stream_set_suspended_callback(priv->stream,
				stream_suspended_callback, self);
			pa_stream_set_read_callback(priv->stream,
				stream_read_callback, self);

			g_message("Connecting");
			pa_stream_connect_record(priv->stream, "sink.hw0.monitor", NULL,
				PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND);
		}
	} else {
		start_stop_anim(self, FALSE);
		if (priv->stream) {
			g_message("Disconnecting");
			pa_stream_disconnect(priv->stream);
			pa_stream_unref(priv->stream);
			priv->stream = NULL;
		}
	}
}

static void context_state_callback(pa_context *c, void *userdata)
{
	IroncopeWidget *self = IRONCOPE_WIDGET(userdata);
	IroncopeWidgetPrivate *priv = self->priv;

	switch (pa_context_get_state(c)) {
		case PA_CONTEXT_READY:
			g_warn_if_fail(priv->stream == NULL);
			if (is_visible(self)) start_stop_pa(self, TRUE);
			break;

		case PA_CONTEXT_FAILED:
			g_critical("PulseAudio connection failed: %s",
				pa_strerror(pa_context_errno(c)));
			break;

		case PA_CONTEXT_UNCONNECTED:
		case PA_CONTEXT_CONNECTING:
		case PA_CONTEXT_AUTHORIZING:
		case PA_CONTEXT_SETTING_NAME:
		case PA_CONTEXT_TERMINATED:
			break;
	}
}

static void visibility_change_callback(GObject *object, GParamSpec *pspec, gpointer data)
{
	IroncopeWidget *self = IRONCOPE_WIDGET(data);
	gboolean visible = is_visible(self);

	start_stop_pa(self, visible);
}

static void ironcope_widget_circle_draw(IroncopeWidget *self, cairo_t *cr, guint size)
{
	IroncopeWidgetPrivate *priv = self->priv;
	const gdouble hsize = size / 2.0;
	const gdouble msize = size / 9.0;

	cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
	cairo_arc(cr, hsize, hsize, hsize, 0.0, 2.0 * M_PI);
	cairo_stroke(cr);
	cairo_arc(cr, hsize, hsize, msize, 0.0, 2.0 * M_PI);
	cairo_fill_preserve(cr);
	cairo_stroke(cr);

	if (priv->silences < 10 || priv->circle1 > 0.0f) {
		cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0 - priv->circle1);
		cairo_arc(cr, hsize, hsize, hsize * priv->circle1, 0.0, 2.0 * M_PI);
		cairo_stroke(cr);

		priv->circle1 += 0.07f;
		if (priv->circle1 > 1.0) priv->circle1 = 0.0f;
	}
}

static void ironcope_widget_rectangle_draw(IroncopeWidget *self, cairo_t *cr, guint width, guint height)
{
	IroncopeWidgetPrivate *priv = self->priv;
	guint hheight = height / 2;
	gint i;

	cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
	cairo_rectangle(cr, 0.0, 0.0, width, height);
	cairo_stroke(cr);

	cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 0.8);
	cairo_move_to(cr, 0.0, hheight);
	for (i = 0; i < width; i++) {
		cairo_line_to(cr, i, (1.0 - priv->levelsL[i]) * hheight);
	}
	cairo_line_to(cr, width, hheight);
	cairo_close_path(cr);
	cairo_fill(cr);

	cairo_move_to(cr, 0.0, hheight);
	for (i = 0; i < width; i++) {
		cairo_line_to(cr, i, (1.0 + priv->levelsR[i]) * hheight);
	}
	cairo_line_to(cr, width, hheight);
	cairo_close_path(cr);
	cairo_fill(cr);
}

static void ironcope_widget_draw(IroncopeWidget *self, cairo_t *cr)
{
	const GtkAllocation *size = &( GTK_WIDGET(self)->allocation );

	cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 0.0);
	cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
	cairo_paint(cr);
	cairo_set_operator(cr, CAIRO_OPERATOR_OVER);

	cairo_save(cr);
	cairo_translate(cr, 2, 2);
	ironcope_widget_circle_draw(self, cr, size->height - 4);
	cairo_restore(cr);

	gint rectangle_x = size->height + 8;
	cairo_save(cr);
	cairo_translate(cr, rectangle_x, 0);
	ironcope_widget_rectangle_draw(self, cr, size->width - rectangle_x, size->height);
	cairo_restore(cr);
}

static void ironcope_widget_realize(GtkWidget *widget)
{
	GdkScreen *screen = gtk_widget_get_screen(widget);

	gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
	gtk_widget_set_app_paintable(widget, TRUE);

	GTK_WIDGET_CLASS(ironcope_widget_parent_class)->realize(widget);
}

static void ironcope_widget_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
	requisition->width = WIDGET_WIDTH;
	requisition->height = WIDGET_HEIGHT;
}

static gboolean ironcope_widget_expose(GtkWidget *widget, GdkEventExpose *event)
{
	IroncopeWidget *self = IRONCOPE_WIDGET(widget);
	cairo_t *cr = gdk_cairo_create(widget->window);
	cairo_rectangle(cr, event->area.x, event->area.y, event->area.width,
		event->area.height);
	cairo_clip(cr);

	ironcope_widget_draw(self, cr);

	cairo_destroy(cr);

	return GTK_WIDGET_CLASS(ironcope_widget_parent_class)->expose_event(widget, event);
}

static void ironcope_widget_init(IroncopeWidget *self)
{
	IroncopeWidgetPrivate *priv;

	self->priv = priv = IRONCOPE_WIDGET_GET_PRIVATE(self);	

	priv->timer = 0;
	priv->circle1 = 0.0f;
	priv->levelsL = g_new0(gfloat, LEVEL_RECORDS);
	priv->levelsR = g_new0(gfloat, LEVEL_RECORDS);
	priv->levelL = 0.0f;
	priv->levelR = 0.0f;
	priv->samples = 0;

	priv->mainloop = pa_glib_mainloop_new(g_main_context_default());
	priv->context = pa_context_new(pa_glib_mainloop_get_api(priv->mainloop), "Ironcope");
	priv->stream = NULL;

	pa_context_set_state_callback(priv->context, context_state_callback, self);
	pa_context_connect(priv->context, NULL, PA_CONTEXT_NOAUTOSPAWN, NULL);

	g_signal_connect(G_OBJECT(self), "notify::is-on-current-desktop",
	                 G_CALLBACK(visibility_change_callback), self);
}

static void ironcope_widget_finalize(GObject *object)
{
	IroncopeWidget *self = IRONCOPE_WIDGET(object);
	IroncopeWidgetPrivate *priv = self->priv;

	start_stop_pa(self, FALSE);

	if (priv->timer) {
		g_source_remove(priv->timer);
		priv->timer = 0;
	}
	if (priv->stream) {
		pa_stream_unref(priv->stream);
		priv->stream = NULL;
	}
	if (priv->context) {
		pa_context_unref(priv->context);
		priv->context = NULL;
	}
	if (priv->mainloop) {
		pa_glib_mainloop_free(priv->mainloop);
		priv->mainloop = NULL;
	}
	if (priv->levelsL) {
		g_free(priv->levelsL);
		priv->levelsL = NULL;
	}
	if (priv->levelsR) {
		g_free(priv->levelsR);
		priv->levelsR = NULL;
	}

	G_OBJECT_CLASS(ironcope_widget_parent_class)->finalize(object);
}

static void ironcope_widget_class_init(IroncopeWidgetClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

	gobject_class->finalize = ironcope_widget_finalize;
	widget_class->realize = ironcope_widget_realize;
	widget_class->size_request = ironcope_widget_size_request;
	widget_class->expose_event = ironcope_widget_expose;

	g_type_class_add_private(klass, sizeof(IroncopeWidgetPrivate));
}

static void ironcope_widget_class_finalize(IroncopeWidgetClass *klass)
{
	
}

GtkWidget* ironcope_widget_new()
{
	return g_object_new(IRONCOPE_TYPE_WIDGET, NULL);
}

