/**
 * intervalometer
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to tomi.suviola@gmail.com so I can send you a copy immediately.
 *
 * @copyright  Copyright (c) 2010 Tomi Suviola
 * @license    http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

#include "camerainterface.h"
#include "plugindialog.h"

#include <QPluginLoader>
#include <QStringList>
#include <QDir>

#include <QLabel>
#include <QGridLayout>
#include <QPushButton>
#include <QListWidget>
#include <QListWidgetItem>
#include <QHeaderView>
#include <QDebug>

PluginDialog::PluginDialog(const QString &path, const QStringList &fileNames,
                           QWidget *parent) :
    QDialog(parent),
    label(new QLabel),
    listWidget(new QListWidget),
    okButton(new QPushButton(tr("OK"))),
    cancelButton(new QPushButton(tr("Cancel")))
{
    listWidget->setAlternatingRowColors(false);
    listWidget->setSelectionMode(QAbstractItemView::NoSelection);

    okButton->setDefault(true);

    connect(okButton, SIGNAL(clicked()), this, SLOT(accept()));
    connect(cancelButton, SIGNAL(clicked()), this, SLOT(reject()));

    QGridLayout *mainLayout = new QGridLayout;
    mainLayout->setColumnStretch(0, 1);
    mainLayout->setColumnStretch(2, 1);
    mainLayout->addWidget(label, 0, 0, 1, 3);
    mainLayout->addWidget(listWidget, 1, 0, 1, 3);
    mainLayout->addWidget(okButton, 2, 1);
    mainLayout->addWidget(cancelButton, 2, 2);
    setLayout(mainLayout);

    interfaceIcon.addPixmap(style()->standardPixmap(QStyle::SP_DirOpenIcon),
                            QIcon::Normal, QIcon::On);
    interfaceIcon.addPixmap(style()->standardPixmap(QStyle::SP_DirClosedIcon),
                            QIcon::Normal, QIcon::Off);
    featureIcon.addPixmap(style()->standardPixmap(QStyle::SP_FileIcon));

    setWindowTitle(tr("Plugin Information"));
    findPlugins(path, fileNames);
    listWidget->setSelectionMode(QAbstractItemView::MultiSelection);

}
void PluginDialog::accept()
{
    qDebug("accept");
    QList<QListWidgetItem *> items=listWidget->selectedItems();

    //write selected plugins into settings
    foreach (QListWidgetItem* item, items) {
        qDebug() << item->text();
    }

    QDialog::accept();
}


void PluginDialog::findPlugins(const QString &path,
                               const QStringList &fileNames)
{
    label->setText(tr("x found the following plugins\n"
                      "(looked in %1):")
                   .arg(QDir::toNativeSeparators(path)));

    const QDir dir(path);

    foreach (QObject *plugin, QPluginLoader::staticInstances())
        populateListWidget(plugin, tr("%1")
                                   .arg(plugin->metaObject()->className()));

    foreach (QString fileName, fileNames) {
        QPluginLoader loader(dir.absoluteFilePath(fileName));
        QObject *plugin = loader.instance();
        if (plugin)
            populateListWidget(plugin, fileName);
    }
    QDir pluginsDir;
    pluginsDir.cd(dir.absolutePath());
    foreach (QString fileName, pluginsDir.entryList(QDir::Files)) {
        qDebug("file: %s",fileName.toLocal8Bit().constData());

        QPluginLoader pluginLoader(pluginsDir.absoluteFilePath(fileName));
        QObject *plugin = pluginLoader.instance();
        if (plugin) {
            qDebug("plugin loaded");
            CameraInterface *intervalInterface = qobject_cast<CameraInterface *>(plugin);
            if (intervalInterface) {
                qDebug("plugin added");

                populateListWidget(plugin, intervalInterface->name());
            }
        }
    }

}


void PluginDialog::populateListWidget(QObject *plugin, const QString &text)
{
    QListWidgetItem *pluginItem = new QListWidgetItem(listWidget);
    pluginItem->setText(text);
//    listWidget->setItemExpanded(pluginItem, true);

    QFont boldFont = pluginItem->font();
    boldFont.setBold(true);
    pluginItem->setFont(boldFont);

    if (plugin) {
        CameraInterface *iBrush = qobject_cast<CameraInterface *>(plugin);
        if (iBrush)
            addItems(pluginItem, "CameraInterface", QStringList(iBrush->name()));
    }
}


void PluginDialog::addItems(QListWidgetItem *pluginItem,
                            const char *interfaceName,
                            const QStringList &features)
{
    QListWidgetItem *interfaceItem = new QListWidgetItem(interfaceName);
    interfaceItem->setText( interfaceName);
    interfaceItem->setIcon( interfaceIcon);

    foreach (QString feature, features) {
        if (feature.endsWith("..."))
            feature.chop(3);
        QListWidgetItem *featureItem = new QListWidgetItem(interfaceName);
        featureItem->setText( feature);
        featureItem->setIcon( featureIcon);
    }
}
