#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "radiostation.h"
#include "texteditautoresizer.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    QTime time;
    time.start();

    ui->setupUi(this);

    mediaPlayer = new QMediaPlayer(this, QMediaPlayer::StreamPlayback);

    stationsDb.checkDB();
    stationsDb.initDB();

    qDebug()<<stationsDb.getGenreList();
    qDebug()<<stationsDb.getCountryList();
    qDebug()<<stationsDb.getFullStationList();

    loadStationsList();
    populateList(stationsList);

    addAction(ui->actionToggleFullscreen);

    labelBox.setFixedHeight(50);
    labelBox.setAlignment(Qt::AlignCenter);
    infoBox.setWidget(&labelBox);
    ui->lyricsWebViewButton->setFont(QFont("Nokia Sans", 12));

    nam = new QNetworkAccessManager(this);
    connect(nam, SIGNAL(finished(QNetworkReply*)), this, SLOT(onRequestFinished(QNetworkReply*)));

    connect(mediaPlayer, SIGNAL(metaDataChanged()), SLOT(getMetaData()));
    connectBluetoothControl();
    new TextEditAutoResizer(ui->cronologyTextEdit);

    //setAttribute(Qt::WA_Maemo5StackedWindow);
    //setWindowFlags(Qt::Window);

    qDebug()<<time.elapsed();
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::on_playButton_clicked()
{
    if ( ui->stationsList->currentRow() == -1 ) return;

    mediaPlayer->stop();
    mediaPlayer->setMedia(QUrl(ui->stationsList->currentItem()->toolTip()));
    mediaPlayer->play();
    ui->titleLabel->setText(QString("<b><u>%1</b></u>").arg(ui->stationsList->currentItem()->text().split("\n").at(0)));
}

void MainWindow::getMetaData()
{

    QStringList extn_metadata = mediaPlayer->availableExtendedMetaData();
    int count_extn = extn_metadata.count();
    QString extn_metadata_key;
    QVariant extn_metadata_value;
    for( int indx = 0; indx < count_extn; indx++ )
    {
        extn_metadata_key = extn_metadata.at(indx);
        extn_metadata_value = mediaPlayer->extendedMetaData(extn_metadata_key);

        qDebug()<<"Key is: "<<extn_metadata_key;
        qDebug()<<"Value is: "<<extn_metadata_value.toString();

        if ( extn_metadata_key == "title")
        {
            QString title = mediaPlayer->extendedMetaData("title").toString();
            appendCronology(title);

            ui->playingLabel->setText(QString(tr("<b>Playing:</b> %1")).arg(title));

            labelBox.setText("Playing '" + extn_metadata_value.toString() + "'");
            infoBox.show();

            QFile file( "/home/user/MyDocs/.documents/InternetRadioPlayerLog.txt" );
            file.open((QIODevice::WriteOnly | QIODevice::Text | QIODevice::Append));
            QTextStream out(&file);
            out<<QDateTime::currentDateTime().toString(Qt::ISODate)<<" "<<extn_metadata_value.toString()<<endl;

            if (title.split('-').size()==2)
            {
                ui->artistLineEdit->setText(title.split('-', QString::SkipEmptyParts).at(0).trimmed());
                ui->titleLineEdit->setText(title.split('-', QString::SkipEmptyParts).at(1).trimmed());

                updateLyrics();
            }
        }

        if ( extn_metadata_key == "genre")
        {
            QString string = mediaPlayer->extendedMetaData("genre").toString();

            ui->genreLabel->setText(QString(tr("<b>Genre:</b> %1")).arg(string));

        }

        if ( extn_metadata_key == "bitrate")
        {
            QString string = mediaPlayer->extendedMetaData("bitrate").toString();

            ui->bitrateLabel->setText(QString(tr("<b>Bitrate:</b> %1")).arg(string));

        }

        if ( extn_metadata_key == "location")
        {
            QString string = mediaPlayer->extendedMetaData("location").toString();

            ui->locationLabel->setText(QString(tr("<b>Location:</b> %1")).arg(string));

        }

        if ( extn_metadata_key == "audio-codec")
        {
            QString string = mediaPlayer->extendedMetaData("audio-codec").toString();

            ui->audiocodecLabel->setText(QString(tr("<b>Audio codec:</b> %1")).arg(string));

        }
    }





}

void MainWindow::onRequestFinished(QNetworkReply *reply)
{
    qDebug()<<"Request Finished";

    QString songTitle;
    QString artistName;
    QString lyrics;
    QString albumArtUrl;

    QXmlStreamReader xml(reply->readAll());
    while(!xml.atEnd() &&
          !xml.hasError()) {
            /* Read next element.*/
            QXmlStreamReader::TokenType token = xml.readNext();
            /* If token is just StartDocument, we'll go to next.*/
            if (token == QXmlStreamReader::StartDocument) {
                    continue;
            }
            /* If token is StartElement, we'll see if we can read it.*/
            if (token == QXmlStreamReader::StartElement) {
                    /* If it's named person, we'll dig the information from there.*/
                if (xml.name() == "LyricSong")
                    songTitle = xml.readElementText();
                if (xml.name() == "Lyric") {
                    lyrics = xml.readElementText();
                }
                if (xml.name() == "LyricArtist")
                    artistName = xml.readElementText();
                if (xml.name() == "LyricCovertArtUrl")
                    albumArtUrl = xml.readElementText();
            }
    }
    QString text;
    text = lyrics;
    text = QString("Song: \"%1\"\nArtist: \"%2\"").arg(songTitle, artistName);
    //ui->metadataLabel->setText(text);
    this->loadImage(albumArtUrl);

    if (lyrics.size()>0) ui->lyricsLabel->setText(lyrics);
    else ui->lyricsLabel->setText(tr("No lyric found"));
}

void MainWindow::on_playingNowButton_clicked()
{
    ui->stackedWidget->setCurrentIndex(0);
}

void MainWindow::on_lyricsButton_clicked()
{
    ui->stackedWidget->setCurrentIndex(1);
}

void MainWindow::updateLyrics()
{
    QString url = "http://api.chartlyrics.com/apiv1.asmx/SearchLyricDirect?artist=%2&song=%1";
    QString getUrl = url.arg(ui->titleLineEdit->text(), ui->artistLineEdit->text());
    nam->get(QNetworkRequest(getUrl));

    ui->lyricsLabel->setText(tr("Loading lyrics..."));
    ui->albumImageLabel->setPixmap(QPixmap(":/icons/globe.png"));
}

void MainWindow::loadImage(QString Url)
{
    QUrl url(Url);
    http = new QHttp(this);
    connect(http, SIGNAL(requestFinished(int, bool)),this, SLOT(imageLoaded(int, bool)));
    buffer = new QBuffer(&bytes);
    buffer->open(QIODevice::WriteOnly);
    http->setHost(url.host());
    imageRequest = http->get(url.path(), buffer);
}

void MainWindow::imageLoaded(int requestId, bool)
{
    if (requestId != imageRequest)
        return;

    if ((QPixmap::fromImage(QImage::fromData(bytes)).isNull()))
    {
        qDebug()<<"No Album  art found";
        return;
    }

    ui->albumImageLabel->setPixmap(QPixmap::fromImage(QImage::fromData(bytes)).scaled(QSize(120, 120)));
    bytes.clear();
}



void MainWindow::on_stationsButton_clicked()
{
    ui->stackedWidget->setCurrentIndex(2);
}

void MainWindow::on_cronologyButton_clicked()
{
    ui->stackedWidget->setCurrentIndex(3);
}

void MainWindow::appendCronology(QString title)
{
    QString string = QString(QTime::currentTime().toString("hh:mm:ss")).append(" ").append(title);
    ui->cronologyTextEdit->append(string);
}

void MainWindow::setIcons()
{
    ui->previousButton->setIcon(QIcon(QPixmap(":/icons/bfzn_001.png")));
    ui->nextButton->setIcon(QIcon(QPixmap(":/icons/bfzn_002.png")));
    ui->playButton->setIcon(QIcon(QPixmap(":/icons/bfzn_004.png")));
    ui->stopButton->setIcon(QIcon(QPixmap(":/icons/bfzn_006.png")));
}

void MainWindow::on_stopButton_clicked()
{
    mediaPlayer->stop();
}

void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::about(this, tr("About"),
                        tr("<h2>Internet Radio Player 0.0.8</h2>"
                           "<p>"
                           "<p>Written by Giovanni Grammatico (ggiovanni88@yahoo.it)"
                           "<p><a href=\"http://code-tinkering.blogspot.com/\">My personal blog.</a>"
                           " <a href=\"mailto:ggiovanni88@yahoo.it\">Send me an email.</a>"));
}

void MainWindow::on_addStationButton_clicked()
{
    bool ok1, ok2, ok3, ok4;
    QString name, url, genre, country;
    name = QInputDialog::getText(this, tr("New Station"), tr("Insert Name: "), QLineEdit::Normal, QString(), &ok1, 0);
    if ( !ok1 ) return;
    url = QInputDialog::getText(this, tr("New Station"), tr("Insert Url: "), QLineEdit::Normal, QString(), &ok2, 0);
    if (!ok2 ) return;
    genre = QInputDialog::getText(this, tr("New Station"), tr("Insert Genre: "), QLineEdit::Normal, QString(), &ok3, 0);
    if (!ok3 ) return;
    country = QInputDialog::getText(this, tr("New Station"), tr("Insert Country: "), QLineEdit::Normal, QString(), &ok4, 0);
    if (!ok4 ) return;

    stationsDb.insertStation(name, url, genre, country);
    stationsList = stationsDb.getFullStationList();
    saveStationsList();
    ui->stationsList->clear();
    populateList(stationsList);


}

void MainWindow::on_editStationButton_clicked()
{
    int currentIndex = ui->stationsList->currentRow();
    if ( currentIndex == -1) return;

    QString currentName = ui->stationsList->currentItem()->text().split("\n").at(0);
    QString currentUrl = ui->stationsList->currentItem()->toolTip();

    bool ok1, ok2, ok3, ok4;
    QString name, url, genre, country;
    name = QInputDialog::getText(this, tr("Edit Station"), tr("Insert Name: "), QLineEdit::Normal, currentName, &ok1, 0);
    if ( !ok1 ) return;
    url = QInputDialog::getText(this, tr("Edit Station"), tr("Insert Url: "), QLineEdit::Normal, currentUrl, &ok2, 0);
    if (!ok2 ) return;
    genre = QInputDialog::getText(this, tr("Edit Station"), tr("Insert Genre: "), QLineEdit::Normal, QString(), &ok3, 0);
    if (!ok3 ) return;
    country = QInputDialog::getText(this, tr("Edit Station"), tr("Insert Country: "), QLineEdit::Normal, QString(), &ok4, 0);
    if (!ok4 ) return;

    stationsDb.updateStation(currentUrl, name, url, genre, country);

    stationsList = stationsDb.getFullStationList();
    saveStationsList();
    ui->stationsList->clear();
    populateList(stationsList);

}

void MainWindow::on_removeStationButton_clicked()
{
    int currentIndex = ui->stationsList->currentRow();
    qDebug()<<"Current Index: "<<currentIndex;
    if ( currentIndex == -1) return;

    int ret = QMessageBox::warning(this, tr("Internet Radio Widget"),
                                    tr("Do you really want to delete the chosen station?"),
                                    QMessageBox::Yes | QMessageBox::Cancel);

    if ( ret == QMessageBox::Yes)
    {
        QString url = ui->stationsList->currentItem()->toolTip();
        stationsDb.deleteStation(url);

        saveStationsList();

        stationsList = stationsDb.getFullStationList();
        saveStationsList();
        ui->stationsList->clear();
        populateList(stationsList);
    }


}

void MainWindow::populateList(QStringList list)
{

    for (int i=0; i<list.size(); i++)
    {
        QListWidgetItem *item = new QListWidgetItem(ui->stationsList);
        item->setText(list.at(i).split("--").at(0));//+"\n"+list.at(i).split("--").at(1));
        item->setToolTip(list.at(i).split("--").at(1));

        //item->setTextAlignment(Qt::AlignCenter);
    }
    ui->stationsList->sortItems();
}

void MainWindow::on_stationsList_itemClicked(QListWidgetItem *item)
{
    mediaPlayer->stop();
    mediaPlayer->setMedia(QUrl(item->toolTip()));
    mediaPlayer->play();
    ui->titleLabel->setText(QString("<b><u>%1</b></u>").arg(item->text().split("\n").at(0)));
}

void MainWindow::loadStationsList()
{


    QSettings settings("/home/user/MyDocs/.documents/InternetRadioPlayerStations.ini", QSettings::IniFormat);
    QStringList stations = settings.value("stationsList", getDefaultList()).toStringList();
    stations.sort();

    stationsList = stations;
}

void MainWindow::saveStationsList()
{
    QSettings settings("/home/user/MyDocs/.documents/InternetRadioPlayerStations.ini", QSettings::IniFormat);
    settings.setValue("stationsList", stationsList);
}

void MainWindow::on_actionToggleFullscreen_triggered()
{
    qDebug()<<"Buffer: "<<mediaPlayer->bufferStatus();
    bool isFullscreen = ui->actionToggleFullscreen->property("isFullScreen").toBool();

    if (isFullscreen)
    {
        this->showNormal();
        ui->actionToggleFullscreen->setProperty("isFullScreen", "false");
    }
    else
    {
        this->showFullScreen();
        ui->actionToggleFullscreen->setProperty("isFullScreen", "true");
    }
}

void MainWindow::on_previousButton_clicked()
{
    int index;
    index = ui->stationsList->currentIndex().row();
    ui->stationsList->setCurrentRow(index-1);

    if ( ui->stationsList->currentRow() == -1 ) return;

    mediaPlayer->stop();
    mediaPlayer->setMedia(QUrl(ui->stationsList->currentItem()->toolTip()));
    mediaPlayer->play();
    ui->titleLabel->setText(QString("<b><u>%1</b></u>").arg(ui->stationsList->currentItem()->text().split("\n").at(0)));
}

void MainWindow::on_nextButton_clicked()
{
    int index;
    index = ui->stationsList->currentIndex().row();
    ui->stationsList->setCurrentRow(index+1);

    if ( ui->stationsList->currentRow() == -1 ) return;

    mediaPlayer->stop();
    mediaPlayer->setMedia(QUrl(ui->stationsList->currentItem()->toolTip()));
    mediaPlayer->play();
    ui->titleLabel->setText(QString("<b><u>%1</b></u>").arg(ui->stationsList->currentItem()->text().split("\n").at(0)));
}

void MainWindow::on_actionAbout_Qt_triggered()
{
    QApplication::aboutQt();
}

void MainWindow::on_stationsList_customContextMenuRequested(const QPoint &pos)
{
    QMenu menu(this);
    menu.addAction(ui->actionDoNotFilter);
    menu.addAction(ui->actionFilterListByGenre);
    menu.addAction(ui->actionFilterListByCountry);
    menu.addAction(ui->actionRestoreDefaultList);
    menu.addAction(ui->actionClearList);
    menu.exec(pos);
}

void MainWindow::on_actionClearList_triggered()
{
    int ret = QMessageBox::warning(this, tr("Internet Radio Widget"),
                                    tr("This will clear your stations list."
                                       "<p>Do you really want to continue?"),
                                    QMessageBox::Yes | QMessageBox::Cancel);

    if ( ret == QMessageBox::Yes)
    {
        stationsList.clear();
        ui->stationsList->clear();
        stationsDb.clearDB();
        saveStationsList();
    }
}

void MainWindow::on_actionRestoreDefaultList_triggered()
{
    int ret = QMessageBox::warning(this, tr("Internet Radio Widget"),
                                    tr("This will clear your stations list and restore default one."
                                       "<p>Do you really want to continue?"),
                                    QMessageBox::Yes | QMessageBox::Cancel);

    if ( ret == QMessageBox::Yes)
    {
        stationsList.clear();
        ui->stationsList->clear();

        stationsDb.restoreDefaultDB();

        stationsList = stationsDb.getFullStationList();

        saveStationsList();
        populateList(stationsList);
    }
}

QStringList MainWindow::getDefaultList()
{
    QStringList list;
    list<<"ITA - Golden Hit Radio--http://87.117.203.226:8014"
        <<"ITA - Ibiza on Radio--http://81.174.67.45:8600"
        <<"ITA - Italian Graffiati--http://s9.viastreaming.net:8000"
        <<"ITA - Radio EmerGeNti--http://stream15.top-ix.org:80/radioemergenti"
        <<"ITA - RTL 102.5--http://shoutcast.rtl.it:3010/"
        <<"ITA - Radio 105 FM--http://shoutcast.unitedradio.it:1101/"
        <<"ITA - Radio BMP--http://84.33.192.44:8090/bpm"
        <<"ITA - Radio Diffusione--http://65.60.24.222:8900/"
        <<"ITA - Radio Italia--mms://89.202.214.2/66360a73-2b6d-483e-ae7e-29eee545d37c"
        <<"ITA - Radio Stop--mmsh://onair5.xdevel.com/radiostop?MSWMExt=.asf"
        <<"ITA - Lifegate Radio--mms://bbwms.libero.it/lifegate"
        <<"ITA - Radio KissKiss Italia--http://str2.creacast.com:80/kisskiss_italia"
        <<"UK - 1Mix Radio--http://marc.1mix.co.uk:8000/"
        <<"UK - Capital FM--http://media-ice.musicradio.com:80/CapitalMP3"
        <<"UK - Chill--mms://mediasrv-the.musicradio.com/Chill"
        <<"UK - Gold--mms://mediasrv-the.musicradio.com/CapitalGold"
        <<"UK - Ministry of Sound Radio--mms://wms.ministryofsound.com/MinistryLive"
        <<"UK - NonStopPlay.com--http://dedi.nonstopplay.co.uk:80"
        <<"USA - Trance FM Dj--http://uk01.dj.trance.fm/128"
        <<"USA - Trance FM--http://uk01.tc.trance.fm/128"
        <<"USA - DigitallyImported Chillout--http://scfire-dtc-aa02.stream.aol.com:80/stream/1035";

    return list;
}

void MainWindow::on_artistLineEdit_returnPressed()
{
    updateLyrics();
}

void MainWindow::on_titleLineEdit_returnPressed()
{
    updateLyrics();
}

void MainWindow::on_actionFilterListByGenre_triggered()
{
    QStringList genreList = stationsDb.getGenreList();
    FilterListByDialog dialog(0, genreList, this);
    connect(&dialog, SIGNAL(choicePicked(int,QString)), this, SLOT(filterList(int,QString)));
    dialog.exec();

}

void MainWindow::on_actionFilterListByCountry_triggered()
{
    QStringList countryList = stationsDb.getCountryList();
    FilterListByDialog dialog(1, countryList);
    connect(&dialog, SIGNAL(choicePicked(int,QString)), this, SLOT(filterList(int,QString)));
    dialog.exec();
}

void MainWindow::filterList(int type, QString filter)
{
    qDebug()<<type<<filter;

    if ( type == 0 )
    {
        stationsList = stationsDb.getListFilteredByGenre(filter);

        saveStationsList();

        ui->stationsList->clear();
        populateList(stationsList);
    }
    else
    {
        stationsList = stationsDb.getListFilteredByCountry(filter);

        saveStationsList();

        ui->stationsList->clear();
        populateList(stationsList);
    }
}

void MainWindow::on_actionDoNotFilter_triggered()
{
    stationsList = stationsDb.getFullStationList();

    saveStationsList();

    ui->stationsList->clear();
    populateList(stationsList);
}

void MainWindow::connectBluetoothControl()
{
    connect(&bluetoothAdapter, SIGNAL(prevButtonPressed()), this, SLOT(on_previousButton_clicked()));
    connect(&bluetoothAdapter, SIGNAL(nextButtonPressed()), this, SLOT(on_nextButton_clicked()));
    connect(&bluetoothAdapter, SIGNAL(toggleButtonPressed()), this, SLOT(toggleButtonPressed()));


}

void MainWindow::toggleButtonPressed()
{
    if ( mediaPlayer->state() == QMediaPlayer::PlayingState )
    {
        on_stopButton_clicked();
    }

    else if ( mediaPlayer->state() == QMediaPlayer::StoppedState )
    {
        on_playButton_clicked();
    }

}




void MainWindow::on_lyricsWebViewButton_clicked()
{
    QString artist = ui->artistLineEdit->text();
    QString title = ui->titleLineEdit->text();
    QString windowTitle = QString("%1 - %2").arg(artist, title);
    qDebug()<<"1";
    artist.replace(' ', '-');
    title.replace(' ', '-');
    qDebug()<<"String:"<<artist<<"    "<<title;
    QString url = QString("http://www.lyrics.com/%1-lyrics-%2.html").arg(title).arg(artist);
    qDebug()<<"Url: "<<url;
    lyricsWebView = new LyricsWebView(this);
    lyricsWebView->setUrl(url);
    lyricsWebView->setWindowTitle(windowTitle);
    lyricsWebView->show();
}
