#include "imagesetmaker.h"
#include "ui_imagesetmaker.h"

#include <QFileDialog>
#include <QMaemo5InformationBox>
#include <QMaemo5ValueButton>

imageSetMaker::imageSetMaker(QWidget* parent) :
    QMainWindow(parent),
    ui(new Ui::imageSetMaker)
{
    ui->setupUi(this);

    // Setup a few QLists with 4 items each
    for(i=0; i<4; i++) {
        // Objects
        copyToBtn.append(new QPushButton);
        copyToBtnExists.append(new bool);
        imgPath.append(new QString);

        // UI elements
        vertLayout.append(new QVBoxLayout);
        imgPrev.append(new QLabel);
    }

    setupIsmMenuBar();

    vertLayout[0] = ui->verticalLayout_1;
    vertLayout[1] = ui->verticalLayout_2;
    vertLayout[2] = ui->verticalLayout_3;
    vertLayout[3] = ui->verticalLayout_4;
    for(i=0; i<4; i++)
        vertLayout[i]->setAlignment(Qt::AlignTop);

    imgPrev[0] = ui->imgPrev1;
    imgPrev[1] = ui->imgPrev2;
    imgPrev[2] = ui->imgPrev3;
    imgPrev[3] = ui->imgPrev4;

    ui->visibChkbox->setChecked(false);

    for(i=0; i<4; i++)
        *copyToBtnExists[i] = false;

    fileWithPath = new QString;
    properImage = new bool;

    copyToDialogExists = false;
    overwriteDialogExists = false;
    aboutDialogExists = false;
    curCopyToBtn = 8;
    // This default dir changes after selecting an image
    lastUsedDir = "/home/user/MyDocs/.images/";

    QRegExp nameRegExp("^[a-zA-Z0-9_-]*$");
    // ^                From the beginning of the string
    // [a-zA-Z0-9_-]    Only alphanumeric, _ and -
    // *                At least 0
    // $                Until the end of the string
    ui->nameEdit->setValidator(new QRegExpValidator(nameRegExp, ui->nameEdit));

    // Create 4 QMaemo5ValueButtons on main screen
    createImgBtns();

    connect(copyToBtn[0], SIGNAL(clicked()),
            this, SLOT(clickedCopyToBtn0()));
    connect(copyToBtn[1], SIGNAL(clicked()),
            this, SLOT(clickedCopyToBtn1()));
    connect(copyToBtn[2], SIGNAL(clicked()),
            this, SLOT(clickedCopyToBtn2()));
    connect(copyToBtn[3], SIGNAL(clicked()),
            this, SLOT(clickedCopyToBtn3()));

    connect(ui->nameEdit, SIGNAL(textChanged(QString)),
            this, SLOT(updateSaveBtn()));

    connect(ui->saveBtn, SIGNAL(clicked()),
            this, SLOT(setSaveFile()));

    connect(ui->visibChkbox, SIGNAL(clicked()),
            this, SLOT(updateSaveBtn()));
}

imageSetMaker::~imageSetMaker() {
    delete ui;
}





//********* START imageSetMaker public functions *******************
void imageSetMaker::createImgBtns() {
    // Setup 4 imgBtn's on main screen
    for(i=0; i<4; i++) {
        imgBtn.append(new QMaemo5ValueButton);
        imgBtn[i]->setParent(this);
        imgBtn[i]->setMinimumSize(190, 60);
        imgBtn[i]->setMaximumSize(190, 60);
        imgBtn[i]->setText(tr("Image ") + QString::number(i+1));
        imgBtn[i]->setValueText(tr("(no image)"));
        imgBtn[i]->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
        vertLayout[i]->addWidget(imgBtn[i], 0, Qt::AlignHCenter);
    }

    connect(imgBtn[0], SIGNAL(clicked()),
            this, SLOT(clickedImgBtn0()));
    connect(imgBtn[1], SIGNAL(clicked()),
            this, SLOT(clickedImgBtn1()));
    connect(imgBtn[2], SIGNAL(clicked()),
            this, SLOT(clickedImgBtn2()));
    connect(imgBtn[3], SIGNAL(clicked()),
            this, SLOT(clickedImgBtn3()));
}

void imageSetMaker::setupIsmMenuBar() {
    ismMenuActionReset = new QAction(tr("Reset screen"), this);
    ismMenuActionAbout = new QAction(tr("About..."), this);

    ismMenu = menuBar()->addMenu("");
    ismMenu->addAction(ismMenuActionReset);
    ismMenu->addAction(ismMenuActionAbout);

    connect(ismMenuActionReset, SIGNAL(triggered()),
            this, SLOT(resetScreen()));
    connect(ismMenuActionAbout, SIGNAL(triggered()),
            this, SLOT(createAboutDialog()));
}

void imageSetMaker::openImage(int imgIndex) {
    *fileWithPath = QFileDialog::getOpenFileName(this,
                                                 "Open image",
                                                 lastUsedDir.path()); // No filedialog title

        if(!fileWithPath->isEmpty()) { // If the filedialog is NOT cancelled (path NOT empty)
        // Save last used directory for next button press
        lastUsedDir.setPath(QFileInfo(*fileWithPath).absolutePath());
        *properImage = resCheck(*fileWithPath); // Check 800x480 resolution

        if(*properImage == true) { // If the file is 800x480
            *imgPath[imgIndex] = *fileWithPath;
            updateImg(imgIndex);
            setupCopyToBtn(imgIndex);
            updateSaveBtn();
        } else { // If the file is NOT 800x480, don't change anything (including lastUsedDir)
            QMaemo5InformationBox::information(this,
                                               tr("Please select an 800 x 480 image"));
        }
    }
}

bool imageSetMaker::resCheck(QString curFile) {
    QLabel* resCheckLabel = new QLabel(this);
    resCheckLabel->setPixmap(curFile);
    resCheckLabel->adjustSize(); // QLabel takes image dimensions

    if(resCheckLabel->width() != 800 || resCheckLabel->height() != 480) {
        delete resCheckLabel;
        return false;
    } else {
        delete resCheckLabel;
        return true;
    }
}

void imageSetMaker::setupCopyToBtn(int imgBtnNum) {
    // Setup the current copyToBtn
    // If hidden after resetScreen(), show it again
    if(*copyToBtnExists[imgBtnNum] == true && copyToBtn[imgBtnNum]->isVisible() == false)
        copyToBtn[imgBtnNum]->setVisible(true);
    else if(*copyToBtnExists[imgBtnNum] == false) {
        copyToBtn[imgBtnNum]->setParent(this);
        copyToBtn[imgBtnNum]->setMaximumSize(190, 50);
        copyToBtn[imgBtnNum]->setText(tr("Copy to..."));
        vertLayout[imgBtnNum]->addWidget(copyToBtn[imgBtnNum], 0, Qt::AlignHCenter);
    }
    *copyToBtnExists[imgBtnNum] = true;
}

void imageSetMaker::createCopyToDialog(int copyToBtnNum) {
    // Don't create a new copyToDialog if it exists already
    // Just enable buttons again
    if(copyToDialogExists == true) {
        for(i=0; i<4; i++) {
            copyToDialogBtn[i]->setEnabled(true);
            copyToDialogBtn[i]->setChecked(false);
        }
    }
    else if(copyToDialogExists == false) {
        copyToDialog = new QDialog(this);
        copyToDialog->setWindowTitle("Copy to...");

        createCopyToDialogBtns();

        copyToDialogOK = new QPushButton;
        copyToDialogOK->setText("OK");
        copyToDialogOK->setMinimumSize(190, 60);
        copyToDialogOK->setMaximumSize(190, 60);

        // OK button closes dialog and copies images
        connect(copyToDialogOK, SIGNAL(clicked()),
                imageSetMaker::copyToDialog, SLOT(close()));
        connect(copyToDialogOK, SIGNAL(clicked()),
                this, SLOT(copyToOthers()));

        // Layout all the buttons
        copyToDialogH1 = new QHBoxLayout();
        for(i=0; i<4; i++)
            copyToDialogH1->addWidget(copyToDialogBtn[i]);
        copyToDialogH1->setAlignment(Qt::AlignHCenter);

        copyToDialogH2 = new QHBoxLayout;
        copyToDialogH2->addWidget(copyToDialogOK);
        copyToDialogH2->setAlignment(Qt::AlignHCenter);

        copyToDialogV = new QVBoxLayout;
        copyToDialogV->addLayout(copyToDialogH1);
        copyToDialogV->addLayout(copyToDialogH2);

        copyToDialog->setLayout(copyToDialogV);
        copyToDialog->adjustSize();
        copyToDialogExists = true;
    }

    // Disable the button that is being copied from
    for(i=0; i<4; i++) {
        if(copyToBtnNum == i) { // If user wants to copy image #, disable button #
            curCopyToBtn = i;
            copyToDialogBtn[i]->setEnabled(false);
            copyToDialogBtn[i]->setChecked(false);
        }
        else if(copyToBtnNum < 0 || copyToBtnNum > 3) { // If current button value is not valid
            // Checks 4 times, so disables all 4 buttons
            copyToDialogBtn[i]->setEnabled(false);
            copyToDialogBtn[i]->setChecked(false);
        }
    }

    copyToDialog->show();
}

void imageSetMaker::createCopyToDialogBtns() {
    // Create and setup 4 copyToDialogBtn's
    for(i=0; i<4; i++) {
        copyToDialogBtn.append(new QPushButton);
        copyToDialogBtn[i]->setText(tr("Image ") + QString::number(i+1));
        copyToDialogBtn[i]->setMinimumSize(190, 60);
        copyToDialogBtn[i]->setMaximumSize(190, 60);
        copyToDialogBtn[i]->setCheckable(true);
        copyToDialogBtn[i]->setChecked(false);
        copyToDialogBtn[i]->setEnabled(true);
    }
}

void imageSetMaker::updateImg(int pathNum) {
    // Update imgBtn and imgPrev according to *imgPath[pathNum]
    for(i=0; i<4; i++) {
        if(pathNum == i) {
            imgPrev[i]->setPixmap(*imgPath[i]);
            imgBtn[i]->setValueText(QFileInfo(*imgPath[i]).fileName());
        }
    }
}

void imageSetMaker::saveDesktopFile() {
    saveFile.open(QFile::WriteOnly | QFile::Truncate);
        QTextStream fileContent(&saveFile);
        fileContent << "[Desktop Entry]\n"
                << "Type=Background Image\nName=" << ui->nameEdit->text() << "\n"
                << "Hidden=True\n\n"
                << "X-File1=" << *imgPath[0] << "\n"
                << "X-File2=" << *imgPath[1] << "\n"
                << "X-File3=" << *imgPath[2] << "\n"
                << "X-File4=" << *imgPath[3] << "\n\n"
                << "X-Order=01";
    saveFile.close();
}

void imageSetMaker::createOverwriteDialog() {
    // Don't create the dialog if it exists already
    // Just update owQuestion
    if(overwriteDialogExists == true) {
        owQuestion->setText(nameEditLower + tr(".desktop already exists, overwrite?"));
        owQuestion->adjustSize();
    }
    else if(overwriteDialogExists == false) {
        overwriteDialog = new QDialog(this);
        overwriteDialog->setWindowTitle(tr("Overwrite existing file?"));

            owQuestion = new QLabel;
            owQuestion->setText(nameEditLower + tr(".desktop already exists, overwrite?"));
            owQuestion->adjustSize();

            owYes = new QPushButton;
            owYes->setText(tr("Yes"));
            owYes->setMinimumSize(190, 70);
            owYes->setMaximumSize(190, 70);
            connect(owYes, SIGNAL(clicked()),
                    this, SLOT(confirmOverwrite()));

            owNo = new QPushButton;
            owNo->setText(tr("No"));
            owNo->setMinimumSize(190, 70);
            owNo->setMaximumSize(190, 70);
            connect(owNo, SIGNAL(clicked()),
                    imageSetMaker::overwriteDialog, SLOT(close()));

            owVLayout1 = new QVBoxLayout;
            owVLayout1->addWidget(owQuestion);
            owVLayout1->setAlignment(Qt::AlignHCenter);

            owVLayout2 = new QVBoxLayout;
            owVLayout2->addWidget(owYes);
            owVLayout2->addWidget(owNo);

            owHLayout = new QHBoxLayout;
            owHLayout->addLayout(owVLayout1);
            owHLayout->addLayout(owVLayout2);

        overwriteDialog->setLayout(owHLayout);
        overwriteDialog->adjustSize();
    }

    overwriteDialog->show();
}

void imageSetMaker::checkDesktopFileSaved() {
    if(saveFile.exists() == true)
        QMaemo5InformationBox::information(this,
                                           "<br><b>"
                                           + tr("Image set saved!")
                                           + "</b><br><br>"
                                           + saveFile.fileName()
                                           + "<br>",
                                           QMaemo5InformationBox::NoTimeout);
    if(saveFile.exists() == false)
        QMaemo5InformationBox::information(this,
                                           "<br><b>"
                                           + tr("Image set NOT saved...")
                                           + "</b><br><br>"
                                           + tr("Please try again")
                                           + "<br>",
                                           QMaemo5InformationBox::NoTimeout);
}
//********* END imageSetMaker public functions *******************





//********* START imageSetMaker public slots *******************
void imageSetMaker::clickedImgBtn0() { openImage(0); }
void imageSetMaker::clickedImgBtn1() { openImage(1); }
void imageSetMaker::clickedImgBtn2() { openImage(2); }
void imageSetMaker::clickedImgBtn3() { openImage(3); }

void imageSetMaker::updateSaveBtn() {
    if(ui->nameEdit->text().isEmpty()
        || *imgPath[0] == ""
        || *imgPath[1] == ""
        || *imgPath[2] == ""
        || *imgPath[3] == "") // If one or more objects are empty
    {
        // saveBtn disabled, set default button text
        ui->saveBtn->setEnabled(false);
        ui->saveBtn->setText(tr("Please set name and images first"));
    } else { // If everything is filled in
        // Enable and update saveBtn, update nameEditLower (nameEdit lowercase)
        ui->saveBtn->setEnabled(true);
        nameEditLower = ui->nameEdit->text().toLower();
        // Check if "Invisible file" is checked, change filename accordingly
        if(ui->visibChkbox->isChecked())
            nameEditLower = "." + nameEditLower;
        ui->saveBtn->setText(tr("Save ") + nameEditLower + ".desktop");
    }
}

void imageSetMaker::setSaveFile() {
    // Always save to /home/user/MyDocs/.images/
    saveFile.setFileName("/home/user/MyDocs/.images/" + nameEditLower + ".desktop");

    if(saveFile.exists()) {
        // File already exists, overwrite?
        createOverwriteDialog();
    } else {
        // File doesn't exist, save the .desktop file
        saveDesktopFile();
        // After saving, check if the file is saved
        checkDesktopFileSaved();
    }
}

void imageSetMaker::clickedCopyToBtn0() { createCopyToDialog(0); }
void imageSetMaker::clickedCopyToBtn1() { createCopyToDialog(1); }
void imageSetMaker::clickedCopyToBtn2() { createCopyToDialog(2); }
void imageSetMaker::clickedCopyToBtn3() { createCopyToDialog(3); }

void imageSetMaker::createAboutDialog() {
    // Only create a new dialog if it hasn't been made yet
    if(aboutDialogExists == false) {
        aboutDialog = new QDialog(this);
        aboutDialog->setWindowTitle("Image Set Maker 0.2.1-1" + tr(" for ") + "Nokia N900");

        aboutIcon = new QLabel;
        aboutIcon->setPixmap(QPixmap(":/images/imagesetmaker.png"));
        aboutIcon->adjustSize();

        aboutText = new QLabel;
        aboutText->setText(tr("This app creates a .desktop file for wallpapers.\n")
                           + tr("It doesn't get much easier than this ;)\n\n\n")
                           + " 2010 Sybren Dijkstra\n"
                           + "('Berserk'"
                           + tr(" on ")
                           + "Maemo.org)");
        aboutText->adjustSize();

        aboutOK = new QPushButton;
        aboutOK->setMinimumSize(190, 60);
        aboutOK->setMaximumSize(190, 60);
        aboutOK->setText("OK");
        connect(aboutOK, SIGNAL(clicked()),
                imageSetMaker::aboutDialog, SLOT(close()));

        aboutLayoutV1 = new QVBoxLayout;
        aboutLayoutV1->addWidget(aboutIcon);
        aboutLayoutV1->setAlignment(Qt::AlignTop);

        aboutLayoutV2 = new QVBoxLayout;
        aboutLayoutV2->addWidget(aboutText);
        aboutLayoutV2->setAlignment(Qt::AlignTop);

        aboutLayoutV3 = new QVBoxLayout;
        aboutLayoutV3->addWidget(aboutOK);
        aboutLayoutV3->setAlignment(Qt::AlignBottom);

        aboutLayoutH = new QHBoxLayout;
        aboutLayoutH->addLayout(aboutLayoutV1);
        aboutLayoutH->addLayout(aboutLayoutV2);
        aboutLayoutH->addLayout(aboutLayoutV3);

        aboutDialog->setLayout(aboutLayoutH);
        aboutDialog->adjustSize();
    }
    aboutDialog->show();
}

void imageSetMaker::confirmOverwrite() {
    overwriteDialog->close();
    // Save the .desktop file
    saveDesktopFile();
    // After saving, check if the file is saved
    checkDesktopFileSaved();
}

void imageSetMaker::copyToOthers() {
    // Current source image number is defined in curCopyToBtn
    srcPath = new QString;

    // Store source
    for(i=0; i<4; i++) {
        if(curCopyToBtn == i)
            *srcPath = *imgPath[i];
    }

    // Copy from source path to imgPath[#], based on checked buttons
    // then updates imgBtn[#], imgPrev[#] and copyToBtn[#]
    for(j=0; j<4; j++) {
        if(copyToDialogBtn[j]->isChecked() == true) {
            *imgPath[j] = *srcPath;
            updateImg(j);
            setupCopyToBtn(j);
        }
    }
    // updateSaveBtn afterwards
    updateSaveBtn();
}

void imageSetMaker::resetScreen() {
    // nameEdit
    ui->nameEdit->setText("");
    ui->visibChkbox->setChecked(false);
    // imgPath, imgPrev, imgBtn, copyToBtn
    for(i=0; i<4; i++) {
        *imgPath[i] = "";
        imgPrev[i]->setText("(no image)");
        imgBtn[i]->setValueText("(no image)");
        copyToBtn[i]->setVisible(false);
    }
    // saveBtn
    updateSaveBtn();
}
//********* END imageSetMaker public slots *******************
