#ifndef LUM_DEF_PROPS_H
#define LUM_DEF_PROPS_H

/*
  This source is part of the Illumination library
  Copyright (C) 2010  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <list>

#include <Lum/Def/Boolean.h>
#include <Lum/Def/Desc.h>
#include <Lum/Def/Number.h>
#include <Lum/Def/OneOfMany.h>
#include <Lum/Def/OneOfN.h>

namespace Lum {
  namespace Def {

    class LUMAPI PropItem
    {
    protected:
      PropItem();

    public:
      virtual ~PropItem();
    };

    class LUMAPI PropBoolean : public PropItem
    {
    private:
      Boolean boolean;

    public:
      PropBoolean(const Boolean& boolean);

      const Boolean& GetValue() const;
    };

    class LUMAPI PropInt : public PropItem
    {
    private:
      Int number;

    public:
      PropInt(const Int& number);

      const Int& GetValue() const;
    };

    class LUMAPI PropUInt : public PropItem
    {
    private:
      UInt number;

    public:
      PropUInt(const UInt& number);

      const UInt& GetValue() const;
    };

    class LUMAPI PropLong : public PropItem
    {
    private:
      Long number;

    public:
      PropLong(const Long& number);

      const Long& GetValue() const;
    };

    class LUMAPI PropULong : public PropItem
    {
    private:
      ULong number;

    public:
      PropULong(const ULong& number);

      const ULong& GetValue() const;
    };

    class LUMAPI PropSizeT : public PropItem
    {
    private:
      SizeT number;

    public:
      PropSizeT(const SizeT& number);

      const SizeT& GetValue() const;
    };

    template<class N>
    class LUMAPI PropInterval : public PropItem
    {
    private:
      Desc desc;
      N    start;
      N    end;

    public:
      PropInterval(const Desc& desc,
                   const N& start,
                   const N& end)
       : desc(desc),
         start(start),
         end(end)
      {
        // no code
      }

      const Desc& GetDesc() const
      {
        return desc;
      }

      const N& GetStart() const
      {
        return start;
      }

      const N& GetEnd() const
      {
        return end;
      }
    };

#define PropInterval_INSTANCIATION(type)\
    class LUMAPI Prop##type##Interval : public PropInterval<type>\
    {\
    public:\
      Prop##type##Interval(const Desc& desc, const type& start, const type& end) : PropInterval<type>(desc,start,end) {};\
    }

    PropInterval_INSTANCIATION(Int);

    PropInterval_INSTANCIATION(UInt);

    PropInterval_INSTANCIATION(Long);

    PropInterval_INSTANCIATION(ULong);

    PropInterval_INSTANCIATION(SizeT);

    PropInterval_INSTANCIATION(Float);

    PropInterval_INSTANCIATION(Double);

#undef PropInterval_INSTANCIATION

    class LUMAPI PropOneOfMany : public PropItem
    {
    private:
      OneOfMany oneOfMany;

    public:
      PropOneOfMany(const OneOfMany& oneOfMany);

      const OneOfMany& GetValue() const;
    };

    class LUMAPI PropOneOfN : public PropItem
    {
    private:
      OneOfN oneOfN;

    public:
      PropOneOfN(const OneOfN& oneOfN);

      const OneOfN& GetValue() const;
    };

    class LUMAPI PropGroup : public PropItem
    {
    private:
      PropGroup            *parent;
      Desc                 description;
      std::list<PropItem*> props;

    private:
      PropGroup(PropGroup* parent,const Desc& description);

    public:
      PropGroup();
      PropGroup(const PropGroup& other);
      ~PropGroup();

      // Base

      PropGroup* Boolean(const Def::Boolean& boolean);
      PropGroup* Int(const Def::Int& number);
      PropGroup* UInt(const Def::UInt& number);
      PropGroup* Long(const Def::Long& number);
      PropGroup* ULong(const Def::ULong& number);
      PropGroup* SizeT(const Def::SizeT& number);
      PropGroup* IntInterval(const Desc& desc,
                             const Def::Int& start,
                             const Def::Int& end);
      PropGroup* UIntInterval(const Desc& desc,
                              const Def::UInt& start,
                              const Def::UInt& end);
      PropGroup* LongInterval(const Desc& desc,
                              const Def::Long& start,
                              const Def::Long& end);
      PropGroup* ULongInterval(const Desc& desc,
                              const Def::ULong& start,
                              const Def::ULong& end);
      PropGroup* SizeTInterval(const Desc& desc,
                              const Def::SizeT& start,
                              const Def::SizeT& end);
      PropGroup* OneOfN(const Def::OneOfN& oneOfN);
      PropGroup* OneOfMany(const Def::OneOfMany& oneOfMany);
      PropGroup* Group(const Desc& description);
      PropGroup* Group(const std::wstring& name);
      PropGroup* End();
      PropGroup* Append(PropGroup* group);

      const std::list<PropItem*>& GetProps() const;

      const Desc& GetDesc() const;

      static PropGroup* Create();
    };
  }
}

#endif
