#include <Lum/Model/Adjustment.h>

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <algorithm>

namespace Lum {
  namespace Model {

    Adjustment::Adjustment()
    : top(new SizeT(1)),
      visible(new SizeT(0)),
      total(new SizeT(0)),
      stepSize(1)
    {
      // no code
    }

    void Adjustment::On(bool notify)
    {
      top->On(notify);
      visible->On(notify);
      total->On(notify);
    }

    void Adjustment::Off()
    {
      top->Off();
      visible->Off();
      total->Off();
    }

    /**
      As it is impossible to ensure that the relation between
      top, visible and total is not invalidated, this procedure
      provides a simple check.
      post: 0<top<=total and 0<visible OR result is FALSE.
     */
    bool Adjustment::IsValid() const
    {
      return !(top->IsNull() || visible->IsNull() || total->IsNull()
               || visible->Get()==0
               || visible->Get()>total->Get()
               || top->Get()<1);
    }

    size_t Adjustment::GetTop() const
    {
      if (top->IsNull()) {
        return 1;
      }
      else {
        return top->Get();
      }
    }

    size_t Adjustment::GetBottom() const
    {
      return top->Get()+visible->Get()-1;
    }

    size_t Adjustment::GetVisible() const
    {
      return visible->Get();
    }

    size_t Adjustment::GetTotal() const
    {
      return total->Get();
    }

    size_t Adjustment::IsCompletelyVisible() const
    {
      return GetVisible()==GetTotal();
    }

    /**
      Set a new value for top, visible and total.
      post: if visible>=1 then new value is accepted.
      post: top will be corrected to fit visible and total
     */
    void Adjustment::Set(size_t top, size_t visible, size_t total)
    {
      bool t,v,a;

      if (visible==0 || total==0) {
        return;
      }

      if (visible>total) {
        visible=total;
      }

      if (visible==total) {
        top=1;
      }

      if (top+visible>total) {
        top=total-visible+1;
      }

      // To make sure, that all values are consistent if one of the three models
      // triggers it change, we first switch off the changed models, set all values
      // and then switch all changed models back on.

      a=this->total->IsNull() || this->total->Get()!=total;
      v=this->visible->IsNull() ||this->visible->Get()!=visible;
      t=this->top->IsNull() ||this->top->Get()!=top;

      if (a) {
        this->total->Off();
        this->total->Set(total);
      }
      if (v) {
        this->visible->Off();
        this->visible->Set(visible);
      }
      if (t) {
        this->top->Off();
        this->top->Set(top);
      }

      if (a) {
        this->total->On();
      }
      if (v) {
        this->visible->On();
      }
      if (t) {
        this->top->On();
      }

      if (t || v || a) {
        Notify();
      }
    }

    void Adjustment::SetInvalid()
    {
      if (IsValid()) {
        // To make sure, that all values are consistent if one of the three models
        // triggers it change, we first switch off the changed models, set all values
        // and then switch all changed models back on.

        top->Off();
        visible->Off();
        total->Off();

        top->SetNull();
        visible->SetNull();
        total->SetNull();

        top->On();
        visible->On();
        total->On();

        Notify();
      }
    }

    /**
      Set a new value for top.
      post: if top>=1 & top<=total then new value is accepted.
     */
    void Adjustment::SetTop(size_t top)
    {
      if (top>=1 && top<=total->Get()-visible->Get()+1) {
        if (this->top->IsNull() || this->top->Get()!=top) {
          this->top->Set(top);
          Notify();
        }
      }
    }

    /**
      Set a new value for visible and total.
      post: if visible>=1 then new value is accepted.
      post: top will be corrected to fit visible and total
     */
    void Adjustment::SetDimension(size_t visible, size_t total)
    {
      size_t top;

      if (visible==0 || total==0) {
        return;
      }

      if (IsValid()) {
        top=GetTop();
      }
      else {
        top=1;
      }

      if (visible>total) {
        visible=total;
      }

      if (visible==total) {
        top=1;
      }

      if (top+visible>total) {
          top=total-visible+1;
      }

      Set(top,visible,total);
    }

    SizeT* Adjustment::GetTopModel() const
    {
      return top.Get();
    }

    SizeT* Adjustment::GetVisibleModel() const
    {
      return visible.Get();
    }

    SizeT* Adjustment::GetTotalModel() const
    {
      return total.Get();
    }

    void Adjustment::IncTop()
    {
      IncTop(stepSize);
    }

    void Adjustment::IncTop(size_t stepSize)
    {
      if (IsValid()) {
        if (top->Get()+stepSize-1<=total->Get()-visible->Get()) {
          top->Add(stepSize);
          Notify();
        }
        else if (top->Get()<=total->Get()-visible->Get()) {
          top->Set(total->Get()-visible->Get()+1);
          Notify();
        }
      }
    }

    void Adjustment::DecTop()
    {
      DecTop(stepSize);
    }

    void Adjustment::DecTop(size_t stepSize)
    {
      if (IsValid()) {
        if (top->Get()>=1+stepSize) {
          top->Sub(stepSize);
          Notify();
        }
        else if (top->Get()>1) {
          top->Set(1);
          Notify();
        }
      }
    }

    void Adjustment::PageBack()
    {
      if (IsValid()) {
        if (top->Get()>visible->Get()) {
          top->Sub(visible->Get());
          Notify();
        }
        else if (top->Get()!=1) {
          top->Set(1);
          Notify();
        }
      }
    }

    void Adjustment::PageForward()
    {
      if (IsValid()) {
        if (top->Get()+visible->Get()<total->Get()-visible->Get()+1) {
          top->Add(visible->Get());
          Notify();
        }
        else if (top->Get()!=total->Get()-visible->Get()+1) {
          top->Set(total->Get()-visible->Get()+1);
          Notify();
        }
      }
    }

    void Adjustment::MakeVisible(size_t pos)
    {
      if (pos>=1 && pos<=total->Get()) {
        if (pos<top->Get()) {
          top->Set(pos);
          Notify();
        }
        else if (pos>top->Get()+visible->Get()-1) {
          if (pos<visible->Get()) {
            top->Set(1);
          }
          else {
            top->Set(pos-visible->Get()+1);
          }
          Notify();
        }
      }
    }

    void Adjustment::CenterOn(size_t pos)
    {
      if (pos<=visible->Get()/2) {
        ShowFirstPage();
      }
      else if (pos>=total->Get()-visible->Get()/2+1) {
        ShowLastPage();
      }
      else {
        SetTop(pos-visible->Get()/2);
      }
    }

    void Adjustment::ShowFirstPage()
    {
      SetTop(1);
    }
    void Adjustment::ShowLastPage()
    {
      SetTop(total->Get()-visible->Get()+1);
    }

    void Adjustment::SetStepSize(size_t stepSize)
    {
      this->stepSize=stepSize;
    }

    size_t Adjustment::GetStepSize() const
    {
      return stepSize;
    }

  }
}
