#ifndef LUM_MODEL_INTEGER_H
#define LUM_MODEL_INTEGER_H

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <assert.h>

#include <limits>

#include <Lum/Model/Simple.h>

namespace Lum {
  namespace Model {

    template<class N>
    class LUMAPI Integer : public Simple<N>
    {
    public:
      Integer()
      : Simple<N>(),
        minimum(std::numeric_limits<N>:: min()),
        maximum(std::numeric_limits<N>:: max())
      {
        // no code
      }

      Integer(const N& value)
      : Simple<N>(value),
        minimum(std::numeric_limits<N>:: min()),
        maximum(std::numeric_limits<N>:: max())
      {
        // no code
      }

      Integer(const N& value, const N& min, const N& max)
      : Simple<N>(value),
        minimum(min),
        maximum(max)
      {
        // no code
      }

      void Inc()
      {
        assert(!this->IsNull());

        this->value.value++;

        this->Notify();
      }

      void Dec()
      {
        assert(!this->IsNull());

        this->value.value--;

        this->Notify();
      }

      void Add(N x)
      {
        assert(!this->IsNull());

        this->value.value+=x;

        this->Notify();
      }

      void Sub(N x)
      {
        assert(!this->IsNull());

        this->value.value-=x;

        this->Notify();
      }

      N GetMin() const
      {
        return minimum;
      }

      N GetMax() const
      {
        return maximum;
      }

      void SetRange(N minimum, N maximum)
      {
        assert(maximum>=minimum);

        if (this->minimum!=minimum || this->maximum!=maximum) {
          this->minimum=minimum;
          this->maximum=maximum;
          this->Notify();
        }
      }

      void SetToMin()
      {
        Set(minimum);
      }

      void SetToMax()
      {
        Set(maximum);
      }

    private:
      N minimum;
      N maximum;
    };

    /**
      This is a (temporary) pre gcc 4.0 hack, since attribute visibility does
      not work in this case together with typedefs.
     */

#define INTEGER_INSTANCIATION(name, type)\
    class LUMAPI name : public Integer<type>\
    {\
    public:\
      name() : Integer<type>() {};\
      name(const type&value) : Integer<type>(value) {};\
      name(const type&value, const type&min, const type&max) : Integer<type>(value,min,max) {};\
    }

    //
    // int
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Integer<int>;
#endif

    INTEGER_INSTANCIATION(Int,int);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<Int>;
#endif

    typedef Base::Reference<Int> IntRef;

    //
    // unsigned int
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Integer<unsigned int>;
#endif

    INTEGER_INSTANCIATION(UInt,unsigned int);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<UInt>;
#endif

    typedef Base::Reference<UInt> UIntRef;

    //
    // long
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Integer<long>;
#endif

    INTEGER_INSTANCIATION(Long,long);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<Long>;
#endif

    typedef Base::Reference<Long> LongRef;

    //
    // unsigned long
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Integer<unsigned long>;
#endif

    INTEGER_INSTANCIATION(ULong,unsigned long);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<ULong>;
#endif

    typedef Base::Reference<ULong> ULongRef;

    //
    // size_t
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Integer<size_t>;
#endif

    INTEGER_INSTANCIATION(SizeT,size_t);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<SizeT>;
#endif

    typedef Base::Reference<SizeT> SizeTRef;

    //
    // double
    //

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Integer<double>;
#endif

    INTEGER_INSTANCIATION(Double,double);

#if defined(LUM_INSTANTIATE_TEMPLATES)
    LUM_EXPTEMPL template class LUMAPI Base::Reference<Double>;
#endif

    typedef Base::Reference<Double> DoubleRef;

#undef INTEGER_INSTANCIATION
  }
}

#endif
