/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/View.h>

#include <cstdlib>
#include <cmath>

#include <Lum/Base/Util.h>

#include <Lum/Scroller.h>

namespace Lum {

  View::View()
  : object(NULL),
    hScroller(NULL),vScroller(NULL),
    kineticScroller(new KineticScroller()),
    hVisible(false),vVisible(false),
    kineticScrolling(true),
    refreshAction(new Model::Action())
  {
    SetBackground(OS::display->GetFill(OS::Display::scrolledBackgroundFillIndex));

    Observe(refreshAction);
  }

  View::~View()
  {
    delete kineticScroller;
    delete hScroller;
    delete vScroller;
    delete object;
  }

  bool View::VisitChildren(Visitor &visitor, bool onlyVisible)
  {
    if (hScroller!=NULL && (!onlyVisible || hVisible)) {
      if (!visitor.Visit(hScroller)) {
        return false;
      }
    }

    if (vScroller!=NULL && (!onlyVisible || vVisible)) {
      if (!visitor.Visit(vScroller)) {
        return false;
      }
    }

    if (object!=NULL) {  // if grid was not yet created (before calcSize)
      if (!visitor.Visit(object)) {
        return false;
      }
    }

    return true;
  }

  Scrollable* View::GetObject() const
  {
    return object;
  }

  void View::SetObject(Scrollable* object)
  {
    assert(object!=NULL);

    if (this->object!=NULL) {
      Forget(object->GetHAdjustment());
      Forget(object->GetVAdjustment());
      delete this->object;
    }

    this->object=object;

    object->SetParent(this);
    object->SetFlex(true,true);

    Observe(object->GetHAdjustment());
    Observe(object->GetVAdjustment());

    kineticScroller->SetObjects(this,this->object);
  }

  bool View::SetModel(Base::Model* model)
  {
    return object->SetModel(model);
  }

  void View::SetKineticScrolling(bool kineticScrolling)
  {
    this->kineticScrolling=kineticScrolling;
  }

  void View::CalcSize()
  {
    assert(object!=NULL);

    vScroller=new Scroller(true);
    vScroller->SetParent(this);
    vScroller->SetFlex(false,true);
    vScroller->SetModel(object->GetVAdjustment());

    hScroller=new Scroller(false);
    hScroller->SetParent(this);
    hScroller->SetFlex(true,false);
    hScroller->SetModel(object->GetHAdjustment());

    if (object->RequestedFocus()) {
      /* Delegate focusing to the real table */
      UnrequestFocus();
    }

    object->CalcSize();

    minWidth=object->GetOMinWidth();
    width=object->GetOWidth();
    minHeight=object->GetOMinHeight();
    height=object->GetOHeight();

    hScroller->CalcSize();

    minWidth=std::max(hScroller->GetOWidth(),minWidth);
    width=std::max(hScroller->GetOWidth(),width);
    minHeight+=hScroller->GetOHeight();
    height+=hScroller->GetOHeight();

    vScroller->CalcSize();

    minWidth+=vScroller->GetOMinWidth();
    width+=vScroller->GetOWidth();
    minHeight=std::max(vScroller->GetOHeight(),minHeight);
    height=std::max(vScroller->GetOHeight(),height);

    Group::CalcSize();
  }

  void View::Layout()
  {
    object->Resize(width-vScroller->GetOWidth(),
                   height-hScroller->GetOHeight());
    object->Layout();

    if (hVisible &&
        (!object->GetHAdjustment()->IsValid() ||
         object->GetHAdjustment()->GetVisible()==object->GetHAdjustment()->GetTotal())) {
      hScroller->Hide();
      hVisible=false;
    }
    else if (!hVisible &&
             object->GetHAdjustment()->IsValid() &&
             object->GetHAdjustment()->GetVisible()<object->GetHAdjustment()->GetTotal()) {
      hVisible=true;
    }

    if (vVisible &&
        (!object->GetVAdjustment()->IsValid() ||
         object->GetVAdjustment()->GetVisible()==object->GetVAdjustment()->GetTotal())) {
      vScroller->Hide();
      vVisible=false;
    }
    else if (!vVisible &&
             object->GetVAdjustment()->IsValid() &&
             object->GetVAdjustment()->GetVisible()<object->GetVAdjustment()->GetTotal()) {
      vVisible=true;
    }

    size_t tWidth,tHeight;

    tWidth=width;
    tHeight=height;

    // Calculate object dimensions

    if (hVisible) {
      tHeight-=hScroller->GetOHeight();
    }

    if (vVisible) {
      tWidth-=vScroller->GetOWidth();
    }

    // Resize and position objects

    object->MoveResize(x,y,tWidth,tHeight);

    if (hVisible) {
      hScroller->MoveResize(x,y+tHeight,tWidth,hScroller->GetOHeight());
    }

    if (vVisible) {
      vScroller->MoveResize(x+tWidth,y,vScroller->GetOWidth(),tHeight);
    }

    Group::Layout();
  }

  bool View::InterceptMouseEvents() const
  {
    return true;
  }

  bool View::HandleMouseEvent(const OS::MouseEvent& event)
  {
    if (object==NULL) {
      return false;
    }

    if (kineticScrolling &&
        kineticScroller->HandleMouseEvent(event)) {
      return true;
    }

    return Group::HandleMouseEvent(event);
  }

  void View::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (object==NULL) {
      return;
    }

    if (model==object->GetHAdjustment() ||
        model==object->GetVAdjustment()) {
      if (visible) {
        OS::display->QueueActionForEventLoop(refreshAction);
      }
    }
    else if (model==refreshAction && refreshAction->IsFinished()) {
      if ((hVisible &&
           (!object->GetHAdjustment()->IsValid() ||
            object->GetHAdjustment()->GetVisible()==object->GetHAdjustment()->GetTotal())) ||
          (!hVisible &&
           object->GetHAdjustment()->IsValid() &&
           object->GetHAdjustment()->GetVisible()<object->GetHAdjustment()->GetTotal()) ||
          (vVisible &&
           (!object->GetVAdjustment()->IsValid() ||
            object->GetVAdjustment()->GetVisible()==object->GetVAdjustment()->GetTotal())) ||
          (!vVisible &&
           object->GetVAdjustment()->IsValid() &&
           object->GetVAdjustment()->GetVisible()<object->GetVAdjustment()->GetTotal())) {
        SetRelayout();
        Redraw();
      }
    }

    if (kineticScrolling) {
      kineticScroller->Resync(model,msg);
    }

    Group::Resync(model,msg);
  }

  View* View::Create(bool horizontalFlex, bool verticalFlex)
  {
    View *v;

    v=new View();
    v->SetFlex(horizontalFlex,verticalFlex);

    return v;
  }

  View* View::Create(Scrollable* object, bool horizontalFlex, bool verticalFlex)
  {
    View *v;

    v=new View();
    v->SetFlex(horizontalFlex,verticalFlex);
    v->SetObject(object);

    return v;
  }
}

