/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Images/Loader.h>

#include <Lum/Private/Config.h>

#ifdef HAVE_LIB_PNG
#include <Lum/Images/Loader/PNG.h>
#endif

#ifdef HAVE_LIB_JPEG
#include <Lum/Images/Loader/JPEG.h>
#endif

#ifdef HAVE_LIB_TIFF
#include <Lum/Images/Loader/TIFF.h>
#endif

#ifdef HAVE_LIB_GIF
#include <Lum/Images/Loader/GIF.h>
#endif

#include <Lum/Base/Path.h>

#include <Lum/OS/Display.h>

namespace Lum {
  namespace Images {

    Loader::~Loader()
    {
      // no code
    }

    static Manager*                 singletonStore;
    ::Lum::Base::Singleton<Manager> loader(&singletonStore);

    Manager::Manager()
    {
#ifdef HAVE_LIB_PNG
      list.push_back(new LoaderPNG);
#endif

#ifdef HAVE_LIB_JPEG
      list.push_back(new LoaderJPEG);
#endif

#ifdef HAVE_LIB_TIFF
      list.push_back(new LoaderTIFF);
#endif

#ifdef HAVE_LIB_GIF
      list.push_back(new LoaderGIF);
#endif
    }

    Manager::~Manager()
    {
      std::list<Loader*>::iterator iter;

      iter=list.begin();
      while (iter!=list.end()) {
        delete *iter;

        ++iter;
      }
    }

    bool Manager::Load(const std::wstring& filename,
                       ImageRef &image) const
    {
      assert(image.Valid());

      std::list<Loader*>::const_iterator iter;

      iter=list.begin();
      while (iter!=list.end()) {
        if ((*iter)->CanLoad(filename)) {
          return (*iter)->Load(filename,image);
        }

        ++iter;
      }

      return false;
    }

    bool Manager::LoadThemed(const std::wstring& filename,
                             ImageRef &image) const
    {
      assert(image.Valid());

      std::list<Loader*>::const_iterator iter;

      iter=list.begin();
      while (iter!=list.end()) {
        Lum::Base::Path path;

        path.SetNativeDir(::Lum::OS::display->GetThemePath());
        path.SetBaseName(filename);

        if ((*iter)->CanLoad(path.GetPath())) {
          return (*iter)->Load(path.GetPath(),image);
        }

        ++iter;
      }

      return false;
    }
  }
}
