/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Combo.h>

#include <Lum/Text.h>

namespace Lum {

  Combo::Table::Table()
  : success(false)
  {
    Observe(GetMouseSelectionAction());
  }

  Combo::Table::~Table()
  {
    // no code
  }

  void Combo::Table::Resync(Base::Model* model,
                            const Base::ResyncMsg& msg)
  {
    if (model==GetMouseSelectionAction() && GetMouseSelectionAction()->IsFinished()) {
      success=true;
      GetWindow()->Exit();
    }
    else {
      TableView::Resync(model,msg);
    }
  }

  Combo::Combo(Object* value, bool editable)
  : imageRight(true),
    editable(editable),
    table(NULL),
    popup(NULL),
    prevAction(new Lum::Model::Action()),
    nextAction(new Lum::Model::Action()),
    tableModel(NULL),
    selection(new Model::SingleLineSelection()),
    value(value)
  {
    if (editable) {
      SetBackground(OS::display->GetFill(OS::Display::editComboBackgroundFillIndex));
    }
    else {
      SetBackground(OS::display->GetFill(OS::Display::comboBackgroundFillIndex));
    }

    SetCanFocus(true);
    RequestFocus();
    SetRedrawOnMouseActive(true);

    value->SetParent(this);

    if (dynamic_cast<Control*>(value)==NULL) {
      value->SetBackground(NULL);
    }

    headerModel=new Model::HeaderImpl();
    headerModel->AddColumn(L"",Lum::Base::Size::pixel,0);

    Observe(prevAction);
    Observe(nextAction);
  }

  Combo::~Combo()
  {
    delete value;
  }

  bool Combo::HasBaseline() const
  {
    return value!=NULL && value->HasBaseline();
  }

  size_t Combo::GetBaseline() const
  {
    assert(value!=NULL);

    if (value->HasBaseline()) {
      // We also assume that is vertically flexible
      return GetTopBorder()+value->GetBaseline();
    }
    else {
      return 0;
    }
  }

  Object* Combo::GetFocusObject()
  {
    if (value->CanFocus()) {
      return value;
    }
    else {
      return this;
    }
  }

  bool Combo::VisitChildren(Visitor &visitor, bool /*onlyVisible*/)
  {
    if (!visitor.Visit(value)) {
      return false;
    }

    return true;
  }

  /**
    Call this method to assign the table model that will be used to display
    all possible values in the pop window.
  */
  void Combo::SetHeaderModel(Model::Header* model)
  {
    headerModel=model;
  }

  /**
    Call this method to assign the table model that will be used to display
    all possible values in the pop window.
  */
  void Combo::SetTableModel(Model::Table* model)
  {
    if (tableModel.Valid()) {
      Forget(tableModel);
    }

    tableModel=model;

    if (tableModel.Valid()) {
      Observe(tableModel);
    }
  }

  void Combo::CalcSize()
  {
    /* Delegate Focus visualisation to value class, if possible */
    if (value->RequestedFocus()) {
      UnrequestFocus();
    }

    /*
    if (editable) {
      if (background.Valid() && background->HasBorder()) {
        value->SetFrame(new OS::EmptyFrame());
      }
    }*/

    value->SetFlex(true,true);
    value->CalcSize();

    if (editable) {
      OS::ImageRef image=OS::display->GetImage(OS::Display::comboEditButtonImageIndex);

      minWidth=image->GetWidth();

      width=minWidth;

      minWidth+=value->GetOMinWidth();
      minWidth+=value->GetOWidth();

      minHeight=std::max(value->GetOMinHeight(),
                         image->GetHeight());
      height=std::max(value->GetOHeight(),
                      image->GetHeight());
    }
    else {
      OS::ImageRef image=OS::display->GetImage(OS::Display::comboImageIndex);
      OS::ImageRef divider=OS::display->GetImage(OS::Display::comboDividerImageIndex);

      minWidth=OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)+
               divider->GetWidth()+
               OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)+
               image->GetWidth();

      width=minWidth;

      minWidth+=value->GetOMinWidth();
      width+=value->GetOWidth();

      minHeight=std::max(value->GetOMinHeight(),image->GetHeight());
      height=std::max(value->GetOHeight(),image->GetHeight());
    }

    Control::CalcSize();
  }

  /**
    Overload this method if you want some inital initialisation of the table
    object used for the popup window. The baseclass is empty.

    Normally this will be done by a concrete implementation of Combo.
  */
  void Combo::InitTable(::Lum::Table* /*table*/)
  {
    // no code
  }

  Combo::ComboPopup::ComboPopup(Combo *combo)
  : combo(combo)
  {
    Observe(GetOpenedAction());
  }

  Combo::ComboPopup::~ComboPopup()
  {
    // no code
  }

  void Combo::ComboPopup::PreInit()
  {
    combo->table=new Lum::Table();

    combo->table->SetTableView(new Table());
    combo->table->SetFlex(true,true);
    combo->table->SetMinHeight(Base::Size::stdCharHeight,1);
    combo->table->SetMaxWidth(Base::Size::workVRel,40);
    combo->table->SetMaxHeight(Base::Size::workVRel,40);
    combo->table->GetTableView()->SetAutoFitColumns(true);
    combo->table->SetHeaderModel(combo->headerModel);
    combo->table->SetModel(combo->tableModel);
    combo->table->SetSelection(combo->selection);
    combo->table->GetTableView()->SetAutoHSize(true);
    combo->table->GetTableView()->SetAutoVSize(true);
    RegisterCommitShortcut(combo->table->GetTableView(),
                           combo->table->GetTableView()->GetMouseSelectionAction());

    combo->InitTable(combo->table);

    SetMain(combo->table,true,false);

    Popup::PreInit();
  }

  void Combo::ComboPopup::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      if (combo->tableModel.Valid() && combo->selection->GetLine()>0) {
        combo->table->GetTableView()->MakeVisible(1,combo->selection->GetLine());
      }
    }

    Popup::Resync(model,msg);
  }

  void Combo::OpenPopup()
  {
    popup=new ComboPopup(this);
    popup->SetParent(GetWindow());
    popup->SetReference(this);

    if (popup->Open()) {
      popup->EventLoop();
      popup->Close();

      if (dynamic_cast<Table*>(table->GetTableView())->success && tableModel.Valid() && selection->HasSelection()) {
        CopySelection(selection->GetLine());
      }
    }

    delete popup;
    popup=NULL;
  }

  void Combo::SetTableRow(size_t row)
  {
    selection->SelectCell(1,row);
    CopySelection(row);
  }

  bool Combo::HandleMouseEvent(const OS::MouseEvent& event)
  {
    if (!visible || (GetModel()==NULL) || !GetModel()->IsEnabled()) {
      return false;
    }

    if (event.type==OS::MouseEvent::down && PointIsIn(event) && event.button==OS::MouseEvent::button1) {
      OpenPopup();
      return true;
    }

    return false;
  }

  bool Combo::HandleKeyEvent(const OS::KeyEvent& event)
  {
    if (dynamic_cast<Control*>(value)!=NULL && dynamic_cast<Control*>(value)->HandleKeyEvent(event)) {
      return true;
    }

    if (event.type==OS::KeyEvent::down) {
      switch (event.key) {
      case OS::keyUp:
        prevAction->Trigger();
        return true;
        break;

      case OS::keyDown:
        nextAction->Trigger();
        return true;
        break;

      default:
        break;
      }
    }

    return false;
  }

  void Combo::Layout()
  {
    if (editable) {
      OS::ImageRef image=OS::display->GetImage(OS::Display::comboEditButtonImageIndex);

      value->Resize(width-image->GetWidth(),height);

      if (imageRight) {
        value->Move(x,y+(height-value->GetOHeight()) / 2);
      }
      else {
        value->Move(x+image->GetWidth(),
                    y+(height-value->GetOHeight()) / 2);
      }
    }
    else {
      OS::ImageRef image=OS::display->GetImage(OS::Display::comboImageIndex);
      OS::ImageRef divider=OS::display->GetImage(OS::Display::comboDividerImageIndex);

      value->Resize(width-
                    image->GetWidth()-
                    2*OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)-
                    divider->GetWidth(),
                    height-2*OS::display->GetSpaceVertical(OS::Display::spaceObjectBorder));

      if (imageRight) {
        value->Move(x,
                    y+(height-value->GetOHeight()) / 2);
      }
      else {
        value->Move(x+image->GetWidth()+
                    2*OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)+
                     divider->GetWidth(),
                    y+(height-value->GetOHeight()) / 2);
      }
    }

    value->Layout();

    Control::Layout();
  }

  void Combo::PrepareForBackground(OS::DrawInfo* draw)
  {
    draw->activated=IsMouseActive();
    draw->focused=HasFocus();
    draw->disabled=!(GetModel()!=NULL) || GetModel()->IsNull() || !GetModel()->IsEnabled();
  }

  void Combo::Draw(int x, int y, size_t w, size_t h)
  {
    OS::DrawInfo *draw=GetDrawInfo();

    Control::Draw(x,y,w,h);

    if (!OIntersect(x,y,w,h)) {
      return;
    }

    /* --- */

    draw->activated=IsMouseActive();
    draw->focused=HasFocus();
    draw->disabled=!(GetModel()!=NULL) || GetModel()->IsNull() || !GetModel()->IsEnabled();

    if (editable) {
      OS::ImageRef image=OS::display->GetImage(OS::Display::comboEditButtonImageIndex);

      if (imageRight) {
        int    bx,by;
        size_t bw,bh;

        bx=this->x+value->GetOWidth();
        by=this->y;
        bw=image->GetWidth();
        bh=height;

        image->Draw(draw,
                    bx,by,
                    image->GetWidth(),
                    height);
      }
      else {
        image->Draw(draw,
                    this->x,this->y,
                    image->GetWidth(),
                    height);
      }
    }
    else {
      OS::ImageRef image=OS::display->GetImage(OS::Display::comboImageIndex);
      OS::ImageRef divider=OS::display->GetImage(OS::Display::comboDividerImageIndex);

      if (imageRight) {
        divider->Draw(draw,
                      this->x+width-
                      image->GetWidth()-
                      OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject)-
                      divider->GetWidth(),
                      this->y,
                      image->GetWidth(),
                      height);

        image->Draw(draw,
                    this->x+width-
                    image->GetWidth(),
                    this->y+(height-image->GetHeight()) / 2,
                    image->GetWidth(),
                    image->GetHeight());
      }
      else {
        image->Draw(draw,
                    x,
                    y+(height-image->GetHeight()) / 2,
                    image->GetWidth(),
                    image->GetHeight());

        divider->Draw(draw,
                      x+image->GetWidth()+
                      OS::display->GetSpaceHorizontal(OS::Display::spaceIntraObject),
                      this->y,
                      image->GetWidth(),
                      height);

      }
    }

    draw->activated=false;
    draw->focused=false;
    draw->disabled=false;
  }

  void Combo::DrawFocus()
  {
    if (editable && value!=NULL) {
      value->SetShowFocus(true);
    }

    Control::DrawFocus();
  }

  void Combo::HideFocus()
  {
    if (editable && value!=NULL) {
      value->SetShowFocus(false);
    }
    Control::HideFocus();
  }

  void Combo::Hide()
  {
    if (visible) {
      value->Hide();
      Control::Hide();
    }
  }

  void Combo::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (tableModel.Valid()) {
      if (model==prevAction && prevAction->IsFinished()) {
        if (selection->HasSelection()) {
          if (selection->GetLine()>1) {
            SetTableRow(selection->GetLine()-1);
          }
        }
        else if (tableModel->GetRows()>0) {
          SetTableRow(1);
        }
      }
      else if (model==nextAction && nextAction->IsFinished()) {
        if (selection->HasSelection()) {
          if (selection->GetLine()<tableModel->GetRows()) {
            SetTableRow(selection->GetLine()+1);
          }
        }
        else if (tableModel->GetRows()>0) {
          SetTableRow(1);
        }
      }
      else if (model==tableModel && visible) {
        Redraw();
      }
    }

    Control::Resync(model,msg);
  }

  TextCombo::TextCombo()
  : Combo(new Text(),false),
    model(NULL)
  {
    // no code
  }

  TextCombo::~TextCombo()
  {
    // no code
  }

  bool TextCombo::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::String*>(model);

    Combo::SetModel(this->model);

    return this->model.Valid();
  }

  /**
    This method will be called if a new value was selected from the popup
    window. The baseclass will try its best to assign a sensefull value to
    the model assigned to Combo. If the model is numemric it will
    assign the index of the current selcted (starting with 0), if its of type
    text, if will assign the text of the current selected (if the table model
    has more than one coloum if will iterate from left to right until some
    valid text will be returned).

    If you want some other (or more) behaviour, overwrite this method
    and call baseclass if necessary.
  */
  void TextCombo::CopySelection(size_t row)
  {
    if (tableModel.Valid()) {
      size_t       x=1;
      std::wstring res;

      while (res.empty() && x<=headerModel->GetColumns()) {
        res=tableModel->GetString(x,row);
        x++;
      }

      dynamic_cast<Text*>(value)->SetText(res);

      if (model.Valid()) {
        model->Set(res);
      }
    }

    if (tableModel.Valid() && tableModel->GetRows()>0) {
      if (row>0) {
        selection->SelectCell(1,row);
      }
      else {
        selection->Clear();
      }
    }
  }

  void TextCombo::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if (model==this->model) {
      dynamic_cast<Text*>(value)->SetText(this->model->Get());
    }

    Combo::Resync(model,msg);
  }


  IndexCombo::IndexCombo()
  : Combo(new Text(),false),
    model(NULL)
  {
    // no code
  }

  IndexCombo::~IndexCombo()
  {
    // no code
  }

  bool IndexCombo::SetModel(Base::Model* model)
  {
    this->model=dynamic_cast<Model::SizeT*>(model);

    Combo::SetModel(this->model);

    return this->model.Valid();
  }

  void IndexCombo::CopySelection(size_t row)
  {
    if (tableModel.Valid()) {
      size_t       x=1;
      std::wstring res;

      while (res.empty() && x<=headerModel->GetColumns()) {
        res=tableModel->GetString(x,row);
        x++;
      }

      dynamic_cast<Text*>(value)->SetText(res);

      if (model.Valid()) {
        model->Set(row);
      }
    }

    if (tableModel.Valid() && tableModel->GetRows()>0) {
      if (row>0) {
        selection->SelectCell(1,row);
      }
      else {
        selection->Clear();
      }
    }
  }

  void IndexCombo::Resync(Base::Model* model, const Base::ResyncMsg& msg)
  {
    if ((model==tableModel || model==this->model) &&
        this->model.Valid() && tableModel.Valid()) {
      size_t       x=1;
      std::wstring res;

      if (!this->model->IsNull()) {
        if (this->model->Get()<=tableModel->GetRows()) {
          selection->SelectCell(1,this->model->Get());

          while (res.empty() && x<=headerModel->GetColumns()) {
            res=tableModel->GetString(x,this->model->Get());
            x++;
          }

          dynamic_cast<Text*>(value)->SetText(res);
        }
        else {
          this->model->SetNull();
          dynamic_cast<Text*>(value)->SetText(L"");
        }
      }

    }
    else if (model==this->model) {
      Redraw();
    }

    Combo::Resync(model,msg);
  }

  IndexCombo* IndexCombo::Create(Lum::Model::SizeT* index,
                                 Lum::Model::Table* table,
                                 bool horizontalFlex, bool verticalFlex)
  {
    IndexCombo *c=new IndexCombo();

    c->SetFlex(horizontalFlex,verticalFlex);
    c->SetModel(index);
    c->SetTableModel(table);

    return c;
  }

  TextEditCombo::TextEditCombo()
  : Combo(new String,true),
    model(NULL)
  {
    string=dynamic_cast<String*>(value);
    string->SetFlex(true,true);
    string->SetCursorUpAction(prevAction);
    string->SetCursorDownAction(nextAction);
  }

  TextEditCombo::~TextEditCombo()
  {
    // no code
  }

  bool TextEditCombo::SetModel(Base::Model* model)
  {
    if (string->SetModel(model)) {
      this->model=dynamic_cast<Model::String*>(model);

      Combo::SetModel(this->model);

      return this->model.Valid();
    }
    else {
      return false;
    }
  }

  String* TextEditCombo::GetString() const
  {
    return string;
  }

  void TextEditCombo::CopySelection(size_t row)
  {
    if (tableModel.Valid()) {
      size_t       x;
      std::wstring res;

      x=1;
      while (res.empty() && x<=headerModel->GetColumns()) {
        res=tableModel->GetString(x,row);
        x++;
      }

      if (model.Valid()) {
        model->Set(res);
      }
    }

    if (tableModel.Valid() && tableModel->GetRows()>0) {
      if (row>0) {
        selection->SelectCell(1,row);
      }
      else {
        selection->Clear();
      }
    }
  }
}

