/****************************************************************************
**
** This file is part of the Hyves PhoneGap container.
** Copyright (C) 2010-2011 Hyves (Startphone Ltd.)
** http://www.hyves.nl
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "hyvesupload.h"

#include "qjson/json_driver.hh"

#include <QSize>
#include <QDebug>
#include <QFile>
#include <QBuffer>
#include <QFileInfo>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QImageReader>

const QSize MAX_IMAGE_SIZE_DEFAULT = QSize(700, 700);

static QByteArray uniqueBoundary(const QByteArray &data) {

    QByteArray boundary;

    do {
        boundary.clear();
        for (int i = 0; i < 6; i++) {
            int character = qrand() % 62;
            if (character < 10) {
                boundary += character + '0';
            } else if (character < 36) {
                boundary += (character - 10) + 'a';
            } else {
                boundary += (character - 36) + 'A';
            }
        }
    } while (data.contains(boundary));

    return boundary;
}

HyvesUpload::HyvesUpload(QNetworkAccessManager *nam, QObject *parent) :
    QObject(parent),
    m_nam(nam),
    m_maxImageSize(MAX_IMAGE_SIZE_DEFAULT) {
}

QByteArray HyvesUpload::loadImage(const QString &path) const {

    QByteArray data;
    QImageReader reader(path);
    QSize maxSize(m_maxImageSize);
    QSize imgSize(reader.size());

    if (imgSize.width() < imgSize.height()) {
        maxSize.transpose();
    }

    if (imgSize.width() > maxSize.width() || imgSize.height() > maxSize.height()) { // scale image if necessary
        QImage img;
        imgSize.scale(maxSize, Qt::KeepAspectRatio);
        reader.setScaledSize(imgSize);
        qDebug() << "scaled image down to " << imgSize.width() << "x" << imgSize.height() << " before uploading";

        if (reader.read(&img)) {
            QBuffer buffer(&data);
            buffer.open(QIODevice::WriteOnly);
            img.save(&buffer, "JPEG");
            buffer.close();
            return data; // return scaled image
        }
    }

    // down-scaling is not necessary (or image reader failed)
    qDebug() << "uploading original file...";
    QFile file(path);
    if (file.open(QIODevice::ReadOnly)) {
        data = file.readAll();
        file.close();
    }
    return data;
}

void HyvesUpload::uploadFile(const QString &ip, const QString &token,
                             const QString &path, const QVariantMap &properties) {

    qDebug() << "HyvesUpload::uploadFile: " << ip << "," << token << "," << path;

    QByteArray imageData(loadImage(path));
    if (imageData.isEmpty()) {
        qWarning("Could not open file to upload");
        emit uploadFailed();
        return;
    }

    QFileInfo info(path);
    QByteArray boundary = uniqueBoundary(imageData);

    QUrl url;
    url.setScheme("http");
    url.setHost(ip);
    url.setPath("upload");
    url.addQueryItem("token", token);

    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "multipart/form-data; boundary=" + boundary + "; charset=utf-8");
    request.setRawHeader("Host", ip.toLatin1());

    QByteArray postData;
    postData.reserve(imageData.size() + 200);

    postData  = "--" + boundary + "\r\n";
    postData += "Content-Disposition: form-data; name=\"file\"; filename=\"" + info.fileName() + "\"\r\n";
    postData += "Content-Type: image/jpeg\r\n\r\n";
    postData += imageData;

    if (properties.contains("title")) {
        postData += "\r\n--" + boundary + "\r\n";
        postData += "Content-Disposition: form-data; name=\"title\"\r\n";
        postData += "Content-Transfer-Encoding: 8bit\r\n";
        postData += "Content-Type: text/plain; charset=utf-8\r\n\r\n";
        postData += properties["title"].toString();
    }

    if (properties.contains("description")) {
        postData += "\r\n--" + boundary + "\r\n";
        postData += "Content-Disposition: form-data; name=\"description\"\r\n";
        postData += "Content-Transfer-Encoding: 8bit\r\n";
        postData += "Content-Type: text/plain; charset=utf-8\r\n\r\n";
        postData += properties["description"].toString();
    }

    if (properties.contains("latitude") && properties.contains("longitude")) {
        postData += "--" + boundary + "\r\n";
        postData += "Content-Disposition: form-data; name=\"geodata\"\r\n\r\n";
        postData += QString("%1,%2").arg(properties["latitude"].toFloat()).arg(properties["longitude"].toFloat());
    }

    postData += "\r\n--" + boundary + "--\r\n";

    request.setHeader(QNetworkRequest::ContentLengthHeader, postData.length());

    QNetworkReply *reply = m_nam->post(request, postData);
    connect(reply, SIGNAL(finished()), SLOT(handleUploadFinished()));
    connect(reply, SIGNAL(uploadProgress(qint64, qint64)), SLOT(handleUploadProgress(qint64, qint64)));
}

void HyvesUpload::handleUploadFinished() {

    QNetworkReply *reply = qobject_cast<QNetworkReply *>(sender());
    reply->deleteLater();
    if (reply->error()) {
        qDebug() << "Error while uploading file: " << reply->error();
        emit uploadFailed();
        return;
    }

    emit uploadProgress(100);

    requestStatusUpdate(reply->url().host(), reply->url().queryItemValue("token"));
}

void HyvesUpload::requestStatusUpdate(const QString &host, const QString &token) {

    QUrl url;
    url.setScheme("http");
    url.setHost(host);
    url.setPath("status");
    url.addQueryItem("token", token);

    QNetworkRequest request(url);

    QNetworkReply *statusReply = m_nam->get(request);
    connect(statusReply, SIGNAL(finished()), SLOT(handleStatusFinished()));
}

void HyvesUpload::handleStatusFinished() {

    QNetworkReply *statusReply = qobject_cast<QNetworkReply *>(sender());
    statusReply->deleteLater();
    if (statusReply->error()) {
        qDebug() << "Error while requesting status: " << statusReply->error();
        emit uploadFailed();
        return;
    }

    bool error;
    JSonDriver driver;
    QVariantMap response = driver.parse(statusReply->readAll(), &error).toMap();
    if (error) {
        qWarning("Could not parse JSON reply from media server");
        emit uploadFailed();
        return;
    }

    QString token = statusReply->url().queryItemValue("token");
    int errorNumber = response["data"].toMap()[token].toList()[0].toMap()["error"].toMap()["errornumber"].toInt();
    if (errorNumber != 0) {
        qDebug() << "Error while processing file: " << errorNumber;
        emit uploadFailed();
        return;
    }

    QString mediaId = response["data"].toMap()[token].toList()[0].toMap()["done"].toMap()["mediaid"].toString();
    if (mediaId.isEmpty()) {
        requestStatusUpdate(statusReply->url().host(), token);
        return;
    }

    emit uploadSucceeded(mediaId);
}

void HyvesUpload::handleUploadProgress(qint64 bytesSent, qint64 bytesTotal) {

    emit uploadProgress(100 * bytesSent / bytesTotal);
}

int HyvesUpload::maxImageWidth() const {

    return m_maxImageSize.width();
}

void HyvesUpload::setMaxImageWidth(int value) {

    m_maxImageSize.setWidth(value);
}

int HyvesUpload::maxImageHeight() const {

    return m_maxImageSize.height();
}

void HyvesUpload::setMaxImageHeight(int value) {

    m_maxImageSize.setHeight(value);
}

