/****************************************************************************
**
** This file is part of the Hyves PhoneGap container.
** Copyright (C) 2010-2011 Hyves (Startphone Ltd.)
** http://www.hyves.nl
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#ifndef PROFILING_H
#define PROFILING_H

#include <QMap>
#include <QTime>
#include <QVariantList>

class QNetworkReply;


class Profiling : public QObject {

    Q_OBJECT

    public:
        // singleton
        static Profiling *instance() {
            static Profiling instance;
            return &instance;
        }

        ~Profiling();

        /**
         * Starts profiling timer for a given key.
         * @param key - profiling key
         */
        void start(const QString &key);

        /**
         * Stops profiling timer for a given key.
         * @param key - profiling key
         */
        void stop(const QString &key);

        /**
         * Starts profiling a request. Only API calls
         * are profiled with a timer.
         */
        void startRequest(const QNetworkReply *reply);

        /**
         * Stops profiling a request.
         * Collects cache statistics for requests on static resources (images, js, css).
         */
        void stopRequest(const QNetworkReply *reply);

        /**
         * Returns a list of profiling marks to the javascript layer
         * A mark has the following properties:
         * {
         *   key: profiling key,
         *   action: start/stop,
         *   timestamp: unix timestamp in ms of start of profiling,
         *   elapsed: duration of profiling in ms
         * }
         */
        Q_INVOKABLE QVariantList getMarks() const;

        /**
         * Returns a tuple containing cache hits and cache misses
         */
        Q_INVOKABLE QVariantMap getCacheStats() const;

        /**
         * Enables profiling.
         */
        Q_INVOKABLE void enable();

        /**
         * Disables profiling.
         * (default is disabled)
         */
        Q_INVOKABLE void disable();

        /**
         * Clears profiling statistics.
         */
        Q_INVOKABLE void clear();

    private:
        Profiling();
        Q_DISABLE_COPY(Profiling); // singleton

        /**
         * @return whether the reply is an API call
         */
        bool isApiCall(const QNetworkReply *reply);

        /**
         * Register a cache hit or miss
         */
        void cacheHit(bool cacheHit);

        /**
         * @return whether profiling is enabled
         */
        bool isEnabled();

        QMap<QString, QTime> m_timerMap;
        QVariantList m_marks;

        int m_cacheHits;
        int m_cacheMisses;
        int m_isEnabled;
};

#endif // PROFILING_H
