/****************************************************************************
**
** This file is part of the Hyves PhoneGap container.
** Copyright (C) 2010-2011 Hyves (Startphone Ltd.)
** http://www.hyves.nl
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "thumbnailnetworkreply.h"
#include "thumbnailloader.h"

#include <QFile>
#include <QDebug>
#include <QImage>
#include <QBuffer>
#include <QMetaObject>

ThumbnailNetworkReply::ThumbnailNetworkReply(const QNetworkRequest &request, QObject *parent) :
    QNetworkReply(parent),
    m_loader(0),
    m_imgData() {

#ifdef Q_OS_SYMBIAN
    m_exifLoader = 0;
#endif

    setRequest(request);
    setOpenMode(QIODevice::ReadOnly);

    QString path = request.url().path();
    //qDebug() << "loading thumbnail for " << path;
    if (!QFile::exists(path)) {
        QMetaObject::invokeMethod(this, "failure", Qt::QueuedConnection);
        return;
    }

#ifdef Q_OS_SYMBIAN
    // grab exif preview on Symbian...
    m_exifLoader = new SymbianPicLoader(this);
    TRAPD(iErr, m_exifLoader->loadImage(path));
    if (iErr == KErrNone) {
        return;
    }
#endif
    // ... on other platforms (and if exif loader fails) generate thumb from full image
    m_loader = new ThumbnailLoader(this);
    connect(m_loader, SIGNAL(thumbLoaded(QByteArray)), this, SLOT(thumbLoaded(QByteArray)));
    m_loader->loadImage(path);
}

ThumbnailNetworkReply::~ThumbnailNetworkReply() {

#ifdef Q_OS_SYMBIAN
    delete m_exifLoader;
#endif
}

void ThumbnailNetworkReply::abort() {

#ifdef Q_OS_SYMBIAN
    delete m_exifLoader;
    m_exifLoader = 0;
#endif
    if (m_loader) {
        disconnect(m_loader, SIGNAL(thumbLoaded(QByteArray)), this, SLOT(thumbLoaded(QByteArray)));
    }
    emit error(QNetworkReply::OperationCanceledError);
    emit finished();
}

qint64 ThumbnailNetworkReply::bytesAvailable() const {

    return m_imgData.length() + QNetworkReply::bytesAvailable();
}

qint64 ThumbnailNetworkReply::readData(char *data, qint64 maxLength) {

    qint64 length = qMin(qint64(m_imgData.length()), maxLength);
    if (length) {
        qMemCopy(data, m_imgData.constData(), length);
        m_imgData.remove(0, length);
    }
    return length;
}

#ifdef Q_OS_SYMBIAN
void ThumbnailNetworkReply::thumbLoaded(QPixmap pic) {

    if (pic.isNull()) {
        // No exif preview available
        qDebug() << "no exif preview available";
        m_loader = new ThumbnailLoader(this);
        connect(m_loader, SIGNAL(thumbLoaded(QByteArray)), this, SLOT(thumbLoaded(QByteArray)));
        m_loader->loadImage(request().url().path());
        return;
    }

    QBuffer buffer(&m_imgData);
    buffer.open(QIODevice::WriteOnly);
    pic.toImage().save(&buffer, "JPEG");
    buffer.close();
    success();
}
#endif

void ThumbnailNetworkReply::thumbLoaded(const QByteArray& data) {

    m_imgData = data;
    delete m_loader;
    m_loader = 0;
    success();
}

void ThumbnailNetworkReply::failure() {

    qDebug() << "Failed to load thumbnail for file " << request().url().path();
    emit error(QNetworkReply::ContentNotFoundError);
    emit finished();
}

void ThumbnailNetworkReply::success() {

    setHeader(QNetworkRequest::ContentTypeHeader, "image/jpeg");
    setHeader(QNetworkRequest::ContentLengthHeader, QByteArray::number(m_imgData.length()));

    emit metaDataChanged();
    emit downloadProgress(m_imgData.length(), m_imgData.length());
    emit readyRead();
    emit finished();
}

