/*
 * Camera.h
 *
 *  Created on: Jan 15, 2010
 *      Author: nitobi-test
 */

#ifndef CAMERA_H
#define CAMERA_H

#include <QFileInfoList>
#include <QStringList>
#include <QPointer>
#include <QObject>
#include <QSize>

#ifdef Q_OS_SYMBIAN
    #include "camerafileobserver.h"
#endif

class QImage;

#ifdef Q_OS_SYMBIAN
class Camera: public QObject, MCameraFileObserver {
#else
class Camera: public QObject {
#endif

    Q_OBJECT

    public:
        explicit Camera(QObject *parent);
        virtual ~Camera();

        /**
         * Error codes passed with the error signal
         */
        enum ErrorCodes {
            ENoError = 0,
            ECameraObserverFailed,
            ECameraAppNotFound,
            ECameraAppFailed,
            ECaptureFileNotReadable,
            EVideoFileCaptured,
            ECanceledByUser,
            EUnknownError
        };

        /**
         * Defines the how the captured picture is returned
         */
        enum EDestinationType {
            EDestinationDataUrl = 0,
            EDestinationFileUri = 1,
            EDestinationDataUrlAndFileUri = 2,
            ENumDestinationTypes = 3
        };

        /**
         * Defines the source for the picture
         * Currently camera and local photo library (needs some js-code) are supported
         */
        enum EPictureSourceType {
            ECamera = 1,
            EPhotoLibrary = 0,
            ESavedPhotoAlbum = 2 // same as EPhotoLibrary for now
        };

        /**
         * @returns in which format a captured picture is returned to js-code
         */
        int destinationType() const;

        /**
         * Define in which format a captured picture is returned to js code
         * @param destination - valid values are defined in EDestinationType (C++) / camera.DestinationType (js)
         */
        void setDestinationType(int destination);

        Q_PROPERTY(int destinationType READ destinationType WRITE setDestinationType);

        /**
         * @returns the image compression quality for captured pictures
         * @note Only for base64 encoded format, has no effect for EDestinationFileUri
         */
        int quality() const;

        /**
         * Sets the image compression quality for captured pictures
         * @param quality 0 - 100
         */
        void setQuality(int quality);

        Q_PROPERTY(int quality READ quality WRITE setQuality);

        /**
         * @returns maximum width for the captured picture
         * @note If the picture taken by the camera is larger it will be scaled down (keeping it's aspect ratio)
         * but the picture will not be scaled up if it is smaller.
         * Only applies for base64 encoded format, ignored for EDestinationFileUri
         */
        int width() const;

        /**
         * Sets the maximum width for the captured picture
         * @param width
         */
        void setWidth(int width);

        Q_PROPERTY(int width READ width WRITE setWidth);

        /**
         * @returns maximum height for the captured picture
         * @note If the picture taken by the camera is larger it will be scaled down (keeping it's aspect ratio)
         * but the picture will not be scaled up if it is smaller.
         * Only applies for base64 encoded format, ignored for EDestinationFileUri
         */
        int height() const;

        /**
         * Sets the maximum height for the captured picture
         * @param height
         */
        void setHeight(int height);
        Q_PROPERTY(int height READ height WRITE setHeight);

        /**
         * @returns a list jpg/jpeg images stored on the device
         * @note the list is sorted by creation timestamp: newest file is head, oldest file at end
         */
        QStringList localImages();
        Q_PROPERTY(QStringList localImages READ localImages);

#ifdef Q_OS_SYMBIAN
        /**
         * @reimp callback for CCameraFileObserver
         */
        virtual void NewCameraFileL(const TFileName &aFileName);

    protected:
        /**
         * @reimp watches for app getting back focus from camera
         */
        virtual bool eventFilter(QObject *obj, QEvent *event);
#endif

    public slots:
        void takePicture(int pictureSource = ECamera);
#ifdef Q_OS_SYMBIAN
        void killCamera();
#endif
        /**
         * Callback for Hyves js-based photo picker
         * @param selectedImage selected item (image) from 'availableImages'
         * @see signal imagesPicker()
         */
        void submitImage(const QString& file);

    signals:
        void pictureCaptured(const QString &dataUrl, const QString &fileUri);
        void error(int errorCode, const QString &message);

        /**
         * This is a hook for our js-based photopicker
         * When emitted the gallery tile should be shown to the user
         */
        void imagePicker();

    private:
        QFileInfoList searchImages(const QString &dir = "") const;

#ifdef Q_OS_SYMBIAN
        CCameraFileObserver *m_observer;
        QPointer<QWidget> m_view;
#endif
        QStringList m_images;
        bool m_waitingForCamera;
        int m_destination;
        int m_quality;
        QSize m_size;
};

#endif // CAMERA_H
