/*
 * Camera.cpp
 *
 *  Created on: Jan 15, 2010
 *      Author: nitobi-test
 */

#include "camera.h"

#ifdef Q_OS_SYMBIAN
    #include <e32std.h>
#endif

#include <QApplication>
#include <QBuffer>
#include <QDebug>
#include <QFile>
#include <QImage>
#include <QImageReader>
#include <QProcess>
#include <QWidget>
#include <QEvent>
#include <QMetaObject>
#include <QMainWindow>
#include <QDesktopWidget>
#include <QRect>
#include <QDir>
#include <QDateTime>
#include <QFileInfo>
#include <QFileInfoList>
#include <QDesktopServices>


const QString CAMERA_APP_PATH = "Z:\\sys\\bin\\cameraapp.exe";

bool fileIsOlderThan(const QFileInfo &i1, const QFileInfo &i2) {

    return i1.created() > i2.created();
}

Camera::Camera(QObject *parent) :
    QObject(parent),
#ifdef Q_OS_SYMBIAN
    MCameraFileObserver(),
    m_observer(0),
    m_view(0),
#endif
    m_images(),
    m_waitingForCamera(false),
    m_destination(EDestinationDataUrl),
    m_quality(75),
    m_size() {

    QRect desktopGeometry(qApp->desktop()->availableGeometry());
    m_size.setHeight(qMin(desktopGeometry.height(), desktopGeometry.width()));
    m_size.setWidth(qMax(desktopGeometry.height(), desktopGeometry.width()));
    qDebug() << "available screen resolution: " << m_size.width() << "x" << m_size.height();

#ifdef Q_OS_SYMBIAN
    TRAPD(iError, m_observer = CCameraFileObserver::NewL(*this);)
    if (iError != KErrNone) {
        m_observer = 0;
        qDebug() << "Camera::Camera: failed to instantiate CCameraFileObserver: error " << iError;
        return;
    }

    qApp->installEventFilter(this);
    m_observer->Start();
#endif
}

Camera::~Camera() {

#ifdef Q_OS_SYMBIAN
    delete m_observer;
#endif
}

int Camera::destinationType() const {

    return m_destination;
}

void Camera::setDestinationType(int destination) {

    if (destination < EDestinationDataUrl || destination >= ENumDestinationTypes) {
        qDebug() << "Camera::setDestinationType: invalid value: " << destination;
        return;
    }

    m_destination = destination;
}

int Camera::quality() const {

    return m_quality;
}

void Camera::setQuality(int quality) {

    if (quality < 0 || quality > 100) {
        qDebug() << "Camera::setQuality: value out of range: " << quality;
        return;
    }

    m_quality = quality;
}

int Camera::width() const {

    return m_size.width();
}

void Camera::setWidth(int width) {

    m_size.setWidth(width);
}

int Camera::height() const {

    return m_size.height();
}

void Camera::setHeight(int height) {

    m_size.setHeight(height);
}

void Camera::takePicture(int pictureSource) {

    switch (pictureSource) {
    case EPhotoLibrary: // fall
    case ESavedPhotoAlbum:
        emit imagePicker();
        return;
    case ECamera:
        break;
    default:
        emit error(EUnknownError, "unknown value passed for picture source");
        return;
    }

#ifdef Q_OS_SYMBIAN
    if (!m_observer) {
        emit error(ECameraObserverFailed, "CameraFileObserver failed to register");
        return;
    }

    // allow Qt to free up some graphics memory for camera app (workaround for #MOB-664)
    QMainWindow *mainWindow = qobject_cast<QMainWindow*>(qApp->activeWindow());
    if (mainWindow && mainWindow->centralWidget()) {
        m_view = mainWindow->centralWidget();
        m_view->hide();
        qApp->processEvents(QEventLoop::AllEvents | QEventLoop::DeferredDeletion, 1000);
    }

    qDebug() << "Camera::takePicture: Starting camera app";
    if (!QProcess::startDetached(CAMERA_APP_PATH)) {
        QFile::exists(CAMERA_APP_PATH) ?
            emit error(ECameraAppFailed, "Camera application failed to start") :
            emit error(ECameraAppNotFound, "Could not find camera application");
        return;
    }
    m_waitingForCamera = true;
#else
    // no camera for platforms other than Symbian yet - only local image picker is supported
    emit error(ECameraAppNotFound, "Not implemented on this platform (yet)");
#endif
}

#ifdef Q_OS_SYMBIAN
void Camera::killCamera() {

    m_waitingForCamera = false;
    TRAPD(iError,
        TFindProcess processFinder(_L("*"));
        TFullName result;
        RProcess processHandle;
        while (processFinder.Next(result) == KErrNone) {
            User::LeaveIfError(processHandle.Open(processFinder, EOwnerThread));
            TFileName fileName = processHandle.FileName();
            if (QString((QChar*) fileName.Ptr(), fileName.Length()) == CAMERA_APP_PATH) {
                qDebug() << "Camera::killCamera: killing process " << QString((QChar*) fileName.Ptr(), fileName.Length());
                processHandle.Kill(KErrNone);
            }
            processHandle.Close();
        })
    if (iError != KErrNone) {
        qDebug() << "Camera::killCamera: failed iError=" << iError;
    }

    if (m_view) {
        m_view->show();
    }
}

void Camera::NewCameraFileL(const TFileName& aFileName) {

    QString filename((QChar*) aFileName.Ptr(), aFileName.Length());
    if (filename.endsWith(".avi")) {
        qDebug() << "Camera::NewCameraFileL: video captured: " << filename << " (can handle images only)";
        emit error(EVideoFileCaptured, "A video was captured (can handle images only)");
        return;
    }
    qDebug() << "Camera::NewCameraFileL: file captured: " << filename;

    // keep list of local images up to date
    if (!m_images.isEmpty()) {
        m_images.insert(0, filename);
    }

    if (m_waitingForCamera) {
        submitImage(filename);
        QMetaObject::invokeMethod(this, "killCamera", Qt::QueuedConnection);
    }
}

bool Camera::eventFilter(QObject *obj, QEvent *event) {

    Q_UNUSED(obj);
    if (event->type() == QEvent::ApplicationActivate && m_waitingForCamera) {
        qDebug() << "App activated while waiting for image, assume canceled by user";
        m_waitingForCamera = false;
        emit error(ECanceledByUser, "No picture available: canceled by user");

        if (m_view) {
            m_view->show();
        }
    }
    return false;
}
#endif

void Camera::submitImage(const QString& file) {

    if (m_destination == EDestinationFileUri) {
        emit pictureCaptured("", file);
        return;
    }

    // read picture, resize and send to webkit as base64-encoded string
    QImageReader reader(file);
    QSize maxSize(m_size);
    QSize imgSize(reader.size());
    if (imgSize.width() < imgSize.height()) { // landscape/portrait format?
        maxSize.transpose();
    }
    if (imgSize.width() > maxSize.width() || imgSize.height() > maxSize.height()) { // scale image if necessary
        qDebug() << "original image size: " << imgSize.width() << "x" << imgSize.height();
        imgSize.scale(maxSize, Qt::KeepAspectRatio);
        reader.setScaledSize(imgSize);
        qDebug() << "scaled image size: " << imgSize.width() << "x" << imgSize.height();
    }

    QImage img;
    if (!reader.read(&img)) {
        qDebug() << "Camera::NewCameraFileL: Failed to read captured file: " << reader.errorString() << " (" << reader.error() << ")";
        emit error(ECaptureFileNotReadable, "Failed to open/read captured file: " + reader.errorString());
        return;
    }

    QByteArray data;
    QBuffer buffer(&data);
    buffer.open(QIODevice::WriteOnly);
    img.save(&buffer, "JPG", m_quality);
    buffer.close();
    emit pictureCaptured(data.toBase64(),
                         m_destination == EDestinationDataUrlAndFileUri ? file : "");
}

QStringList Camera::localImages() {

    if (m_images.isEmpty()) {
        // search default picture location
        QFileInfoList images = searchImages();
#ifdef Q_OS_SYMBIAN
        // Search from E drive
        images << searchImages("E:/images");
#elif defined Q_WS_MAEMO_5
        // Camera pictures
        images << searchImages("/home/user/MyDocs/DCIM");
#endif
        // sort list (newest/youngest files first)
        qSort(images.begin(), images.end(), fileIsOlderThan);
        foreach(const QFileInfo &info, images) {
            m_images << info.filePath();
        }
    }
    return m_images;
}

QFileInfoList Camera::searchImages(const QString &dir) const {

    QDir searchDir;
    searchDir.setFilter(QDir::Files | QDir::Dirs);
    searchDir.setPath(dir.isEmpty() ? QDesktopServices::storageLocation(QDesktopServices::PicturesLocation) : dir);
    //qDebug() << "searching in directory: " << searchDir.path();

    QFileInfoList result;
    QFileInfoList currentDir = searchDir.entryInfoList();
    for (int i = 0; i < currentDir.size(); ++i) {
        QFileInfo fileInfo = currentDir.at(i);
        if (fileInfo.isFile()) {
            if (fileInfo.suffix().toLower() == "jpg" || fileInfo.suffix().toLower() == "jpeg") {
                //qDebug() << "found image: " << fileInfo.filePath() << "created: " << fileInfo.created().toString();
                result << fileInfo;
            }
            continue;
        }
        // traverse sub directories
        if (!fileInfo.fileName().startsWith('.')) {
            result << searchImages(fileInfo.filePath());
        }
    }
    return result;
}

