#include "horizontalcall.h"
#include "ui_horizontalcall.h"

using namespace QtMobility;

horizontalcall::horizontalcall(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::horizontalcall)
{
    m_Maximized = true;
    m_Landscape = true;
    m_FetchPending = true;
    m_DisplayingContact = false;
    createStyle();
    ui->setupUi(this);
    m_VisibleItem = 0;
    m_LastVisibleItemRow = 0;
    m_CurrentRow = 0;
    m_FetchRequest = new QContactFetchRequest();
    numberOfLettersPerRow = 6;
    m_Contacts = QList<QContact>();
    m_NameStrings = QList<QStringList>();
    existingLetterButtons = new QList<QPushButton*>();
    existingLetters = new QList<QString>();
    m_CurrentFilter = QString();
    createSharedUI();
    //createPortraitUI();
    getRotationSettings();
    if (m_Landscape) {
        qDebug() << "Init : Autorotation is set to Landscape";
        //setAttribute(Qt::WA_Maemo5PortraitOrientation, false);
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
    } else {
        qDebug() << "Init : Autorotation is set to Portrait";
        //setAttribute(Qt::WA_Maemo5LandscapeOrientation, false);
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
    }
    if (m_AutorotationEnabled)
    {
        qDebug() << "Init : Autorotation is set to automatic";
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    }

    if (m_Landscape)
        createLandscapeUI();
    else
        createPortraitUI();

    m_scrollArea = NULL;
    QTimer::singleShot(1000, this, SLOT(populateList()));
    labelSelector->setText("Loading...");

}

void horizontalcall::getRotationSettings()
{
    QSettings settings("horizontal-call");
    m_AutorotationEnabled = settings.value("AutoRotation", true).value<bool>();
    qDebug() << "AutoRotation : " << m_AutorotationEnabled;
    m_Landscape = settings.value("ManualRotationLandscape", true).value<bool>();
    qDebug() << "ManualRotation : " << m_Landscape;
}

void horizontalcall::doAction1()
{
    //Autorotation
    if (m_AutorotationEnabled) {
        QMaemo5InformationBox::information(this, "AutoRotation is now <b>Disabled</b>", QMaemo5InformationBox::DefaultTimeout);
        m_AutorotationEnabled = false;
        QSettings settings("horizontal-call");
        settings.setValue("AutoRotation", false);
        setAttribute(Qt::WA_Maemo5AutoOrientation, false);
        if (m_Landscape) {
            //setAttribute(Qt::WA_Maemo5PortraitOrientation, false);
            setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
        } else {
            //setAttribute(Qt::WA_Maemo5LandscapeOrientation, false);
            setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
        }
    } else {
        QMaemo5InformationBox::information(this, "AutoRotation is now <b>Enabled</b>", QMaemo5InformationBox::DefaultTimeout);
        m_AutorotationEnabled = true;
        QSettings settings("horizontal-call");
        settings.setValue("AutoRotation", true);
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    }
}

void horizontalcall::doAction2()
{
    //Manualrotation
    if (m_AutorotationEnabled) {
        m_AutorotationEnabled = false;
        QSettings settings("horizontal-call");
        settings.setValue("AutoRotation", false);
    }
    if (m_Landscape) {
        qDebug() << "Switching from landscape to portrait, manual.";
        QSettings settings("horizontal-call");
        settings.setValue("ManualRotationLandscape", false);
        QMaemo5InformationBox::information(this, "Perspective is now : <b>Portrait</b>", QMaemo5InformationBox::DefaultTimeout);
        //rotator->setCurrentBehavior(QMaemo5Rotator::PortraitBehavior);
        //setAttribute(Qt::WA_Maemo5LandscapeOrientation, false);
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
    } else {
        qDebug() << "Switching from portrait to landscape, manual.";
        QSettings settings("horizontal-call");
        settings.setValue("ManualRotationLandscape", true);
        QMaemo5InformationBox::information(this, "Perspective is now : <b>Landscape</b>", QMaemo5InformationBox::DefaultTimeout);
        //rotator->setCurrentBehavior(QMaemo5Rotator::LandscapeBehavior);
        //setAttribute(Qt::WA_Maemo5PortraitOrientation, false);
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
    }
}

void horizontalcall::createSharedUI()
{
    //Set mainLayout.
    this->setWindowTitle("Horizontal-Call");

    //Create menu
    action1 = new QAction(tr("&AutoRotation"), this);
    action2 = new QAction(tr("&ManualRotation"), this);

    menu = menuBar()->addMenu(tr("Menu"));
    menu->addAction(action1);
    menu->addAction(action2);

    connect(action1, SIGNAL(triggered()), this , SLOT(doAction1()));
    connect(action2, SIGNAL(triggered()), this , SLOT(doAction2()));
}

void horizontalcall::createPortraitUI()
{
    qDebug("Creating Portrait UI");
    //Create Main Layout
    mainLayoutPortrait = new QVBoxLayout();
    mainLayoutPortrait->setSpacing(0);
    mainLayoutPortrait->setMargin(0);
    centralWidget()->setLayout(mainLayoutPortrait);

    //Take care of top middle and bottom
    //Top
    topWidget = new QWidget(centralWidget());
    topWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    mainLayoutPortrait->addWidget(topWidget);
    topLayout = new QHBoxLayout();
    topLayout->setSpacing(0);
    topLayout->setMargin(0);
    topWidget->setLayout(topLayout);
    //Middle
    middleWidget = new QWidget(centralWidget());
    middleWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    middleWidget->setFixedHeight(70);
    mainLayoutPortrait->addWidget(middleWidget);
    middleLayout = new QHBoxLayout();
    middleLayout->setSpacing(0);
    middleLayout->setMargin(0);
    middleWidget->setLayout(middleLayout);
    //Bottom
    lettersGridLayoutWidget = new QWidget(centralWidget());
    lettersGridLayoutWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    lettersGridLayoutWidget->setFixedHeight(400);
    mainLayoutPortrait->addWidget(lettersGridLayoutWidget);
    lettersGridLayout = new QGridLayout();
    lettersGridLayout->setSpacing(0);
    lettersGridLayout->setMargin(0);
    lettersGridLayoutWidget->setLayout(lettersGridLayout);

    //List Widget filtered in top right
    listWidgetFiltered = new QListWidget(topWidget);
    connect(listWidgetFiltered, SIGNAL(clicked(QModelIndex)), this, SLOT(itemClicked(QModelIndex)));
    listWidgetFiltered->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    listWidgetFiltered->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    listWidgetFiltered->setStyleSheet(m_StyleQListWidget);
    topLayout->addWidget(listWidgetFiltered);

    //Take care of window controls in top left
    windowControlsWidget = new QWidget(topWidget);
    windowControlsWidget->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);
    windowControlsWidget->setFixedWidth(200);
    topLayout->addWidget(windowControlsWidget);
    windowControlsLayoutPortrait = new QVBoxLayout();
    windowControlsLayoutPortrait->setSpacing(0);
    windowControlsLayoutPortrait->setMargin(0);
    windowControlsWidget->setLayout(windowControlsLayoutPortrait);
    //Exit
    pushButtonExit     = new QPushButton(topWidget);
    pushButtonExit->setStyleSheet(m_StylePhoneNumbersShort);
    connect(pushButtonExit, SIGNAL(clicked()), this, SLOT(exitButtonClicked()));
    pushButtonExit->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    pushButtonExit->setText("Exit");
    windowControlsLayoutPortrait->addWidget(pushButtonExit);
    //Minimize
    pushButtonMinimize = new QPushButton(topWidget);
    pushButtonMinimize->setStyleSheet(m_StylePhoneNumbersShort);
    connect(pushButtonMinimize, SIGNAL(clicked()), this, SLOT(minimizeButtonClicked()));
    pushButtonMinimize->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    pushButtonMinimize->setText("Toggle");
    windowControlsLayoutPortrait->addWidget(pushButtonMinimize);
    //Label
    labelSelector = new QLabel(topWidget);
    labelSelector->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    labelSelector->setAlignment(Qt::AlignCenter);
    windowControlsLayoutPortrait->addWidget(labelSelector);

    //Take care of filter controls middle widget
    //LineEdit
    lineEditFilter  = new QLineEdit(middleWidget);
    connect(lineEditFilter, SIGNAL(textChanged(QString)),this, SLOT(lineEditTextChanged(QString)));
    lineEditFilter->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    //Clear Button
    pushButtonClear = new QPushButton(middleWidget);
    connect(pushButtonClear, SIGNAL(clicked()), this, SLOT(clearButtonClicked()));
    pushButtonClear->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    pushButtonClear->setText("<");
    //Reset Button
    pushButtonReset = new QPushButton(middleWidget);
    connect(pushButtonReset, SIGNAL(clicked()), this, SLOT(resetButtonClicked()));
    pushButtonReset->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    pushButtonReset->setText("Reset");
    //Add to layout
    middleLayout->addWidget(lineEditFilter);
    middleLayout->addWidget(pushButtonClear);
    middleLayout->addWidget(pushButtonReset);

    ui->centralWidget->show();
    init();
}

void horizontalcall::createLandscapeUI()
{
    qDebug("Creating Landscape UI");
    //Create Main Layout
    mainLayout = new QHBoxLayout();
    mainLayout->setSpacing(0);
    mainLayout->setMargin(0);
    centralWidget()->setLayout(mainLayout);

    //Take care of left side and right side
    leftWidget = new QWidget(centralWidget());
    leftWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    mainLayout->addWidget(leftWidget);
    leftLayout = new QVBoxLayout();
    leftLayout->setSpacing(0);
    leftLayout->setMargin(0);
    //leftLayout->setSizeConstraint(QLayout::SetMaximumSize);
    leftWidget->setLayout(leftLayout);
    rightWidget = new QWidget(centralWidget());
    rightWidget->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding);
    rightWidget->setFixedWidth(240);
    mainLayout->addWidget(rightWidget);
    rightLayout = new QVBoxLayout();
    rightLayout->setSpacing(0);
    rightLayout->setMargin(0);
    rightWidget->setLayout(rightLayout);

    //Take care of filter controls in left side
    filterControlsWidget = new QWidget(leftWidget);
    filterControlsWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    filterControlsWidget->setFixedHeight(70);
    leftLayout->addWidget(filterControlsWidget);
    filterControlsLayout = new QHBoxLayout();
    filterControlsLayout->setSpacing(0);
    filterControlsLayout->setMargin(0);
    filterControlsWidget->setLayout(filterControlsLayout);
    lineEditFilter  = new QLineEdit(leftWidget);
    connect(lineEditFilter, SIGNAL(textChanged(QString)),this, SLOT(lineEditTextChanged(QString)));
    lineEditFilter->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    pushButtonClear = new QPushButton(leftWidget);
    connect(pushButtonClear, SIGNAL(clicked()), this, SLOT(clearButtonClicked()));
    pushButtonClear->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    pushButtonClear->setText("<");
    pushButtonReset = new QPushButton(leftWidget);
    connect(pushButtonReset, SIGNAL(clicked()), this, SLOT(resetButtonClicked()));
    pushButtonReset->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    pushButtonReset->setText("Reset");
    filterControlsLayout->addWidget(lineEditFilter);
    filterControlsLayout->addWidget(pushButtonClear);
    filterControlsLayout->addWidget(pushButtonReset);

    //Take care of grid layout for letters / details
    lettersGridLayoutWidget = new QWidget(leftWidget);
    lettersGridLayoutWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    leftLayout->addWidget(lettersGridLayoutWidget);
    lettersGridLayout = new QGridLayout();
    lettersGridLayout->setSpacing(0);
    lettersGridLayout->setMargin(0);
    lettersGridLayoutWidget->setLayout(lettersGridLayout);

    //Take care of window controls
    windowControlsWidget = new QWidget(rightWidget);
    windowControlsWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    windowControlsWidget->setFixedHeight(70);
    rightLayout->addWidget(windowControlsWidget);
    windowControlsLayout = new QHBoxLayout();
    windowControlsLayout->setSpacing(0);
    windowControlsLayout->setMargin(0);
    windowControlsWidget->setLayout(windowControlsLayout);
    rightWidget->setLayout(windowControlsLayout);
    pushButtonMinimize = new QPushButton(rightWidget);
    connect(pushButtonMinimize, SIGNAL(clicked()), this, SLOT(minimizeButtonClicked()));
    pushButtonMinimize->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    pushButtonMinimize->setText("Toggle");
    pushButtonExit     = new QPushButton(rightWidget);
    connect(pushButtonExit, SIGNAL(clicked()), this, SLOT(exitButtonClicked()));
    pushButtonExit->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    pushButtonExit->setText("Exit");
    windowControlsLayout->addWidget(pushButtonMinimize);
    windowControlsLayout->addWidget(pushButtonExit);

    //Label
    labelSelector = new QLabel(rightWidget);
    labelSelector->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    labelSelector->setFixedHeight(50);
    labelSelector->setAlignment(Qt::AlignCenter);
    labelSelector->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    rightLayout->addWidget(labelSelector);

    //List Widget filtered
    listWidgetFiltered = new QListWidget(rightWidget);
    connect(listWidgetFiltered, SIGNAL(clicked(QModelIndex)), this, SLOT(itemClicked(QModelIndex)));
    listWidgetFiltered->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    listWidgetFiltered->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    listWidgetFiltered->setStyleSheet(m_StyleQListWidget);
    rightLayout->addWidget(listWidgetFiltered);

    ui->centralWidget->show();
    init();
}

void horizontalcall::resizeEvent(QResizeEvent *e)
{
    qDebug("Resize Event");
    if (m_FetchPending) {
        qDebug("Still fetching contacts, Ignoring rotation.");
        return;
    }
    if (width() > height())
    {
        qDebug("Landscape Mode Recognized");
        if (m_Landscape)
        {
            qDebug("Already In Landscape Mode");
            return;
        } else {
            m_Landscape = true;
            destroyPortraitUI();
            createLandscapeUI();
        }
    } else {
        qDebug("Portratin Mode Recognized");
        if (!m_Landscape) {
            qDebug("Already In Portrait Mode");
            return;
        } else {
            m_Landscape = false;
            destroyLandscapeUI();
            createPortraitUI();
        }
    }
    fillListWidget();

    if (m_DisplayingContact) {
        listWidgetFiltered->setCurrentRow(m_CurrentRow);
        lineEditFilter->setText(m_CurrentFilter);
        showContactDetails(m_CurrentRow);
        m_LabelSelector = QString::number(m_CurrentRow) + " / " + QString::number(m_Contacts.length());
        labelSelector->setText(m_LabelSelector);

    } else {
        lineEditFilter->setText(m_CurrentFilter);
        qDebug("Calling lineEdit Text Changed After UI Change");
        lineEditTextChanged(m_CurrentFilter);
    }
    labelSelector->setText(m_LabelSelector);
}

void horizontalcall::fillListWidget()
{
    for (int i = 0; i < m_Contacts.size(); i++)
    {
        QContact temp = m_Contacts.at(i);
        QContactDisplayLabel displayLabel = temp.detail(QContactDisplayLabel::DefinitionName);

        if (!isAllEnglish(displayLabel.label()))
        {
            m_Contacts.removeAt(i);
            i--;
            continue;

        }

        if (!doesPhoneNumberExist(&temp))
        {
            m_Contacts.removeAt(i);
            i--;
            continue;
        }

        listWidgetFiltered->addItem(displayLabel.label());
        QStringList tempQStringList = displayLabel.label().split(" ", QString::SkipEmptyParts);
        tempQStringList.append(displayLabel.label());
        m_NameStrings.append(tempQStringList);
        //qDebug() << "NameStrings Size : " << m_NameStrings.size();
    }
}

void horizontalcall::init()
{
    existingLetterButtons->clear();
    existingLetters->clear();
    m_NameStrings.clear();
    row = col = currentLetter = 0;
}

void horizontalcall::destroyLandscapeUI()
{
    qDebug("Destroying Landscape UI");
    leftWidget->close();
    rightWidget->close();
    delete mainLayout;
}

void horizontalcall::destroyPortraitUI()
{
    qDebug("Destroying Portrait UI");
    topWidget->close();
    middleWidget->close();
    lettersGridLayoutWidget->close();
    delete mainLayoutPortrait;
}

void horizontalcall::minimizeButtonClicked()
{
    if (m_Maximized)
    {
        this->showNormal();
        m_Maximized = false;
        //this->setGeometry(0,0,800,390);
    }
    else
    {
        this->showFullScreen();
        m_Maximized = true;
        //this->setGeometry(0,0,800,480);
    }
}

void horizontalcall::createStyle()
{
    m_StyleLetters =
            "QPushButton {\
                min-height: 1.5em;\
                font: 1em;\
                margin: 0 1px 0 1px;\
                color: white;\
                background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1, stop: 0 #2198c0, stop: 1 #0d5ca6);\
                border-style: outset;\
                border-radius: 3px;\
                border-width: 1px;\
                border-color: #0c457e;\
                font-size: 40px;\
            }\
            QPushButton:pressed {\
                background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1, stop: 0 #0d5ca6, stop: 1 #2198c0);\
    }";
    getExternalStyleSheet(&m_StyleLetters, "/opt/horizontal-call/styleSheets/styleSheetLetters.qss");
    m_StylePhoneNumbers =
            "QPushButton {\
                min-height: 1.5em;\
                font: 1em;\
                margin: 0 1px 0 1px;\
                color: white;\
                background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1, stop: 0 #2198c0, stop: 1 #0d5ca6);\
                border-style: outset;\
                border-radius: 3px;\
                border-width: 1px;\
                border-color: #0c457e;\
                font-size: 80px;\
            }\
            QPushButton:pressed {\
                background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1, stop: 0 #0d5ca6, stop: 1 #2198c0);\
    }";
    getExternalStyleSheet(&m_StylePhoneNumbers, "/opt/horizontal-call/styleSheets/styleSheetPhoneNumbers.qss");
    m_StylePhoneNumbersShort =
            "QPushButton {\
                min-height: 1.5em;\
                font: 1em;\
                margin: 0 1px 0 1px;\
                color: white;\
                background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1, stop: 0 #2198c0, stop: 1 #0d5ca6);\
                border-style: outset;\
                border-radius: 3px;\
                border-width: 1px;\
                border-color: #0c457e;\
                font-size: 60px;\
            }\
            QPushButton:pressed {\
                background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1, stop: 0 #0d5ca6, stop: 1 #2198c0);\
    }";
    getExternalStyleSheet(&m_StylePhoneNumbersShort, "/opt/horizontal-call/styleSheets/styleSheetPhoneNumbersShort.qss");
    m_StyleQListWidget =
            "QListWidget {\
                font-size: 30px;\
    }";
    getExternalStyleSheet(&m_StyleQListWidget, "/opt/horizontal-call/styleSheets/styleSheetQListWidget.qss");
}

void horizontalcall::clearButtonClicked()
{
    //this->setUpdatesEnabled(false);
    if (m_scrollArea != NULL) {
        m_scrollArea->close();
        m_scrollArea = NULL;
        lineEditFilter->end(false);
        lineEditFilter->cursorBackward(true);
        lineEditFilter->del();
        lineEditTextChanged(lineEditFilter->text());
    } else {
        lineEditFilter->end(false);
        lineEditFilter->cursorBackward(true);
        lineEditFilter->del();
        lineEditTextChanged(lineEditFilter->text());
    }
    //this->setUpdatesEnabled(true);
}

void horizontalcall::lineEditTextChanged(QString text)
{

    //this->setUpdatesEnabled(false);
    m_CurrentFilter = text;
    m_DisplayingContact = false;
    pushButtonClear->setText("<");
    if (m_scrollArea != NULL) {
        m_scrollArea->close();
        m_scrollArea = NULL;
    }
    m_VisibleItem = listWidgetFiltered->count();
    if (text == "") {
        for (int i = 0; i < listWidgetFiltered->count(); i++)
        {
            listWidgetFiltered->item(i)->setHidden(false);
        }
    } else {
        for (int i = 0; i < m_NameStrings.count(); i++)
        {
            bool foundMatch = false;

            for (int k = 0; k < m_NameStrings.at(i).count(); k++)
            {
                if (((QString)m_NameStrings.at(i).at(k)).toUpper().startsWith(lineEditFilter->text().toUpper()))
                {
                    foundMatch = true;
                }
            }

            if (!foundMatch)
            {
                listWidgetFiltered->item(i)->setHidden(true);
                m_VisibleItem--;
            } else {
                listWidgetFiltered->item(i)->setHidden(false);
                m_LastVisibleItemRow = i;
            }
        }
    }

    deleteLetters();
    if (m_VisibleItem == 1)
    {
        m_DisplayingContact = true;
        m_CurrentRow = m_LastVisibleItemRow;
        showContactDetails(m_LastVisibleItemRow);
        labelSelector->setText(QString::number(m_LastVisibleItemRow) + " / " + QString::number(m_Contacts.length()));
        listWidgetFiltered->setCurrentRow(m_LastVisibleItemRow);
    } else {
        scanContacts();
        createLetters();
    }
    //this->setUpdatesEnabled(true);
}

void horizontalcall::createLetters()
{
    qDebug() << "Size : " << existingLetters->size();
    for (int i = 0; i < existingLetters->size(); i++)
    {
        CreateLetter(existingLetters->at(i));
    }
}

void horizontalcall::deleteLetters()
{
    for (int i = 0; i < existingLetterButtons->size(); i++)
    {
        QPushButton *tempButton = existingLetterButtons->at(i);
        delete tempButton;
    }
    existingLetterButtons->clear();
    row = col = 0;
}

void horizontalcall::populateList()
{
    lineEditFilter->insert("Populating List...");
    getContactsFromMemory();
}

void horizontalcall::exitButtonClicked()
{
    exit(0);
}

void horizontalcall::scanContacts()
{
    //qDebug() << "Scanning Contacts - Namestrings size is : " << m_NameStrings.size();
    existingLetters->clear();
    for (int i = 0; i < (m_NameStrings.count()); i++)
    {

        for (int j = 0; j < m_NameStrings.at(i).count(); j++)
        {
            if (m_NameStrings.at(i).at(j).length() <= lineEditFilter->text().length())
            {
                continue;
            }
            QString temp = m_NameStrings.at(i).at(j).at(lineEditFilter->text().size());

            if (!existingLetters->contains(temp.toUpper()) &&
                !listWidgetFiltered->item(i)->isHidden() &&
                temp.at(0) != '\0' &&
                ((QString)m_NameStrings.at(i).at(j)).toUpper().startsWith(lineEditFilter->text().toUpper()))
            {
                //qDebug() << "Adding : '" << temp << "' to existing letters";
                existingLetters->append(temp.toUpper());
            }
        }
    }
    qSort(*existingLetters);
}

/*
bool caseInsensitiveLessThan(const QString &s1, const QString &s2)
{
    return s1.toLower() < s2.toLower();
}
*/

void horizontalcall::makeDbusGSMCall(QString number)
{
    QDBusMessage msg = QDBusMessage::createMethodCall(
        "com.nokia.csd.Call", // --dest
        "/com/nokia/csd/call", // destination object path
        "com.nokia.csd.Call", // message name (w/o method)
        "CreateWith" // method
    );
    msg << number;
    msg << 0;
    msg = QDBusConnection::systemBus().call(msg);
}

void horizontalcall::populateListWidget()
{
    getContactsFromMemory();
}

bool horizontalcall::doesPhoneNumberExist(QContact *i_Contact)
{
    bool foundPhoneNumber = false;
    QList<QContactDetail> details = i_Contact->details();
    for (int z=0;z < details.size(); z++)
    {
        QString temp = details.at(z).definitionName();
        if (temp == "PhoneNumber") {
            foundPhoneNumber = true;
        }
    }

    return foundPhoneNumber;
}

void horizontalcall::showContactDetails(int row)
{
    //this->setUpdatesEnabled(false);
    qDebug() << "Printing contact in row : " << row;
    pushButtonClear->setText("Back");
    deleteLetters();

    if (m_scrollArea != NULL) {
        m_scrollArea->close();
        m_scrollArea = NULL;
    }

    m_scrollArea = new QScrollArea();
    m_scrollArea->setAttribute(Qt::WA_DeleteOnClose);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    lettersGridLayout->addWidget(m_scrollArea, 0, 0);
    m_scrollArea->show();


    QWidget *myWidget = new QWidget(m_scrollArea);
    myWidget->setLayout(new QVBoxLayout());

    m_scrollArea->setWidget(myWidget);
    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->widget()->layout()->setContentsMargins(0,0,0,0);
    m_scrollArea->widget()->layout()->setSpacing(0);
    myWidget->show();


    QPushButton *aButton = new QPushButton(listWidgetFiltered->item(row)->text(), myWidget);
    aButton->setStyleSheet("font-size: 43px");
    aButton->setMaximumSize(10000,70);
    m_scrollArea->widget()->layout()->addWidget(aButton);
    aButton->show();


    QContact a = m_Contacts.at(row);
    qDebug(a.displayLabel().toAscii());

    QList<QContactPhoneNumber> allDetails = a.details<QContactPhoneNumber>();

    for (int i = 0; i < allDetails.size(); i++) {

        QPushButton *aButton = new QPushButton(allDetails.at(i).number(), myWidget);
        aButton->sizePolicy().setVerticalPolicy(QSizePolicy::Expanding);
        if (aButton->text().length() > 11) {
            aButton->setStyleSheet(m_StylePhoneNumbersShort);
        } else {
            aButton->setStyleSheet(m_StylePhoneNumbers);
        }
        aButton->setMaximumSize(10000,10000);
        m_scrollArea->widget()->layout()->addWidget(aButton);
        aButton->show();
        connect(aButton, SIGNAL(clicked()),this, SLOT(phoneNumberClicked()));

    }

    QContact temp = m_Contacts.at(row);
    QString myString = temp.detail(QContactPhoneNumber::DefinitionName).value(QContactPhoneNumber::FieldNumber);

    //pushButtonDial->setText(myString);
    //this->setUpdatesEnabled(true);
    m_DisplayingContact = true;
}

void horizontalcall::phoneNumberClicked()
{
    QPushButton *tempButton = qobject_cast<QPushButton *>(QObject::sender());
    m_CallButtons.append(tempButton);
    QString temp = tempButton->text();
    m_PhoneNumbers.append(new QString(tempButton->text()));
    tempButton->setText("Dialing...");
    QTimer::singleShot(1000, this, SLOT(enableButtons()));
    tempButton->setEnabled(false);
    makeDbusGSMCall(temp);
}

void horizontalcall::enableButtons()
{
    m_CallButtons.at(0)->setEnabled(true);
    m_CallButtons.at(0)->setText(*m_PhoneNumbers.at(0));
    m_CallButtons.removeFirst();
    m_PhoneNumbers.removeFirst();
}

void horizontalcall::itemClicked(QModelIndex model)
{
    //qDebug("Row Clicked : " + QString::number(model.row()).toAscii());
    showContactDetails(model.row());
    labelSelector->setText(QString::number(model.row()) + " / " + QString::number(m_Contacts.length()));
    m_CurrentRow = model.row();
}

void horizontalcall::getContactsFromMemory()
{
    QContactManager *pManager = new QContactManager();
    m_FetchRequest->setManager(pManager);
    QContactSortOrder nameOrder;
    nameOrder.setDetailDefinitionName(QContactName::DefinitionName, QContactName::FieldFirstName);
    m_FetchRequest->setSorting(QList<QContactSortOrder>() << nameOrder);
    connect(m_FetchRequest, SIGNAL(resultsAvailable()), this, SLOT(printContacts()));

    // async
    if (!m_FetchRequest->start()) {

    }
}

bool horizontalcall::getExternalStyleSheet(QString *styleSheet, QString fileName)
{
    QFile qss(fileName);
    if (!qss.exists())
    {
        qDebug("Style sheet file not found.");
        return false;
    }

    qss.open(QFile::ReadOnly);
    QString styleSheetData = (qss.readAll());
    qss.close();

    *styleSheet = styleSheetData;
    return true;
}

bool horizontalcall::isAllEnglish(QString label)
{
    bool retVal = true;
    for (int i = 0; i < label.length(); i++)
    {
        if (!(label.at(i) >= 32 && label.at(i) <= 126))
        {
            retVal = false;
        }
    }

    return retVal;
}

void horizontalcall::resetButtonClicked()
{
    lineEditFilter->setText("");
    lineEditTextChanged("");
}

void horizontalcall::printContacts()
{
    qDebug("Printing contacts");
    m_Contacts = m_FetchRequest->contacts();

    fillListWidget();

    lineEditFilter->clear();

    lineEditTextChanged("");
    labelSelector->setText(QString::number(m_Contacts.length()));
    lineEditFilter->setReadOnly(false);
    qDebug("Done Printing Contacts");
    m_FetchPending = false;
}

void horizontalcall::CreateLetter(QString letter)
{
    QPushButton *myButton;
    if (m_Landscape) {
        myButton = new QPushButton(leftWidget);
    } else {
        myButton = new QPushButton(lettersGridLayoutWidget);
    }

    //http://wiki.forum.nokia.com/index.php/CS001501_-_Creating_a_gradient_background_for_a_QPushButton_with_style_sheet

    myButton->setStyleSheet(m_StyleLetters);
    myButton->setMaximumSize(10000,10000);
    myButton->setMinimumSize(0,0);
    myButton->sizePolicy().setVerticalPolicy(QSizePolicy::Expanding);
    myButton->sizePolicy().setHorizontalPolicy(QSizePolicy::Expanding);
    existingLetterButtons->insert(existingLetterButtons->size(),myButton);
    myButton->setText(letter.toUpper());
    connect(myButton, SIGNAL(clicked()), this, SLOT(insertChar()));
    lettersGridLayout->addWidget(myButton, col, row);
    if (row == numberOfLettersPerRow - 1)
    {
        row = 0;
        col++;
    } else {
        row++;
    }
    myButton->show();
}

void horizontalcall::insertChar()
{
    QPushButton *tempButton = qobject_cast<QPushButton *>(QObject::sender());
    lineEditFilter->insert(tempButton->text());
    lineEditTextChanged(lineEditFilter->text());
}

horizontalcall::~horizontalcall()
{
    delete ui;
}
