# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
FreeMind document parser. 
Provides functions for creating a mind map object from XML file and vice versa

@authors: Andrew Golovchenko
'''

# Import itself to invoke its methods via getattr()
import parser #pylint: disable=W0406

from lxml import etree as xml
from PyQt4.QtCore import QFile, QIODevice, QTextStream, QCoreApplication
from PyQt4.QtGui import QTextDocument, QColor, QFont, QUndoStack
from hivemind.core import Node, MindMap, Cloud, Attribute, AttributeRegistry
from hivemind.commands import AddNodeCommand, MoveNodeCommand, RemoveNodeCommand, \
    StateModificationCommand, ActionCommand, CurrentNodeCommand, ShutdownCommand, \
    AffiliationCommand, ConfigurationCommand


def _parseNode(root, parent = None):
    '''
    Save XML subtree information into a mind map node object
    
    @param root: XML subtree root
    @type root: lxml Element
    
    @param parent: parent of node to be parsed
    @type parent: Node
    
    @return: root map object
    @rtype: Node
    '''
    QCoreApplication.instance().processEvents()
    storeAttrs = {}
    attributes = _castAttributes(root.attrib)
    # prevent '__text' field creation
    nodeText = attributes.pop('text', None)
    if nodeText is not None: storeAttrs['text'] = nodeText
    nodeStyle = attributes.pop('style', None)
    if nodeStyle is not None: storeAttrs['style'] = nodeStyle
    nodePosition = attributes.pop('left', None)
    if nodePosition is not None: storeAttrs['left'] = nodePosition
    nodeColor = attributes.pop('color', None)
    if nodeColor is not None: storeAttrs['color'] = nodeColor
    backgroundColor = attributes.pop('background_color', None)
    if backgroundColor is not None: storeAttrs['backgroundColor'] = backgroundColor
    if attributes.pop('propagate_formatting', None):
        storeAttrs['propagateNodeFormatting'] = True
    else:
        storeAttrs['propagateNodeFormatting'] = False
    node = parent.createChildPrototype(**attributes) if parent else Node(**attributes)
    icons = []
    # parse child node attributes
    for element in root:
        if element.tag == 'richcontent':
            # save <html></html> child element as simple text
            htmlText = xml.tostring(element[0])
            if element.attrib['TYPE'] == 'NODE':
                storeAttrs['html'] = htmlText
            else: # element type is 'NOTE'
                richContent = QTextDocument()
                richContent.setHtml(htmlText)
                storeAttrs['note'] = richContent
        elif element.tag == 'edge':
            edgeAttributes = _castAttributes(element.attrib)
            width = edgeAttributes.get('width')
            if width: storeAttrs['edgeWidth'] = width
            color = edgeAttributes.get('color')
            if color: storeAttrs['edgeColor'] = color
            style = edgeAttributes.get('style')
            if style: storeAttrs['edgeStyle'] = style
        elif element.tag == 'label':
            labelAttributes = _castAttributes(element.attrib)
            text = labelAttributes.get('text')
            if text:
                storeAttrs['labelText'] = text
                color = labelAttributes.get('color')
                if color: storeAttrs['labelColor'] = color
                font = labelAttributes.get('font')
                if font: storeAttrs['labelFont'] = font
                if labelAttributes.get('propagate_formatting'):
                    storeAttrs['propagateLabelFormatting'] = True
                else:
                    storeAttrs['propagateLabelFormatting'] = False
        elif element.tag == 'cloud':
            storeAttrs['cloud'] = Cloud(**_castAttributes(element.attrib))
        elif element.tag == 'font':
            storeAttrs['font'] = _parseFont(element)
        elif element.tag == 'icon':
            icons.append(element.get('BUILTIN'))
        # FIXME: simplify Attribute class processing
        elif element.tag == 'attribute_layout':
            node.setAttributes(nodeAttribute = Attribute(**dict(element.attrib)))
        elif element.tag == 'attribute':
            node.addNodeAttribute(element.get('NAME'), element.get('VALUE'))
        elif element.tag == 'node':
            if icons: storeAttrs['icons'] = icons
            node.setAttributes(**storeAttrs)
            storeAttrs.clear()
            node.addChild(_parseNode(element, node))
        else:
            print "Unknown element %s" % element.tag
    if storeAttrs:
        if icons: storeAttrs['icons'] = icons
        node.setAttributes(**storeAttrs)
    return node

__INTEGERS = set(('created', 'modified', 'hgap', 'vgap', 'vshift',
              'font_size', 'name_width', 'value_width'))
'''Integer attributes'''

__WIDTHS = set(('width', 'edgeWidth'))
'''Width holding attributes'''

__COLORS = set(('color', 'background_color', 'edgeColor', 'labelColor', 'backgroundColor'))
'''QColor attributes'''

__FONTS = set(('font', 'labelFont'))
'''QFont attributes'''

__BOOLEANS = set(('folded', 'propagate_formatting', 'propagateNodeFormatting',
        'propagateLabelFormatting'))
'''Attribute that store boolean values'''

__NEW_LINE = '?_newline_?'
'''Special value to replace \n in network mode in node text attribute'''

def _castAttributes(attributes, fileMode = True):
    '''
    Cast attribute values from string to more convenient type

    @param attributes: dict of objects to casting
    @type attributes: dict

    @param fileMode: if True, attribute names will be converted to lover case and
    \n won't be replaced in node text attribute
    @type fileMode: bool
    '''
    casted = {}
    for attrName, attrValue in attributes.items():
        if fileMode: attrName = attrName.lower()
        if attrName == 'position': attrName = 'left'
        if attrValue == "":
            casted[attrName] = None
        elif attrName in __INTEGERS:
            casted[attrName] = int(attrValue)
        elif attrName in __COLORS:
            casted[attrName] = QColor(attrValue)
        elif attrName in __WIDTHS:
            # The 'thin' defined in mind map file format, but it is equals to '1' value
            # in paint context. For compatibility used mod10(width) value
            casted[attrName] = 11 if attrValue == 'thin' else int(attrValue)
        elif attrName in __BOOLEANS:
            if attrValue.lower() == 'true':
                casted[attrName] = True
            else:
                casted[attrName] = False
        elif attrName in __FONTS:
#           Font attribute is a comma-separated list of the attributes of QFont object.
#           To get this attribute use QFont::toString() method
#
#           Format (see QFont documentation for details):
#               "family,pointSizeF,pixelSize,styleHint,weight,style,underline,strikeOut,
#                fixedPitch,rawMode"
#           Example: FONT="Helvetica,12,-1,5,50,0,0,0,0,0"
            font = QFont()
            font.fromString(attrValue)
            casted[attrName] = font
        elif attrName == 'icons':
            casted[attrName] = eval(attrValue, None, None)
        elif attrName == 'text' and not fileMode:
            casted[attrName] = attrValue.replace(__NEW_LINE, '\n')
        elif attrName == 'left':
            casted[attrName] = True if attrValue == 'left' else False
        else:
            casted[attrName] = attrValue
    return casted

def _toString(attributes, fileMode = True):
    '''
    Cast various type objects to string

    @param attributes: dict of objects to casting
    @type attributes: dict

    @param fileMode: if True, attribute names will be converted to upper case and boolean
    attributes won't be saved
    @type fileMode: bool

    @return: modified attributes
    @rtype: dict
    '''
    stringAttributes = {}
    for attrName, attrValue in attributes.items():
        if attrName == 'left':
            attrName = 'position'
        strAttrKey = attrName.upper() if fileMode else attrName
        if attrValue is None:
            stringAttributes[strAttrKey] = ""
        elif attrName in __COLORS:
            stringAttributes[strAttrKey] = str(attrValue.name())
        # replace 'True' with FreeMind format specified value 'true'
        elif attrName in __BOOLEANS:
            if attrValue is True:
                stringAttributes[strAttrKey] = 'true'
            elif not fileMode: # send 'false' value over the network
                stringAttributes[strAttrKey] = 'false'
        elif attrName == 'width':
            stringAttributes[strAttrKey] = str(attrValue) if attrValue != 11 else 'thin'
        elif attrName == 'font':
            stringAttributes[strAttrKey] = unicode(attrValue.toString())
        elif attrName == 'position':
            stringAttributes[strAttrKey] = 'left' if attrValue else 'right'
        elif attrName == 'text' and not fileMode:
            stringAttributes[strAttrKey] = attrValue.replace('\n', __NEW_LINE)
        else:
            stringAttributes[strAttrKey] = unicode(attrValue)
    return stringAttributes

def _nodeToXml(root):
    '''
    Create XML subtree root from map object
    
    @param root: root map object
    @type root: Node
   
    @return: XML subtree root
    @rtype: lxml Element
    '''
    nodeAttributes = root.attributes
    # do not store position attribute if node is not first-level child
    if root.parent is not None and root.parent.parent is not None:
        nodeAttributes.pop('left', None)
    if 'backgroundColor' in nodeAttributes:
        backgroundColor = nodeAttributes.pop('backgroundColor')
        nodeAttributes['background_color'] = backgroundColor
    if 'propagateNodeFormatting' in nodeAttributes:
        propagateNodeFormatting = nodeAttributes.pop('propagateNodeFormatting')
        nodeAttributes['propagate_formatting'] = propagateNodeFormatting
    xmlElement = xml.Element('node')
    xmlElement.attrib.update(_toString(nodeAttributes))
    if root.isHtml():
        child = xml.Element('richcontent')
        child.set('TYPE', 'NODE')
        content = xml.XML(root.text)
        child.append(content)
        xmlElement.append(child)
    else: # node content is a plain text
        xmlElement.set('TEXT', root.text)
    if root.note:
        child = xml.Element('richcontent')
        child.set('TYPE', 'NOTE')
        content = xml.XML(str(root.note.toHtml()))
        child.append(content)
        xmlElement.append(child)
    attrsStore = {}
    if root.edgeWidth:
        attrsStore['width'] = root.edgeWidth
    if root.edgeColor:
        attrsStore['color'] = root.edgeColor
    if root.edgeStyle:
        attrsStore['style'] = root.edgeStyle
    if attrsStore:
        edge = xml.Element('edge')
        edge.attrib.update(_toString(attrsStore))
        xmlElement.append(edge)
    attrsStore.clear()
    if root.labelText:
        attrsStore['text'] = root.labelText
        if root.labelColor:
            attrsStore['color'] = root.labelColor
        if root.labelFont:
            attrsStore['font'] = root.labelFont
        if root.propagateLabelFormatting:
            attrsStore['propagate_formatting'] = root.propagateLabelFormatting
    if attrsStore:
        label = xml.Element('label')
        label.attrib.update(_toString(attrsStore))
        xmlElement.append(label)
    if root.cloud:
        element = xml.Element('cloud')
        element.attrib.update(_toString(root.cloud.attributes))
        xmlElement.append(element)
    if root.font:
        xmlElement.append(_fontToXml(root.font))
    if root.icons:
        for icon in root.icons:
            element = xml.Element('icon')
            element.set('BUILTIN', icon)
            xmlElement.append(element)
    if root.nodeAttribute:
        layout = _toString(root.nodeAttribute.attributes)
        if layout:
            layoutElement = xml.Element('attribute_layout')
            layoutElement.attrib.update(layout)
            xmlElement.append(layoutElement)
        for name, value in root.allNodeAttributes:
            attrElement = xml.Element('attribute', NAME = name, VALUE = value)
            xmlElement.append(attrElement)
    for child in root.children:
        xmlElement.append(_nodeToXml(child))
    return xmlElement

def _parseMindMap(root):
    '''
    Load information from XML tree into MindMap object
    
    @param root: XML tree root
    @type root: lxml Element
    
    @raise Exception: if no root node found
    
    @return: mind map object
    @rtype: MindMap
    '''
    rootNode = attributeRegistry = None
    for element in root:
        if element.tag == 'attribute_registry':
            attributeRegistry = _parseAttrRegistry(element)
        elif element.tag == 'node':
            rootNode = _parseNode(element)
            break
    if rootNode is None:
        raise Exception('Mind map must have root node')
    return MindMap(rootNode, attributeRegistry)

def _parseAttrRegistry(xmlElement):
    '''
    Save XML tree information into AttributeRegistry object
    
    @param xmlElement: XML element
    @type xmlElement: lxml Element
    
    @return: attribute registry object
    @rtype: AttributeRegistry
    '''
    registry = AttributeRegistry(**dict(xmlElement.attrib))
    # parse attribute_name subelements
    for attrName in xmlElement:
        name = attrName.get('NAME')
        # up-casting is safe
        registry.viewOptions[name] = dict(attrName.attrib)
        # remove duplicate info from nested dict
        del registry.viewOptions[name]['NAME']
        # parse attribute_value subelements
        values = []
        for attrValue in attrName:
            values.append(attrValue.get('VALUE'))
        registry.attrValues[name] = values
    return registry

def _attrRegistryToXml(registry):
    '''
    Create XML element from AttributeRegistry object
    
    @param registry: registry object for converting to XML element
    @type registry: AttributeRegistry
   
    @return: XML element
    @rtype: lxml Element
    '''
    xmlElement = xml.Element('attribute_registry')
    xmlElement.attrib.update(_toString(registry.attributes))
    # add attribute_name subelements
    for name in registry.viewOptions.keys():
        attrName = xml.SubElement(xmlElement, 'attribute_name')
        attrName.set('NAME', name)
        attrName.attrib.update(registry.viewOptions[name])
        # add attribute_value subelements
        for value in registry.attrValues[name]:
            xml.SubElement(attrName, 'attribute_value', VALUE = value)
    return xmlElement

def _parseFont(xmlElement):
    '''
    Save XML tree information into Qt font object
    
    @param xmlElement: XML element
    @type xmlElement: lxml Element
    
    @return: Qt font object
    @rtype: QFont
    '''
    attributes = xmlElement.attrib
    font = QFont()
    font.setFamily(attributes['NAME'])
    font.setPointSize(int(attributes['SIZE']))
    if attributes.get('BOLD'):
        font.setBold(True)
    if attributes.get('ITALIC'):
        font.setItalic(True)
    return font

def _fontToXml(qfont):
    '''
    Create XML element from QFont object
    
    @param qfont: font for converting to XML element
    @type qfont: QFont
   
    @return: XML element
    @rtype: lxml Element
    '''
    xmlElement = xml.Element('font')
    xmlElement.set('NAME', str(qfont.family()))
    xmlElement.set('SIZE', str(qfont.pointSize()))
    if qfont.bold():
        xmlElement.set('BOLD', 'true')
    if qfont.italic():
        xmlElement.set('ITALIC', 'true')
    return xmlElement

def loadMindMap(filename):
    '''
    Load FreeMind document from file into mind map root object
    
    @param filename: path to FreeMind document
    @type filename: str
    
    @return: root mind map object
    @rtype: MindMap
    '''
    mapFile = QFile(filename)
    mapFile.open(QIODevice.ReadOnly)
    reader = QTextStream(mapFile)
    stringData = unicode(reader.readAll())
    mapFile.close()
    return _parseMindMap(xml.fromstring(stringData))

def mindMapToString(mindMap, formatted = False):
    '''
    Serialize mind map to a string representation of its XML tree

    @param mindMap: mind map to be serialized
    @type mindMap: MindMap

    @param formatted: enable formatted XML generation
    @type formatted: bool

    @return: mind map as XML-string
    @rtype: string
    '''
    resultXml = xml.Element('map', version = '0.9.0')
    comment = xml.Comment(' To view this file, download free mind mapping software ' + \
            'FreeMind from http://freemind.sourceforge.net ')
    resultXml.append(comment)
    if mindMap.attributeRegistry is not None:
        resultXml.append(_attrRegistryToXml(mindMap.attributeRegistry))
    resultXml.append(_nodeToXml(mindMap.root))
#    FIXME: use UTF8 encoding in new file format
#    return xml.tostring(map, pretty_print = formatted, encoding = 'utf8')
    return xml.tostring(resultXml, pretty_print = formatted)

def mindMapFromString(serializedMindMap):
    '''
    Load mind map from string

    @param serializedMindMap: mind map as XML-string
    @type serializedMindMap: string

    @return: deserialized mind map
    @rtype: MindMap
    '''
    return _parseMindMap(xml.fromstring(serializedMindMap))

def _fillAddNodeCommandElement(command, commandElement):
    '''
    Add required content to AddNodeCommand XML element

    @param command: command object
    @type command: AddNodeCommand

    @param commandElement: XML element for processing
    @type commandElement: lxml Element
    '''
    parentElement = xml.Element('parent', id = command.parentNode.id)
    if command.childIndex is not None:
        parentElement.set('childIndex', str(command.childIndex))
    commandElement.append(parentElement)
    node = xml.Element('node')
    node.attrib.update(_toString(command.childNode.getAttributes(getAll = True),
            fileMode = False))
    commandElement.append(node)


def _fillMoveNodeCommandElement(command, commandElement):
    '''
    Add required content to MoveNodeCommand XML element

    @param command: command object
    @type command: MoveNodeCommand

    @param commandElement: XML element for processing
    @type commandElement: lxml Element
    '''
    moveParams = xml.Element('move_params')
    moveParams.set('node_id', command.currentNode.id)
    moveParams.set('target', command.newParentNode.id)
    moveParams.set('index', str(command.newPosition))
    moveParams.set('side', str(command.newSide))
    commandElement.append(moveParams)

def _fillRemoveNodeCommandElement(command, commandElement):
    '''
    Add required content to RemoveNodeCommand XML element

    @param command: command object
    @type command: RemoveNodeCommand

    @param commandElement: XML element for processing
    @type commandElement: lxml Element
    '''
    commandElement.append(xml.Element('node', id = command.currentNode.id))

def _fillStateModificationCommandElement(command, commandElement):
    '''
    Add required content to StateModificationCommand XML element

    @param command: command object
    @type command: StateModificationCommand

    @param commandElement: XML element for processing
    @type commandElement: lxml Element
    '''
    commandElement.append(xml.Element('node', id = command.node.id))
    stateAfter = xml.Element('state_after')
    stateAfter.attrib.update(_toString(command.stateAfter, fileMode = False))
    commandElement.append(stateAfter)

def _fillActionCommandElement(command, commandElement):
    '''
    Add required content to string XML element

    @param command: command object
    @type command: ActionCommand

    @param commandElement: XML element for processing
    @type commandElement: lxml Element
    '''
    commandElement.append(xml.Element('action', type = command.action))

def _fillCurrentNodeCommandElement(command, commandElement):
    '''
    Add required content to string XML element

    @param command: command object
    @type command: CurrentNodeCommand

    @param commandElement: XML element for processing
    @type commandElement: lxml Element
    '''
    commandElement.append(xml.Element('node', id = command.node.id))

def _fillShutdownCommandElement(command, commandElement):
    '''
    Add required content to string XML element

    @param command: command object
    @type command: ShutdownCommand

    @param commandElement: XML element for processing
    @type commandElement: lxml Element
    '''
    commandElement.append(xml.Element('reason', reason = command.reason))

def _fillAffiliationCommandElement(command, commandElement):
    '''
    Add required content to string XML element

    @param command: command object
    @type command: AffiliationCommand

    @param commandElement: XML element for processing
    @type commandElement: lxml Element
    '''
    params = xml.Element('affiliation_params')
    params.set('entity', command.entity)
    params.set('affiliation', command.affiliation)
    commandElement.append(params)

def _fillConfigurationCommandElement(command, commandElement):
    '''
    Add required content to string XML element

    @param command: command object
    @type command: ConfigurationCommand

    @param commandElement: XML element for processing
    @type commandElement: lxml Element
    '''
    params = xml.Element('configuration')
    for option, value in command.config.iteritems():
        params.set(option, value)
    commandElement.append(params)

def _commandToXml(command):
    '''
    Create lxml Element from command object

    @param command: command object to be serialized
    @type command: QUndoCommand

    @return: XML element
    @rtype: lxml Element
    '''
    commandElement = xml.Element('command')
    commandElement.set('type', command.__class__.__name__)
    fillElementFunction = getattr(parser, '_fill%sElement' % command.__class__.__name__)
    fillElementFunction(command, commandElement)
    return commandElement

def serializeCommand(command):
    '''
    Serialize the command to XML in the following format:
        <command type="...">
            <parent id="..."/>
            <node ...>...</node>
        </command>

    @param command: command object to be serialized
    @type command: QUndoCommand

    @return: XML representation of the command object
    @rtype: unicode
    '''
    return xml.tostring(_commandToXml(command), encoding = 'UTF-8')

def serializeCommandStack(stack):
    '''
    Serialize the stack of commands to XML in the following format:
        <commandStack index="...">
            <command type="..."></command>
            ...
            <command type="..."></command>
        </commandStack>

    @param stack: command stack object to be serialized
    @type stack: QUndoStack

    @return: XML representation of the command stack object
    @rtype: unicode
    '''
    stackElement = xml.Element('commandStack')
    stackElement.set('index', str(stack.index()))
    for index in xrange(stack.count()):
        stackElement.append(_commandToXml(stack.command(index)))
    return xml.tostring(stackElement, encoding = 'UTF-8')

def _deserializeAddNodeCommand(commandElement, mindMap):
    '''
    Deserialize AddNodeCommand command from XML string

    @rtype: AddNodeCommand
    '''
    parentElement = commandElement[0]
    parent = mindMap.getNodeById(parentElement.get('id'))
    nodeElement = commandElement[1]
    nodeAttributes = _castAttributes(nodeElement.attrib, fileMode = False)
    child = parent.createChildPrototype()
    child.setAttributes(**nodeAttributes)
    if child.left is None: child.setAttributes(left = parent.left)
    index = parentElement.get('childIndex')
    if index is not None:
        return AddNodeCommand(parent, child, int(index))
    return AddNodeCommand(parent, child)

def _deserializeMoveNodeCommand(commandElement, mindMap):
    '''
    Deserialize MoveNodeCommand command from XML string

    @rtype: MoveNodeCommand
    '''
    moveParamsElement = commandElement[0]
    node = mindMap.getNodeById(moveParamsElement.get('node_id'))
    target = mindMap.getNodeById(moveParamsElement.get('target'))
    index = int(moveParamsElement.get('index'))
    side = True if moveParamsElement.get('side') == 'True' else False
    return MoveNodeCommand(node, target, index, side)

def _deserializeRemoveNodeCommand(commandElement, mindMap):
    '''
    Deserialize RemoveNodeCommand command from XML string

    @rtype: RemoveNodeCommand
    '''
    nodeElement = commandElement[0]
    node = mindMap.getNodeById(nodeElement.get('id'))
    return RemoveNodeCommand(node)

def _deserializeStateModificationCommand(commandElement, mindMap):
    '''
    Deserialize StateModificationCommand command from XML string

    @rtype: StateModificationCommand
    '''
    nodeElement = commandElement[0]
    node = mindMap.getNodeById(nodeElement.get('id'))
    afterElement = commandElement[1]
    stateAfter = _castAttributes(afterElement.attrib, fileMode = False)
    stateBefore = node.getAttributes(*stateAfter.keys())
    return StateModificationCommand(node, stateBefore, stateAfter)

def _deserializeActionCommand(commandElement, mindMap):
    '''
    Deserialize action command from XML string

    @rtype: ActionCommand
    '''
    nodeElement = commandElement[0]
    return ActionCommand(nodeElement.get('type'))

def _deserializeCurrentNodeCommand(commandElement, mindMap):
    '''
    Deserialize current node command from XML string

    @rtype: CurrentNodeCommand
    '''
    nodeElement = commandElement[0]
    node = mindMap.getNodeById(nodeElement.get('id'))
    return CurrentNodeCommand(node)

def _deserializeShutdownCommand(commandElement, mindMap):
    '''
    Deserialize shutdown command from XML string

    @rtype: ShutdownCommand
    '''
    nodeElement = commandElement[0]
    return ShutdownCommand(nodeElement.get('reason'))

def _deserializeAffiliationCommand(commandElement, mindMap):
    '''
    Deserialize AffiliationCommand command from XML string

    @rtype: AffiliationCommand
    '''
    paramsElement = commandElement[0]
    entity = paramsElement.get('entity')
    affiliation = paramsElement.get('affiliation')
    return AffiliationCommand(entity, affiliation)

def _deserializeConfigurationCommand(commandElement, mindMap):
    '''
    Deserialize ConfigurationCommand command from XML string

    @rtype: AffiliationCommand
    '''
    paramsElement = commandElement[0]
    command = ConfigurationCommand()
    for option in paramsElement.items():
        name, value = option
        command.setOption(name, value)
    return command

def _xmlToCommand(commandElement, mindMap):
    '''
    Create command object from lxml Element

    @param commandElement: XML element for processing
    @type commandElement: lxml Element

    @param mindMap: mind map object for search nodes by id
    @type mindMap: MindMap

    @return: command object
    @rtype: QUndoCommand
    '''
    deserializeFunction = getattr(parser, '_deserialize%s' % commandElement.get('type'))
    return deserializeFunction(commandElement, mindMap)

def deserializeCommand(xmlString, mindMap):
    '''
    Deserialize the command from XML string

    @param xmlString: serialized command object
    @type xmlString: unicode

    @param mindMap: mind map object for search nodes by id
    @type mindMap: MindMap

    @return: command object
    @rtype: QUndoCommand
    '''
    return _xmlToCommand(xml.fromstring(xmlString), mindMap)

def deserializeCommandStack(xmlString, mindMap):
    '''
    Deserialize the stack of commands from XML string

    @param xmlString: serialized command stack object
    @type xmlString: unicode

    @param mindMap: mind map object for search nodes by id
    @type mindMap: MindMap

    @return: command stack object
    @rtype: QUndoStack
    '''
    commandStack = QUndoStack()
    stackElement = xml.fromstring(xmlString)
    for commandElement in stackElement:
        commandStack.push(_xmlToCommand(commandElement, mindMap))
    commandStack.setIndex(int(stackElement.get('index')))
    return commandStack

def storeMindMap(fileName, mindMap):
    '''
    Store mind map root object to file as FreeMind document
    
    @param fileName: path to FreeMind document
    @type fileName: str
    
    @param mindMap: mind map to be saved
    @type mindMap: MindMap
    '''
    mapFile = file(fileName, 'w')
    mapFile.write(mindMapToString(mindMap, True))
    mapFile.close()
