# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Network core
'''
from twisted.words.protocols import jabber
from twisted.internet import reactor
from twisted import internet
from twisted.words.protocols.jabber import jid
from wokkel.client import HybridAuthenticator
from wokkel.ping import PingHandler
from hivemind.attribute import readable, Observable
from idavoll.backend import BackendService
from PyQt4.QtCore import QObject
from PyQt4.QtGui import QMessageBox
from hivemind.commands import StateModificationCommand, CurrentNodeCommand, ShutdownCommand, \
    AffiliationCommand
from hivemind.network.models import AffiliationModel, RosterModel
from hivemind.network import messages
from hivemind.network import protocol_layer
from hivemind import gui_factory, parser, settings, enum
from hivemind.network.changeset import Changeset
from hivemind.network.transport_layer import HivemindXmlStreamFactory, HivemindXMPPClient
from hivemind.network.protocol_layer import HivemindStorage, HivemindResourceFromBackend, \
    HivemindClient, HivemindRosterClientProtocol, HivemindPubSubService, XMPP_NODE_NAME
from hivemind.network.ping import PingManager, Pinger

#pylint: disable=C0103
'''Possible connection states'''
connectionState = enum.enum('disconnected', 'collaboration', 'initialized', 'connecting',
                            'connected')
'''Possible subscription states'''
subscriptionState = enum.enum('subscribed', 'unsubscribed')
''' Time in seconds we must wait for unsubscribe '''
UNSUBSCRIBE_TIMEOUT = 2
''' Cold shutdown of network if timeout expires'''
SHUTDOWN_TIMEOUT = 4


class NetworkController(QObject):
    '''
    Asynchronous command processor. Sends commands into shared space, listens to incoming
    messages and sends correct commands to mind map controller.
    @author: Andrew Vasilev, Oleg Kandaurov
    '''

    def __init__(self, mindMapController, actionBag):
        '''
        Constructor
        @type mindMapController: MindMapController
        @type actionBag: ActionBag
        '''
        QObject.__init__(self)
        self.__errorHandler = ErrorHandler()
        self.__transport = Transport(self.__errorHandler)
        self.__affiliationModel = AffiliationModel()
        self.__clientDialog = gui_factory.createNetworkClientDialog(
                self.__transport.rosterModel)
        self.__protocol = Protocol(self.__transport, self.__errorHandler, mindMapController,
                self.__affiliationModel)
        self.__affiliationModel.addObserver(self.__protocol)
        self.__actionBag = actionBag
        self.__affiliationDialog = gui_factory.createAffiliationDialog(self.__affiliationModel,
                self.__transport.rosterModel)
        self.__networkDialog = gui_factory.createNetworkConnectionDialog()
        self.__networkSettingsDialog = gui_factory.createNetworkSettingsDialog()
        self.__secondConnectionStep = None
        self._connectActions()
        self.updateActions(connectionState.disconnected)
        settings.connect(self.__protocol.updateConfiguration)

    def initialize(self):
        '''Initialize controller'''
        self.__transport.addObserver(self)
        self.__protocol.addObserver(self)
        self.__errorHandler.addObserver(self)

    def publishCommand(self, command):
        '''
        Publish command in common space
        @param command: command to publish in shared space
        '''
        self.__protocol.publishCommand(command)

    def terminateNetworkConnection(self):
        '''Terminate network connection'''
        def stopReactor(result):
            '''Stop reactor'''
            if reactor.running:
                reactor.stop()
        if not self.__protocol.mindMapController.prohibitChangeMapAction():
            return
        reactor.callLater(SHUTDOWN_TIMEOUT, stopReactor, self)
        defer = self.closeNetwork()
        defer.addCallback(stopReactor)

    def _connectActions(self):
        '''Connect actions'''
        def prepareService():
            '''Second connection step is service'''
            self.__secondConnectionStep = 'service'

        def prepareClient():
            '''Second connection step is client'''
            self.__secondConnectionStep = 'client'

        def showNetworkDialog():
            '''Perform actions according to network state'''
            if self.__actionBag.disconnectXMPPAction.isEnabled():
                self._performSecondConnectionStep()
            else:
                self.__networkDialog.exec_()

        def startClient():
            '''Configure and start client'''
            self.__protocol.configureClient(
                    self.__clientDialog.attributes['serviceJid'] + '/hivemind')
            self.__protocol.startProtocol()

        self.__actionBag.disconnectXMPPAction.triggered.connect(self.closeNetwork)
        self.__actionBag.stopProtocolAction.triggered.connect(self.__protocol.close)
        self.__actionBag.startServiceAction.triggered.connect(prepareService)
        self.__actionBag.startServiceAction.triggered.connect(showNetworkDialog)
        self.__actionBag.startClientAction.triggered.connect(prepareClient)
        self.__actionBag.startClientAction.triggered.connect(showNetworkDialog)
        self.__actionBag.editAffiliationAction.triggered.connect(
                self.__affiliationDialog.exec_)
        self.__actionBag.networkSettingsAction.triggered.connect(
                self.__networkSettingsDialog.exec_)
        self.__networkDialog.accepted.connect(self._startNetwork)
        self.__clientDialog.accepted.connect(startClient)

    def updateActions(self, state):
        '''
        Update actions according to state
        @type state: str
        '''
        self.__actionBag.connectionStatusAction.status = state
        self.__actionBag.fileNewAction.setDisabled(state == connectionState.collaboration)
        self.__actionBag.fileOpenAction.setDisabled(state == connectionState.collaboration)
        self.__actionBag.disconnectXMPPAction.setEnabled(state != connectionState.disconnected)
        self.__actionBag.stopProtocolAction.setEnabled(state == connectionState.collaboration)
        self.__actionBag.startServiceAction.setDisabled(state == connectionState.collaboration)
        self.__actionBag.startClientAction.setDisabled(state == connectionState.collaboration)
        self.__actionBag.editAffiliationAction.setEnabled(
                state == connectionState.collaboration and self.__protocol.isServiceRole)
        self.__actionBag.networkSettingsAction.setDisabled(
                state == connectionState.collaboration and not self.__protocol.isServiceRole)

    def _startNetwork(self):
        '''Start network connection'''
        self.__transport.start(self.__networkDialog.attributes['userJid'] + '/hivemind',
                self.__networkDialog.attributes['password'],
                self.__networkDialog.attributes['host'],
                self.__networkDialog.attributes['port'])
        reactor.callLater(2, self._performSecondConnectionStep)

    def _performSecondConnectionStep(self):
        ''' Perform second connection step'''
        if self.__secondConnectionStep and self.__transport.xmppClient is not None:
            {'client': self.__clientDialog.exec_,
                        'service': self.__protocol.startService}[self.__secondConnectionStep]()
        self.__secondConnectionStep = None

    def closeNetwork(self):
        '''Closes network connection'''
        defer = self.__protocol.close()
        defer.addCallback(lambda success: self.__transport.close())
        return defer

    def processNotification(self, *values, **kvargs):
        '''Process notification'''
        for event in kvargs.keys():
            getattr(self, '_process%s%sEvent' % (event[0].upper(), event[1:]))(kvargs[event])

    def _processActionEvent(self, action):
        '''Process action event'''
        mapAction = {
                'finishTeamwork': self.__protocol.close,
                'finishAll': self.closeNetwork}
        performAction = mapAction[action]
        performAction()

    def _processStateEvent(self, stateInfo):
        '''Process connection state'''
        mapConnectionState = {
                connectionState.initialized: self.__transport.startHelpers,
                connectionState.disconnected: self.closeNetwork,
                connectionState.connecting: None,
                connectionState.connected: None,
                connectionState.collaboration:  self.__protocol.startProtocolHelpers}
        state, perform = stateInfo
        processEvent = mapConnectionState[state]
        if processEvent is not None and perform:
            processEvent()
        self.updateActions(state)


class ErrorHandler(QObject, Observable):
    '''
    Handles network errors
    @author: Oleg Kandaurov
    '''

    def __init__(self):
        '''Constructor'''
        QObject.__init__(self)
        Observable.__init__(self)
        self.__streamMessages = messages.StreamErrorMessages()
        self.__saslMessages = messages.SASLErrorMessages()
        self.__stanzaMessages = messages.StanzaErrorMessages()
        self.__pubsubMessages = messages.PubSubErrorMessages()

    def _handleSASLError(self, failure):
        '''
        Handle sasl error
        @type failure: Failure
        '''
        errorDialog = gui_factory.createNetworkErrorDialog('finishAll')
        errorDialog.performAction.connect(self._notify)
        errorDialog.showError(self.tr('Authentication fails.'),
                self.tr('Seems that you provide invalid username or password. ' +
                'See details for more info.'),
                self.__saslMessages.condition[failure.value.condition], True)

    def _handleStreamError(self, failure):
        '''
        Handle stream error
        @type failure: Failure
        '''
        errorDialog = gui_factory.createNetworkErrorDialog('finishAll')
        errorDialog.performAction.connect(self._notify)
        errorDialog.showError(self.tr('XMPP connection fails.'),
                self.tr('The error is nonrecoverable. ' +
                'You are automatically disconnected. ' +
                'See details for more info.'),
                self.__streamMessages.condition[failure.value.condition], True)

    def _handleStanzaError(self, failure):
        '''
        Handle stanza error
        @type failure: Failure
        '''
        errorDialog = gui_factory.createNetworkErrorDialog('finishTeamwork')
        errorDialog.performAction.connect(self._notify)
        errorDialog.showError(self.tr('Collaboration error has been occured.'),
                self.tr('Fortunately, the error is not fatal. ' +
                'Often this means that you are not allowed to do some action. ' +
                'See details for more info.'),
                self.__stanzaMessages.condition[failure.value.condition], False)

    def _handlePubSubError(self, failure):
        '''
        Handle pubsub error
        @type failure: Failure
        '''
        errorDialog = gui_factory.createNetworkErrorDialog('finishTeamwork')
        errorDialog.performAction.connect(self._notify)
        condition, feature = protocol_layer.pubsubConditionFromStanzaError(failure.value)
        message = self.__stanzaMessages.condition[failure.value.condition] + \
                self.tr('\nCondition: %s') % (
                self.__pubsubMessages.condition[condition])
        if feature:
            message += self.tr('\nFeature: %s') % (
                    self.__pubsubMessages.feature[feature])
        errorDialog.showError(self.tr('Collaboration error has been occured.'),
                self.tr('Fortunately, the error is not fatal. ' +
                'Often this means that you are not allowed to do some action. ' +
                'See details for more info.'),
                message, False)

    def _notify(self, action):
        '''Notify observers'''
        self.notifyObservers(action = str(action))

    def mapErrorEvent(self, failure):
        ''''
        Map different network errors
        @type failure: Failure
        '''
        error = failure.trap(jabber.sasl.SASLError, jabber.error.StanzaError,
                jabber.error.StreamError, internet.error.ConnectionLost)
        if error == jabber.sasl.SASLError:
            self._handleSASLError(failure)
        elif error == jabber.error.StreamError:
            self._handleStreamError(failure)
        elif error == jabber.error.StanzaError:
            if protocol_layer.isPubSubError(failure.value):
                self._handlePubSubError(failure)
            else:
                self._handleStanzaError(failure)


class Protocol(Observable):
    '''
    Manages protocol handlers

    @author: Oleg Kandaurov
    '''

    readable('mindMapController', 'isServiceRole')

    def __init__(self, transport, errorHandler, mindMapController, affiliationModel):
        '''Constructor'''
        Observable.__init__(self)
        self.__transport = transport
        self.__errorHandler = errorHandler
        self.__mindMapController = mindMapController
        self.__affiliationModel = affiliationModel
        self.__isServiceRole = False
        self.__oldProtocol = None
        self.__pingHandler = None
        self.__pingManager = None
        self.__storage = None
        self.__clientRole = None

    def publishCommand(self, command):
        '''
        Publish command in common space
        @param command: command to publish in shared space
        '''
        def isClientPropagationAllowed():
            '''Whether client can publish or not'''
#            if (self.__clientRole in ['outcast', 'member'] and
#                    self.__serverPublishModel == 'publishers'):
#                return False
            return True

        if self.__oldProtocol:
            if self.__isServiceRole == True or isClientPropagationAllowed():
                if _isPropagationAllowed(command):
                    self.__oldProtocol.proposeChangeset(
                        Changeset(parser.serializeCommand(command), 'command'))
                else:
                    self.__mindMapController.executeCommand(command, False)
        else:
            self.__mindMapController.executeCommand(command)

    def updateConfiguration(self):
        '''Set node configuration according to settings'''
        if self.__storage is None:
            return
        config = {'pubsub#access_model': settings.get('accessModel'),
                   'pubsub#publish_model': settings.get('publishModel')}
        defer = self.__storage.getNode(XMPP_NODE_NAME)
        defer.addCallback(lambda node: node.setConfiguration(config))

    def configureService(self):
        '''Configure HiveMind service'''
        self.__isServiceRole = True
        self.__storage = HivemindStorage()
        backend = BackendService(self.__storage)
        resource = HivemindResourceFromBackend(backend)
        self.__oldProtocol = HivemindPubSubService(resource)
        resource.pubsubService = self.__oldProtocol
        resource.serviceJID = self.__transport.xmppClient.jid
        mindMap = parser.mindMapToString(self.__mindMapController.initialMindMap, False)
        undoStack = parser.serializeCommandStack(self.__mindMapController.undoStack)
        backend.createNode(XMPP_NODE_NAME, resource.serviceJID)
        self.updateConfiguration()
        defer = self.__storage.getNode(XMPP_NODE_NAME)
        defer.addCallback(lambda node: (node.addObserver(self),
                node.storeChangeset(Changeset(mindMap, 'mindmap'), resource.serviceJID),
                node.storeChangeset(Changeset(undoStack, 'undostack'), resource.serviceJID)))

    def configureClient(self, service):
        '''
        Configure HiveMind client
        @param service: Service to connect to
        @type service: str
        '''
        self.__isServiceRole = False
        self.__oldProtocol = HivemindClient(jid.internJID(service))

    def startProtocol(self):
        '''Start Hivemind protocol'''
        self.__oldProtocol.addObserver(self)
        self.__oldProtocol.setHandlerParent(self.__transport.xmppClient)

    def close(self):
        '''Close protocol level'''
        def destroyProtocol(result):
            '''Destroy protocol'''
            if self.__transport.xmppClient:
                self.__oldProtocol.disownHandlerParent(self.__transport.xmppClient)
            self.__oldProtocol.deleteObserver(self)
            self.__oldProtocol = None
            return result
        if self.__transport is None: return
        if self.__transport.xmppClient:
            if self.__pingHandler:
                self.__pingHandler.disownHandlerParent(self.__transport.xmppClient)
                self.__pingHandler = None
        self._destroyPingManager()
        defer = internet.defer.succeed(None)
        if self.__oldProtocol is None or self.__transport.xmppClient is None:
            return defer
        if self.__isServiceRole:
            defer = self.__oldProtocol.proposeChangeset(
                    Changeset(parser.serializeCommand(ShutdownCommand()), 'command'))
        defer.addBoth(self.__oldProtocol.close)
        defer.addBoth(destroyProtocol)
        return defer

    def _destroyPingManager(self):
        '''Destroy ping manager'''
        if self.__pingManager:
            if self.__transport.xmppClient:
                self.__pingManager.removeAll()
            self.__pingManager = None

    def startService(self):
        '''Start service'''
        self.configureService()
        self.startProtocol()

    def startProtocolHelpers(self):
        '''Start protocol helpers'''
        self.__pingManager = PingManager(self, self.__transport.xmppClient)
        self.__pingHandler = PingHandler()
        self.__pingHandler.setHandlerParent(self.__transport.xmppClient)

    def processNotification(self, *values, **kvargs):
        '''Process notification'''
        for event in kvargs.keys():
            getattr(self, '_process%s%sEvent' % (event[0].upper(), event[1:]))(kvargs[event])

    def _processAffiliationEvent(self, event):
        '''Process affiliation event from node'''
        entity, affiliation = event
        self.__affiliationModel.updateAffiliation(entity, affiliation)

    def _processModelAffiliationEvent(self, event):
        '''Process affiliation event from model'''
        entity, affiliation = event
        defer = self.__storage.getNode(XMPP_NODE_NAME)
        defer.addCallback(lambda node: node.setAffiliation(jid.internJID(entity), affiliation))
        command = AffiliationCommand(entity, affiliation)
        self.__oldProtocol.proposeChangeset(
                Changeset(parser.serializeCommand(command), 'command'))

    def _processErrorEvent(self, failure):
        '''Process error event'''
        self.__errorHandler.mapErrorEvent(failure)

    def _processProtocolPingEvent(self, event):
        '''Process ping event from protocol'''
        entity = event['entity']
        state = event['state']
        if state == Pinger.entityState.timeout:
            if self.__oldProtocol and self.__transport.xmppClient:
                # seems broken, need inspection
                self.__oldProtocol.unsubscribe(entity, self.__transport.xmppClient.jid,
                            XMPP_NODE_NAME, entity)
            if self.__pingManager:
                self.__pingManager.stopPing(entity)

    def _processSubscriptionEvent(self, event):
        '''Process subscription event'''
        entity = event['entity']
        state = event['state']
        self.__affiliationModel.updateSubscriptionState(entity, state)
        if self.__pingManager is None: return
        if state == subscriptionState.subscribed:
            self.__pingManager.addPinger(entity,
                    settings.get('protocolPingInterval'),
                    settings.get('protocolPingMaxCount'))
            self.__pingManager.startPing(entity)
        else:
            self.__pingManager.removePinger(entity)

    def _processConnectionEvent(self, state):
        '''Process connection event'''
        self.notifyObservers(state = (state, True))

    def _processChangesetEvent(self, changeset):
        '''Process changeset event'''
        if self.__isServiceRole and changeset.type != 'command':
            return
        data = self.__mindMapController.deserializeData(changeset.data, changeset.type)
        if not self.__isServiceRole:
            if isinstance(data, ShutdownCommand):
                self.close()
                QMessageBox.warning(None, self.__mindMapController.tr('Warning'),
                        self.__mindMapController.tr('Service has been disconnected'))
                return
            elif isinstance(data, AffiliationCommand):
                if data.entity == self.__transport.xmppClient.jid.userhost():
                    self.__clientRole = data.affiliation
                return
        mapChangesetType = {'mindmap': 'loadMindMap',
                'command': 'executeCommand',
                'undostack': 'loadUndoStack'}
        processChangeset = mapChangesetType[changeset.type]
        getattr(self.__mindMapController, processChangeset)(data)


class Transport(Observable):
    '''
    Manages XMPP connection

    @author: Oleg Kandaurov
    '''

    readable('xmppClient', 'rosterModel')
    def __init__(self, errorHandler):
        '''Constructor'''
        Observable.__init__(self)
        self.__errorHandler = errorHandler
        self.__rosterModel = RosterModel()
        self.__rosterProtocol = None
        self.__pinger = None
        self.__xmppClient = None

    def start(self, userJid, password, host, port):
        '''
        Start XMPP connection
        @param userJid: JID for XMPP connection
        @type userJid: str
        @param password: password for XMPP connection
        @type password: str
        @type host: str
        @type port: int
        '''
        authenticator = HybridAuthenticator(jid.internJID(userJid), password)
        factory = HivemindXmlStreamFactory(authenticator)
        factory.addObserver(self)
        self.__xmppClient = HivemindXMPPClient(authenticator, factory, host, port)
        self.__xmppClient.addObserver(self)
        self.__xmppClient.startService()

    def _createRoster(self):
        '''Create roster'''
        self.__rosterProtocol = HivemindRosterClientProtocol()
        self.__rosterProtocol.addObserver(self)
        self.__rosterProtocol.setHandlerParent(self.__xmppClient)

    def _createPinger(self):
        '''Create pinger'''
        self.__pinger = Pinger(self.__xmppClient.server,
                settings.get('transportPingInterval'),
                settings.get('transportPingMaxCount'), isServer = True)
        self.__pinger.setHandlerParent(self.__xmppClient)
        self.__pinger.addObserver(self)
        self.__pinger.startPing()

    def startHelpers(self):
        '''Start transport helpers'''
        self._createRoster()
        self._createPinger()

    def close(self):
        ''' Close transport level'''
        self._destroyRoster()
        self._destroyPinger()
        if self.__xmppClient:
            self.__xmppClient.stopService()
            self.__xmppClient = None
        self.notifyObservers(state = (connectionState.disconnected, False) )

    def _destroyRoster(self):
        '''Destroy roster protocol'''
        if self.__rosterProtocol:
            if self.__xmppClient:
                self.__rosterProtocol.disownHandlerParent(self.__xmppClient)
            self.__rosterProtocol.deleteObserver(self)
            self.__rosterProtocol = None

    def _destroyPinger(self):
        '''Destroy pinger'''
        if self.__pinger:
            self.__pinger.stopPing()
            if self.__xmppClient:
                self.__pinger.disownHandlerParent(self.__xmppClient)
            self.__pinger.deleteObserver(self)
            self.__pinger = None

    def processNotification(self, *values, **kvargs):
        '''Process notification'''
        for event in kvargs.keys():
            getattr(self, '_process%s%sEvent' % (event[0].upper(), event[1:]))(kvargs[event])

    def _processErrorEvent(self, failure):
        '''Process error event'''
        self.__errorHandler.mapErrorEvent(failure)

    def _processTransportPingEvent(self, event):
        '''Process ping event from transport'''
        state = event['state']
        if state == Pinger.entityState.timeout:
            self._processConnectionEvent(connectionState.disconnected)
            if self.__pinger:
                self.__pinger.stopPing()
            if self.__xmppClient:
                self._reconnect()

    def _processRosterEvent(self, event):
        '''Process roster event'''
        mapEvt = {HivemindRosterClientProtocol.eventType.roster: self.__rosterModel.setRoster,
         HivemindRosterClientProtocol.eventType.remove: self.__rosterModel.removeContact,
         HivemindRosterClientProtocol.eventType.set: self.__rosterModel.setContact}
        processEvent = mapEvt[event['type']]
        processEvent(event['data'])

    def _processConnectionEvent(self, state):
        '''Process connection event'''
        self.notifyObservers(state = (state, True))

    def _reconnect(self):
        '''
        Perform reconnect. Implementation is somewhat tricky.
        '''
        self.__xmppClient.factory.resetDelay()
        self.__xmppClient.factory.retry(self.__xmppClient.connection)


def startReactor():
    '''Starts the reactor and defines exit behaviour'''
    reactor.runReturn()
    # stopping twisted event shuts down application
    reactor.addSystemEventTrigger('after', 'shutdown', reactor.qApp.quit)
    # stop twisted event loop when last window is closed
    reactor.qApp.lastWindowClosed.connect(reactor.stop)


def _isPropagationAllowed(command):
    '''
    Check if command must be propagated through network or not
    @param command: command to inspect
    '''
    if settings.get('presentationMode'):
        return True
    if isinstance(command, StateModificationCommand) and command.isFoldCommand():
        return False
    if isinstance(command, CurrentNodeCommand):
        return False
    return True
