# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
View delegates for core objects
'''

import math
from PyQt4.QtCore import QRectF, QPointF, Qt, QObject, pyqtSignal, QCoreApplication
from PyQt4.QtGui import QGraphicsScene, QGraphicsItem, QAction, QPen, \
QPainterPath, QPolygonF, QIcon, QTextDocument, QPainter, QGraphicsView
import hivemind.resource as resource
import hivemind.settings as settings
from hivemind.attribute import readable, writable
from hivemind.core import MindMap, Node

class MindMapDelegate(QGraphicsScene):
    '''
    MindMap delegate

    Warning! C{currentNode} field is self-encapsulated!
    You must not use direct invocation (C{self.__currentNode})
    inside the class!

    @authors: Andrew Golovchenko, Andrew Vasilev, Alexander Kulikov
    '''

    readable('rootDelegate', 'mindMap', 'nodeMapCreator')

    mapChanged = pyqtSignal()
    curMapChanged = pyqtSignal()
    currentNodeChanged = pyqtSignal(Node)

    def _getCurrentNode(self):
        '''Get current node'''
        return self.__currentNode

    def _setCurrentNode(self, currentNode):
        '''
        Set the current node
        @type currentNode: Node
        '''
        if self.__nodeToDelegate.has_key(self.__currentNode):
            self.__nodeToDelegate[self.__currentNode].selected = False

        self.__currentNode = currentNode
        delegate = self.__nodeToDelegate[currentNode]
        delegate.selected = True
        delegate.ensureVisible(delegate.textBoundingRect())
        self.currentNodeChanged.emit(self.__currentNode)

    currentNode = property(_getCurrentNode, _setCurrentNode, None, 'Current node')

    def __init__(self, actionBag):
        '''Initialize empty mind map'''
        QGraphicsScene.__init__(self)
        settings.connect(self._settingsChanged)
        self.__actionBag = actionBag
        self.__nodeToDelegate = {}
        self.__rootDelegate = None
        self.__currentNode = None
        self.__mindMap = None
        self.__nodeMapCreator = NodeMapCreator()
        # used as 'switch' operator in processNotification() method
        self.__signalHandlers = {'childNodeAdded' : self.addNodeDelegate,
                                 'childNodeRemoved' : self.removeNodeDelegate,
                                 'nodeMoved' : self.relocateNodeDelegate,
                                 'localAttributeChanged' : self.nodeBoundsChanged,
                                 'inheritedAttributeChanged' : self.inheritedAttributeChanged}
        self.setMap(MindMap())
        self.__actionBag.selectLeftNodeAction.triggered.connect(self._selectLeftNode)
        self.__actionBag.selectRightNodeAction.triggered.connect(self._selectRightNode)
        self.__actionBag.selectUpNodeAction.triggered.connect(self._selectUpNode)
        self.__actionBag.selectDownNodeAction.triggered.connect(self._selectDownNode)
        self.__actionBag.selectRootNodeAction.triggered.connect(self._selectRootNode)

    def createAction(self, text, handler, shortcut = None, icon = None):
        '''
        Create action
        @param text: descriptive text for tool buttons
        @type text: string
        @param handler: slot
        @type shortcut: string
        @rtype: QAction
        '''
        action = QAction(text, self)
        if shortcut: action.setShortcut(shortcut)
        if icon: action.setIcon(QIcon.fromTheme(icon))
        if handler: action.triggered.connect(handler)
        return action

    def addNodeDelegate(self, node):
        '''
        This method called, then signal for created delegate has come
        @param node: node was clicked
        @type node: Node
        '''
        #pylint: disable=C0103
        self._addDelegatesForNodeSubtree(node)
        self.positionNodes()
        if self.__nodeToDelegate[node].isVisible():
            self.currentNode = node

    def _addDelegatesForNodeSubtree(self, root):
        '''
        Create delegates for all node descendants
        @param root: subtree root
        @type root: Node
        '''
        parentDelegate = self.__nodeToDelegate[root.parent]
        newDelegate = NodeDelegate(root, parentDelegate)
        if root.parent.folded:
            newDelegate.setVisible(False)
        root.addObserver(self)
        self.__nodeToDelegate[root] = newDelegate
        for child in root.children:
            self._addDelegatesForNodeSubtree(child)

    def calculateVisibilty(self, node, propagate):
        '''
        Calculates visibility for given node and optionally for it children
        @param node: Given node
        @param propagate: Propagate calculation to children nodes
        '''
        visible = False if node.folded is not None else True
        for child in node.children:
            if not self.__nodeToDelegate[child].matched:
                self.__nodeToDelegate[child].setVisible(False)
                continue
            if self.__nodeToDelegate[child].isVisible() != visible:
                self.__nodeToDelegate[child].setVisible(visible)
            if propagate:
                self.calculateVisibilty(child, propagate)

    def nodeBoundsChanged(self, node):
        '''
        This method called, then signal for node bounds changed has come
        @param node: modified node
        @type node: Node
        '''
        self.calculateVisibilty(node, propagate = False)
        self.__nodeToDelegate[node].updateNodeBounds(False)
        self.positionNodes(node)

    def _settingsChanged(self):
        '''Update and position all nodes on the mind map'''
        self.__mindMap.root.updateProperties()
        self.__rootDelegate.updateNodeBounds()
        self.positionNodes()

    def processNotification(self, sender, *values, **args):
        '''
        Call handlers of notifications by type

        @param sender: source of notification
        @type sender: Observable
        '''
        notifyType = args.get('type')
        self.__signalHandlers.get(notifyType)(*values)
        if notifyType in ['childNodeAdded', 'childNodeRemoved', 'nodeMoved',
                'inheritedAttributeChanged']:
            self.curMapChanged.emit()

    def removeNodeDelegate(self, node):
        '''
        Remove the delegate that contains the specified node
        @type node: Node
        '''
        delegate = self.__nodeToDelegate[node]
        if node.isAncestorOf(self.__currentNode):
            self.__nodeToDelegate[node.parent].setLastChild(None, node.left)
            self.currentNode = node.parent
        self._removeNodeDelegateSubtree(node)
        self.removeItem(delegate)
        self.positionNodes()

    def _removeNodeDelegateSubtree(self, root):
        '''
        Remove the node and its children from the node-to-delegate dictionary
        @type root: Node
        '''
        for child in root.children:
            self._removeNodeDelegateSubtree(child)
        self.__nodeToDelegate.pop(root)

    def relocateNodeDelegate(self, node, dest):
        '''
        Relocate the delegate of the node to the destination

        @param node: transfered node
        @type node: Node

        @param dest: destination node
        @type dest: Node
        '''
        roamingDelegate = self.__nodeToDelegate[node]
        previousParentDelegate = roamingDelegate.parentItem()
        previousParentDelegate.setLastChild(None, not node.left)
        parentDelegate = self.__nodeToDelegate[dest]
        roamingDelegate.setParentItem(parentDelegate)
        if dest.folded: roamingDelegate.hide()
        # node relocation require recalculate position of all nodes
        roamingDelegate.updateNodeBounds()
        self.positionNodes()

    def setMap(self, mindMap):
        '''
        Load new mind map and show it

        @param mindMap: mind map to be shown
        @type mindMap: MindMap
        '''
        if len(self.views()):
            self.views()[0].setViewportUpdateMode(QGraphicsView.NoViewportUpdate)
        # clear old values
        self.__nodeToDelegate.clear()
        for item in self.items(): self.removeItem(item)
        # set new values
        self.__mindMap = mindMap
        self.__rootDelegate = NodeDelegate(mindMap.root)
        self.__nodeToDelegate[mindMap.root] = self.__rootDelegate
        self.addItem(self.__rootDelegate)
        self.createNodeDelegates(self.__rootDelegate)
        self.positionNodes()
        self.currentNode = mindMap.root
        if len(self.views()):
            self.views()[0].setViewportUpdateMode(QGraphicsView.SmartViewportUpdate)
        self.mapChanged.emit()

    def getFilteredMindMap(self):
        '''
        Get filtered mind map
        @rtype: MindMap
        '''
        return MindMap(self.__rootDelegate.getFilteredTree())

    _MAP_MARGIN_WIDTH = 300
    ''' Width of mind map margin '''
    _MAP_MARGIN_HEIGHT = 150
    ''' Height of mind map margin '''

    def _updateSceneRectangle(self):
        '''Set new scene rectangle for the map. Add additional margin.'''
        sceneRect = self.itemsBoundingRect()
        sceneRect.adjust(-self._MAP_MARGIN_WIDTH, -self._MAP_MARGIN_HEIGHT,
                self._MAP_MARGIN_WIDTH, self._MAP_MARGIN_HEIGHT)
        self.setSceneRect(sceneRect)

    def createNodeDelegates(self, parentNodeDelegate):
        '''
        Create node delegates from parent

        @type parentNodeDelegate: NodeDelegate
        '''
        QCoreApplication.instance().processEvents()
        parentNodeDelegate.node.addObserver(self)
        childNodes = parentNodeDelegate.node.children
        for node in childNodes:
            delegate = NodeDelegate(node, parentNodeDelegate)
            self.__nodeToDelegate[node] = delegate
            if parentNodeDelegate.node.folded:
                delegate.hide()
            self.createNodeDelegates(delegate)

    def positionNodes(self, node = None):
        '''
        Position all nodes on the mind map

        @param node: changed node
        @type node: Node
        '''
        self.__rootDelegate.positionTree()
        self.update()
        self._updateSceneRectangle()

    def inheritedAttributeChanged(self, node):
        '''Update node and it's subtree bounding rectangle'''
        self.__nodeToDelegate[node].updateNodeBounds()
        self.positionNodes()
        self.__nodeToDelegate[node].updateSubtree()

    def getDelegate(self, node):
        '''
        @return: delegate for the passed Node or None if no delegate found
        @rtype: NodeDelegate
        '''
        return self.__nodeToDelegate.get(node)

    def itemAt(self, position):
        '''
        @type position: QPointF
        @return: item at specified coordinates or None if no item was found.
        Uses textBoundingRect for accurate choice.
        @rtype: NodeDelegate
        '''
        for item in self.items(position):
            if isinstance(item, NodeDelegate) and item.textRectOnScene().contains(position):
                return item
        return None

    def parentAt(self, position):
        '''
        Return parent at specified coordinates and location relative to the parent.

        @type position: QPointF
        @rtype: NodeDelegate, NodeLocation
        '''
        parent = self._getParent(position)
        if parent is None: return None, None
        above, below = self._neighbourDelegates(position, parent)
        right = MindMapDelegate.isPositionToTheRight(position, self.__rootDelegate)
        if self.__rootDelegate.textRectOnScene().contains(position):
            return parent, NodeLocation(right,
                    MindMapDelegate.isPositionAbove(position, self.__rootDelegate), None)
        nearest = MindMapDelegate.nearestDelegateByOrdinate(position, above, below)
        leftSideOfTheNode = (position.x() < nearest.textRectOnScene().center().x())
        if leftSideOfTheNode and right or not leftSideOfTheNode and not right:
            if below is None:
                insertPos = parent.node.getChildPosition(above.node) + 1
            else:
                insertPos = parent.node.getChildPosition(below.node)
            return parent, NodeLocation(right, None, insertPos)
        else:
            return nearest, NodeLocation(
                    right, MindMapDelegate.isPositionAbove(position, nearest), None)

    def _neighbourDelegates(self, position, parent):
        '''
        Return two children of the parent that are neighbor to the position by y-coordinate

        @type position: QPointF
        @type parent: NodeDelegate
        @rtype: NodeDelegate, NodeDelegate
        '''
        if parent is None: return None, None
        prev = None
        for node in parent.node.children:
            if node.left == MindMapDelegate.isPositionToTheRight(
                    position, self.__rootDelegate): continue
            child = self.getDelegate(node)
            if prev is None:
                if position.y() <= child.textRectOnScene().center().y():
                    return None, child
            elif position.y() >= prev.textRectOnScene().center().y() and \
                    position.y() <= child.textRectOnScene().center().y():
                return prev, child
            prev = child
        return prev, None

    @classmethod
    def nearestDelegateByOrdinate(cls, position, above, below):
        '''
        Return one of the delegates that are closer to the position by y-coordinate

        @type position: QPointF
        @type above: NodeDelegate
        @type below: NodeDelegate
        @rtype: NodeDelegate
        '''
        if above is None:
            return below
        if below is None:
            return above
        return above if above.yDistanceTo(position) < below.yDistanceTo(position) else below

    @classmethod
    def nearestDelegateByAbscissa(cls, position, delegates):
        '''
        Return one of the delegates that are closer to the position by x-coordinate

        @type position: QPointF
        @param delegates: list of the delegates
        @type delegates: list
        @rtype: NodeDelegate
        '''
        if delegates == []: return None
        nearest = delegates[0]
        for item in delegates:
            if item is None: continue
            if nearest.xDistanceTo(position) > item.xDistanceTo(position):
                nearest = item
        return nearest

    def _getParent(self, position):
        '''
        Return parent for the position on scene

        @type position: QPointF
        @rtype: NodeDelegate
        '''
        children = self.items(position)
        if children is None: return None
        parents = set()
        for child in children:
            if (child.parentItem()) is not None and isinstance(child, NodeDelegate):
                parents.add(child.parentItem())
            else:
                parents.add(self.__rootDelegate)
        parent = MindMapDelegate.nearestDelegateByAbscissa(position, list(parents))
        return parent

    @classmethod
    def isPositionToTheRight(cls, position, item):
        '''
        Determine if the given position is to the right from the item

        @type position: QPointF
        @type item: NodeDelegate
        @rtype: bool
        '''
        return position.x() > item.textRectOnScene().center().x()

    @classmethod
    def isPositionAbove(cls, position, item):
        '''
        Determine if the given position is above the item

        @type position: QPointF
        @type item: NodeDelegate
        @rtype: bool
        '''
        return position.y() < item.textRectOnScene().center().y()

    def _selectRootNode(self):
        '''Select root node'''
        self.currentNode = self.rootDelegate.node

    def _selectLeftNode(self):
        '''Select left node'''
        self._selectHorizontalNode(left = True)

    def _selectRightNode(self):
        '''Select right node'''
        self._selectHorizontalNode(left = False)

    def _selectUpNode(self):
        '''Select up node'''
        self._selectVerticalNode(up = True)

    def _selectDownNode(self):
        '''Select down node'''
        self._selectVerticalNode(up = False)

    def _sortNodesByDistance(self, nodes, currentNode):
        '''
        Create list of nodes sorted by y-distance relative to the current node
        @param nodes: list of nodes for sorting
        @type currentNode: Node
        @return: sorted list of nodes
        '''
        def compare(first, second):
            '''Compare nodes by y position'''
            firstPos = self.getDelegate(first).textRectOnScene().center().y()
            secondPos = self.getDelegate(second).textRectOnScene().center().y()
            return int(abs(firstPos - curPos) - abs(secondPos - curPos))

        currentDelegate = self.getDelegate(currentNode)
        curPos = currentDelegate.textRectOnScene().center().y()
        return sorted(nodes, cmp = compare)

    def _selectHorizontalNode(self, left):
        '''
        Select node in horizontal direction
        @param left: select left node if True.
        '''
        if self.currentNode.left != left and self.currentNode.parent is not None:
            parentNodeDelegate = self.__nodeToDelegate[self.currentNode.parent]
            parentNodeDelegate.setLastChild(self.currentNode, self.currentNode.left)
            self.currentNode = self.currentNode.parent
            return
        if not self.currentNode.children or self.currentNode.folded:
            return
        lastChild = self.__nodeToDelegate[self.currentNode].getLastChild(left)
        if lastChild:
            self.currentNode = lastChild
            return
        self._selectNearestChildNode(left)

    def _selectNearestChildNode(self, left):
        '''
        Select nearest child node of current selected node
        @param left: if true search node in left subtree, search in right subtree otherwise
        '''
        nodeMap = self.__nodeMapCreator.createNodeMap(self.rootDelegate.node)
        nodes = nodeMap[self.currentNode.depth() + (-1 if left else 1)]
        nodes = self._sortNodesByDistance(nodes, self.currentNode)
        for node in nodes:
            if not self.currentNode.isAncestorOf(node): continue
            if self.getDelegate(node).isVisible():
                self.currentNode = node
                return

    def _selectVerticalNode(self, up):
        '''
        Select node in vertical direction
        @param up: if True select node above otherwise below
        '''
        nodeMap = self.__nodeMapCreator.createNodeMap(self.rootDelegate.node)
        depth = self.currentNode.depth()
        nodes = nodeMap[depth]
        pos = nodes.index(self.currentNode)
        prevNode = self.currentNode
        movement = -1 if up else 1
        reduceDepth = 1 if prevNode.left else -1
        pos += movement
        while True :
            if pos not in range(0, len(nodes)):
                if depth == 0 or abs(depth) == 1:
                    return
                depth += reduceDepth
                nodes = nodeMap[depth]
                pos = nodes.index(prevNode.parent) + movement
                prevNode = prevNode.parent
                continue
            if self.getDelegate(nodes[pos]).isVisible():
                self.currentNode = nodes[pos]
                return
            prevNode = nodes[pos]
            pos += movement


class NodeDelegate(QGraphicsItem):
    '''
    Node delegate

    @authors: Andrew Golovchenko, Andrew Vasilev, Alexander Kulikov
    '''
    readable('node', 'verticalSpace')

    def _isSelected(self):
        '''Get selected flag'''
        return self.__selected

    def _setSelected(self, selected):
        '''
        Set the selected flag
        @type selected: boolean
        '''
        self.__selected = selected
        self.update()

    selected = property(_isSelected, _setSelected, None, 'Selected flag')

    def _isHighlighted(self):
        '''Get highlighted flag'''
        return self.__highlighted

    def _setHighlighted(self, highlighted):
        '''
        Set the highlighted flag
        @type highlighted: boolean
        '''
        self.__highlighted = highlighted
        self.update()

    highlighted = property(_isHighlighted, _setHighlighted, None, 'Highlighted flag')

    def _isMatched(self):
        '''Get filtered flag'''
        return self.__matched

    def _setMatched(self, matched):
        '''
        Set the filtered flag
        @type matched: boolean
        '''
        self.__matched = matched

    matched = property(_isMatched, _setMatched, None, 'Matched flag')

    def __init__(self, node, parent = None):
        '''
        Initialize delegate with specified node

        @param node: delegated Node object
        @type node: Node
        @param parent: parent object of this delegate
        @type parent: NodeDelegate
        '''
        self.__node = node
        self.__verticalSpace = 0
        self.__highlighted = False
        self.__matched = True
        self.__selected = False
        self.__positionChanged = True
        self.__textBoundingRect = None
        self.__lastChild = None if node.parent else [None, None]
        self.__boundingRect = None
        QGraphicsItem.__init__(self, parent)
        self.__edge = Edge(self.__node, self)
        self.setCacheMode(QGraphicsItem.DeviceCoordinateCache)
        self.updateNodeBounds(False)
        self._updateNodePosition()

    __NODE_VERTICAL_SPACING = 4
    '''Minimal vertical space required for node decorations and spacing'''

    __NODE_HORIZONTAL_SPACING = 50
    '''Space between the node and its parent'''

    __ICON_SIZE = 16
    '''Icon width and height'''

    __LINK_SIZE = 24
    '''Link icon width and height'''

    __TEXT_MARGIN = 4.0
    '''Default node content margin. It is never changed manually'''

    __FOLD_MARK_SIZE = 4.0
    '''Radius of round node folded mark'''

    __LABEL_MARGIN = 4.0
    '''Horisontal margin of edge label'''

    __MAX_ROOT_ECCENTRICITY_VERTICALLY = 0.98
    '''Maximum eccentricity of root node if it is stretched vertically'''

    __MAX_ROOT_ECCENTRICITY_HORIZONTALLY = 0.95
    '''Maximum eccentricity of root node if it is stretched horizontally'''

    def positionTree(self):
        '''
        Position all subtree nodes of the current node according to their requirements
        '''
        self._computeSoloVerticalSpace()
        for leftSubtree in [True, False]:
            if self.__node.folded: continue
            subtree = self._subtree(leftSubtree)
            if not subtree: continue
            for child in subtree:
                child.positionTree()
            self._positionChildNodes(subtree)

    def _subtree(self, left):
        '''
        Get list of child node delegates that belong to specified tree.
        @param left: subtree position
        @type left: bool
        @return: list of child delegates that belong to specified tree
        @rtype: list
        '''
        subtree = [item for item in self._childDelegates() if item.node.left is left]
        subtree.sort(key = lambda delegate: self.__node.children.index(delegate.node))
        return subtree

    def _childDelegates(self):
        '''Get all child node delegates of the current delegate, filter out other elements'''
        return [item for item in self.childItems() if isinstance(item, NodeDelegate)]

    def getFilteredTree(self):
        '''
        Create a clone of the current node and it's filtered subtree
        @rtype: Node
        '''
        root = self.node.clone()
        for child in self._childDelegates():
            if child.matched: root.addChild(child.getFilteredTree())
        return root

    def updateSubtree(self):
        '''
        Update view of the delegate and it subtree
        '''
        self.update()
        for child in self._childDelegates():
            child.updateSubtree()

    def _computeSoloVerticalSpace(self):
        '''
        Compute minimum vertical space needed for the current node
        '''
        self.__verticalSpace = self.textBoundingRect().height() + self._labelOverhead() + \
            NodeDelegate.__NODE_VERTICAL_SPACING
        if self.__node.folded and self.__node.effectiveStyle == 'fork':
            self.__verticalSpace += NodeDelegate.__FOLD_MARK_SIZE

    @classmethod
    def computeVerticalSpace(cls, subtree):
        '''
        Calculate vertical space needed for current node specified subtree

        @param subtree: left or right subtree of the node
        @type subtree: list

        @return: required vertical space for the current node subtree
        @rtype: float
        '''
        subtreeSpace = 0.0
        for child in subtree:
            subtreeSpace += child.verticalSpace
        return subtreeSpace

    def _childHorizontalPosition(self, child):
        '''
        Calculate horizontal position of the child

        @param child: child node of current parent
        @type child: NodeDelegate

        @return: horizontal position of the passed delegate
        @rtype: float
        '''
        if child.node.left:
            return -NodeDelegate.__NODE_HORIZONTAL_SPACING \
                - child.textBoundingRect().width() - child.linkPointOffset()
        else:
            return NodeDelegate.__NODE_HORIZONTAL_SPACING \
                + self.textBoundingRect().width() - child.linkPointOffset()

    def _positionChildNodes(self, nodes):
        '''
        Position child node delegates of current node delegate

        @param nodes: child nodes to position
        @type nodes: list of NodeDelgates
        '''
        #pylint: disable=W0212
        baseVerticalPosition = (self.textBoundingRect().height() - \
            NodeDelegate.computeVerticalSpace(nodes)) / 2.0
        initialVerticalPosition = baseVerticalPosition
        for child in nodes:
            verticalPosition = baseVerticalPosition + (child.verticalSpace - \
                child.textBoundingRect().height() + child._labelOverhead()) / 2.0
            horizontalPosition = self._childHorizontalPosition(child)
            child.setPos(horizontalPosition, verticalPosition)
            baseVerticalPosition += child.verticalSpace
        self._computeSoloVerticalSpace()
        self.__verticalSpace = max(self.__verticalSpace, baseVerticalPosition -
            initialVerticalPosition)

    def setPos(self, x, y):
        '''
        Overloaded method. Set new position of the item
        @type x: float
        @type y: float
        '''
        if self.x() != x or self.y() != y:
            QGraphicsItem.setPos(self, x, y)
            self._updateNodePosition()
            self.update()

    def _updateNodePosition(self):
        '''Calculate new node position-aware properties'''
        if self.parentItem() is not None:
            self.__edge.calculateParentLinkPoint()
        self._calculateBoundingRect()

    def updateNodeBounds(self, propagate = True):
        '''
        Calculate new bounds for the given node
        @param propagate: propagate border calculation on children or not
        @type propagate: bool
        '''
        self._calculateTextBoundingRect()
        self.__edge.calculateLinkPoint()
        self._updateNodePosition()
        if propagate:
            for delegate in self._childDelegates():
                delegate.updateNodeBounds(self)

    def paint(self, painter, option, widget):
        '''
        Paint node content and its joint link
        @type painter: QPainter
        '''
        # calculate bounding rect (without joint link)
        boundRect = self.textBoundingRect()
        # join link color
        edgeColor = self.__node.effectiveEdgeColor
        # join link width
        edgeWidth = self.__node.effectiveEdgeWidth
        # join link shape style
        edgeStyle = self.__node.effectiveEdgeStyle
        # node background color
        bgColor = self.__node.effectiveBackgroundColor
        if self.__highlighted:
            bgColor = settings.get('defaultSelectedNodeBackgroundColor').lighter(60)
        if self.__selected:
            bgColor = settings.get('defaultSelectedNodeBackgroundColor')
        painter.setPen(QPen(edgeColor, 1.0))
        painter.setBrush(bgColor)
        # if root node
        if self.__node.parent is None:
            self._paintRoot(painter, boundRect)
            return
        # calculate self link point and paint shape
        if self.__node.effectiveStyle == 'bubble':
            painter.drawRoundedRect(boundRect, 5.0, 5.0)
        else:
            boundRect.setHeight(boundRect.height() - edgeWidth / 2.0)
            painter.fillRect(boundRect, bgColor)
            # some decorative setting
            if edgeStyle.startswith('sharp') and not self.__node.children: edgeWidth = 0.5
            painter.setPen(QPen(edgeColor, edgeWidth, Qt.SolidLine, Qt.RoundCap))
            painter.drawLine(boundRect.bottomLeft(), boundRect.bottomRight())
        # paint edge label
        if self.__node.labelText:
            self._paintLabel(painter)
        # disable brush to prevent color filling
        painter.setBrush(Qt.NoBrush)
        # paint icons and text content
        self._paintNodeContent(painter)
        # paint folded mark (circle)
        if self.__node.folded:
            self._paintFoldedMark(painter, boundRect)

    def _paintLabel(self, painter):
        ''' Paint text label of node'''
        painter.setFont(self.__node.effectiveLabelFont)
        painter.setPen(self.__node.effectiveLabelColor)
        labelPoint = QPointF(self.__edge.linkPoint)
        if self.__node.left:
            labelPoint.setX(labelPoint.x() - self.linkPointOffset() \
                     + NodeDelegate.__LABEL_MARGIN)
        else:
            labelPoint.setX(labelPoint.x() - self.linkPointOffset() \
                     - self.__node.labelSize.width() - NodeDelegate.__LABEL_MARGIN)
        labelPoint.setY(labelPoint.y() - self.__node.effectiveEdgeWidth / 2.0 - 2.0)
        painter.drawText(labelPoint, self.__node.labelText)
        pen = QPen(self.__node.effectiveEdgeColor, self.__node.effectiveEdgeWidth,
                Qt.SolidLine, Qt.RoundCap)
        if self.__node.effectiveEdgeStyle.startswith('sharp') and not self.__node.children:
            pen.setWidthF(0.5)
        painter.setPen(pen)
        linkPoint = QPointF(self.__edge.linkPoint)
        painter.drawLine(QPointF(linkPoint.x() - self.linkPointOffset(), linkPoint.y())
                , linkPoint)

    def _paintRoot(self, painter, boundRect):
        '''
        Paint root node

        @type painter: QPanter

        @param boundRect: bounding rect of root node
        @type boundRect: QRectF
        '''
        painter.drawEllipse(boundRect)
        if self.__node.folded:
            self._paintFoldedMark(painter, boundRect)
        self._paintNodeContent(painter)

    def _paintFoldedMark(self, painter, boundRect):
        '''
        Paint folded mark
        '''
        painter.setPen(QPen(self.__node.effectiveEdgeColor, 1.0))
        painter.setBrush(Qt.white)
        if self.__edge.linkPoint is None:
            markPoint = None
        else:
            markPoint = QPointF(self.__edge.linkPoint)
        if markPoint is None: # root
            markPoint = QPointF()
            markPoint.setY(boundRect.height() / 2.0)
            if any(not node.left for node in self.__node.children):
                markPoint.setX(boundRect.width())
                painter.drawEllipse(markPoint, 4.0, 4.0)
            if any(node.left for node in self.__node.children):
                markPoint.setX(0.0)
        else: # child node
            markPoint.setX(0.0 if markPoint.x() - self.linkPointOffset() \
                     else boundRect.width())
        painter.drawEllipse(markPoint, 4.0, 4.0)

    def _rootBoundingRect(self, size):
        '''
        Calculate bounding rectangle of the root node

        @type size: QSize

        @return: Root's bounding rect
        @rtype: QRectF
        '''
        #pylint: disable=C0103
        # 'a' and 'b' - sides of ellipse; 'e' - eccentricity
        b = size.height()
        a = size.width()
        if b > a:
            e = math.sqrt(1 - pow(a / b, 2))
            if e > self.__MAX_ROOT_ECCENTRICITY_VERTICALLY:
                e = self.__MAX_ROOT_ECCENTRICITY_VERTICALLY
            yAxis = math.sqrt(b * b + a * a / (1 - e * e))
            xAxis = yAxis * math.sqrt(1 - e * e)
        else:
            e = math.sqrt(1 - pow(b / a, 2))
            if e > self.__MAX_ROOT_ECCENTRICITY_HORIZONTALLY:
                e = self.__MAX_ROOT_ECCENTRICITY_HORIZONTALLY
            xAxis = math.sqrt(a * a + b * b / (1 - e * e))
            yAxis = xAxis * math.sqrt(1 - e * e)
        return QRectF(0.0, 0.0, xAxis, yAxis)

    def _rootTextPosition(self):
        '''
        Calculate the start position of the root text

        @rtype: QPointF
        '''
        boundRect = self.textBoundingRect()
        return QPointF((boundRect.width() -
                        len(self.__node.icons) * NodeDelegate.__ICON_SIZE -
                        self.__node.textSize().width() -
                        (NodeDelegate.__LINK_SIZE if self.__node.link else 0)) / 2.0,
                        (boundRect.height() - self.__node.textSize().height()) / 2.0)

    def _paintNodeContent(self, painter):
        '''Paint internal content of the node'''
        painter.save()
        if not self.parentItem(): painter.translate(self._rootTextPosition())
        # folded mark offset
        if self.__node.folded and self.__node.left:
            painter.translate(NodeDelegate.__FOLD_MARK_SIZE, 0)
        # paint icons
        xIconPos = NodeDelegate.__TEXT_MARGIN
        yIconPos = (self.__node.textSize().height() - NodeDelegate.__ICON_SIZE) / 2.0
        for icon in self.__node.icons:
            svgIcon = resource.getSvgIconByName(icon)
            if svgIcon:
                svgIcon.render(painter, QRectF(xIconPos, yIconPos,
                        NodeDelegate.__ICON_SIZE, NodeDelegate.__ICON_SIZE))
            else:
                painter.setRenderHint(QPainter.SmoothPixmapTransform)
                painter.drawImage(QRectF(xIconPos, yIconPos,
                        NodeDelegate.__ICON_SIZE, NodeDelegate.__ICON_SIZE),
                        resource.getIconByName(icon))
            xIconPos += NodeDelegate.__ICON_SIZE
        # paint text content
        if xIconPos > NodeDelegate.__TEXT_MARGIN:
            painter.translate(xIconPos, 0)
        self.__node.paint(painter)
        # paint link icon
        if self.__node.link:
            xIconPos = self.__node.textSize().width()
            yIconPos = (self.__node.textSize().height() - NodeDelegate.__LINK_SIZE) / 2.0
            painter.drawImage(QRectF(xIconPos, yIconPos, NodeDelegate.__LINK_SIZE,
                    NodeDelegate.__LINK_SIZE),
                    resource.getImage('link').toImage())
        painter.restore()

    def _calculateTextBoundingRect(self):
        '''Calculate bounding region only for node without link to the parent'''
        size = self.__node.textSize()
        if self.__node.icons:
            size.setWidth(size.width() + NodeDelegate.__TEXT_MARGIN
                    + len(self.__node.icons) * NodeDelegate.__ICON_SIZE)
            size.setHeight(max(size.height(), NodeDelegate.__ICON_SIZE))
        if self.__node.link:
            size.setWidth(size.width() + NodeDelegate.__LINK_SIZE +
                    NodeDelegate.__TEXT_MARGIN)
            size.setHeight(max(size.height(), NodeDelegate.__LINK_SIZE))
        if not self.parentItem():
            self.__textBoundingRect = self._rootBoundingRect(size)
            return
        if self.__node.folded: size.setWidth(size.width() + NodeDelegate.__FOLD_MARK_SIZE)
        boundRect = QRectF(0.0, 0.0, size.width(), size.height())
        if self.__node.effectiveStyle == 'fork':
            boundRect.setHeight(boundRect.height() + self.__node.effectiveEdgeWidth / 2.0)
        self.__textBoundingRect = boundRect

    def textBoundingRect(self):
        '''Text bounding rectangle of the node'''
        return QRectF(self.__textBoundingRect)

    def _calculateBoundingRect(self):
        '''Calculate bounding rectangle'''
        if self.parentItem() is None:
            self.__boundingRect = self.textBoundingRect()
            if self.__node.folded:
                self.__boundingRect.setX(self.__boundingRect.x() \
                         - NodeDelegate.__FOLD_MARK_SIZE)
                self.__boundingRect.setWidth(self.__boundingRect.width() \
                         + NodeDelegate.__FOLD_MARK_SIZE * 2)
            return
        textBounds = self.textBoundingRect()
        left = textBounds.x()
        if not self.__node.left:
            left += self.linkPointOffset()
        top = min(textBounds.y(), -self._labelOverhead())
        width = textBounds.width() + abs(self.linkPointOffset())
        if self.__node.folded and self.__node.left: left -= NodeDelegate.__FOLD_MARK_SIZE
        height = textBounds.height() + self._labelOverhead()
        if self.__node.folded:
            width += NodeDelegate.__FOLD_MARK_SIZE
            if self.__node.effectiveStyle == 'fork': height += NodeDelegate.__FOLD_MARK_SIZE
        self.__boundingRect = QRectF(left, top, width, height)

    def boundingRect(self):
        '''Bounding rectangle of the node'''
        return self.__boundingRect

    def textRectOnScene(self):
        '''
        Return text bounding rectangle in scene coordinates

        @rtype: QRectF
        '''
        return self.mapRectToScene(self.textBoundingRect())

    def xDistanceTo(self, position):
        '''
        Return absolute distance by x-coordinate from center of the delegate
        to the position on scene

        @type position: QPointF
        @rtype: float
        '''
        return abs(self.textRectOnScene().center().x() - position.x())

    def yDistanceTo(self, position):
        '''
        Return absolute distance by y-coordinate from center of the delegate
        to the position on scene

        @type position: QPointF
        @rtype: float
        '''
        return abs(self.textRectOnScene().center().y() - position.y())

    def boundingRectOnScene(self):
        '''
        Return bounding rectangle in scene coordinates

        @rtype: QRectF
        '''
        return self.mapRectToScene(self.boundingRect())

    def _labelOverhead(self):
        '''
        @return: difference between top points value of node (text bounding rect) and its label
        @rtype: float
        '''
        if not self.__node.labelText: return 0.0
        textBounds = self.textBoundingRect()
        # 2.0 is a decorative space between label and edge
        labelTop = textBounds.height() - self.__node.labelSize.height() - \
                self.__node.effectiveEdgeWidth / 2.0 - 2.0
        if self.__node.effectiveStyle == 'bubble':
            labelTop -= textBounds.height() / 2.0
        return -labelTop if labelTop < 0.0 else 0.0

    def linkPointOffset(self):
        '''
        @return: offset of edge link point in pixels specified
                 by label width and its margins
        @rtype: float
        '''
        labelWidth = self.__node.labelSize.width()
        if labelWidth == 0.0:
            return 0.0
        offset = labelWidth + NodeDelegate.__LABEL_MARGIN * 2.0
        return offset if self.__node.left else -offset

    def getLastChild(self, leftDirection):
        '''
        Retrieve last selected child on specified direction
        @param leftDirection: indication whether child was on the left or right
        @type leftDirection: bool
        @return: link to the child node
        @rtype: Node
        '''
        lastChild = None
        if self.__node.parent is None:
            if leftDirection:
                lastChild = self.__lastChild[0]
            else:
                lastChild = self.__lastChild[1]
        else:
            lastChild = self.__lastChild
        if lastChild not in self.__node.children: lastChild = None
        return lastChild

    def setLastChild(self, child, left):
        '''
        Set this node as a last child
        @param child: child node that was selected before this node
        @type child: Node
        @param left: indicator that child is in left subtree
        @type left: bool
        '''
        if self.__node.parent is None:
            if left:
                self.__lastChild[0] = child
            else:
                self.__lastChild[1] = child
        else:
            self.__lastChild = child


class Edge(QGraphicsItem):
    '''
    Edge delegate

    @authors: Andrew Golovchenko, Andrew Vasilev, Alexander Kulikov, Eldar Mamedov
    '''

    readable('parentLinkPoint', 'linkPoint')

    def __init__(self, node, parent = None):
        '''
        Initialize delegate with specified node

        @param node: delegated Node object
        @type node: Node
        @param parent: parent object of this delegate
        @type parent: NodeDelegate
        '''
        self.__node = node
        self.__parentLinkPoint = None
        self.__linkPoint = None
        QGraphicsItem.__init__(self, parent)

    def calculateParentLinkPoint(self):
        '''Calculate connection point to parent node'''
        parentLinkPoint = QPointF()
        nodeDelegate = self.parentItem()
        parentRect = nodeDelegate.parentItem().textBoundingRect()
        if not self.__node.left: parentLinkPoint.setX(parentRect.width())
        parentNode = nodeDelegate.parentItem().node
        # root node never has fork style and only root has no parent
        if parentNode.effectiveStyle == 'bubble' or parentNode.parent is None:
            parentLinkPoint.setY(parentRect.height() / 2.0)
        else:
            parentRect.setHeight(parentRect.height() - parentNode.effectiveEdgeWidth / 2.0)
            parentLinkPoint.setY(parentRect.height())
        self.__parentLinkPoint = nodeDelegate.mapFromParent(parentLinkPoint)

    def calculateLinkPoint(self):
        '''Calculate connection point for joint link'''
        if self.__node.parent is None:
            self.__linkPoint = None
            return
        linkPoint = QPointF()
        nodeDelegate = self.parentItem()
        if self.__node.effectiveStyle == 'bubble':
            linkPoint.setY(nodeDelegate.textBoundingRect().height() / 2.0)
        else:
            linkPoint.setY(nodeDelegate.textBoundingRect().height() -
                           self.__node.effectiveEdgeWidth / 2.0)
        linkPoint.setX(nodeDelegate.linkPointOffset())
        if self.__node.left:
            linkPoint.setX(linkPoint.x() + nodeDelegate.textBoundingRect().width())
        self.__linkPoint = linkPoint

    def paint(self, painter, option, widget):
        '''
        Paint join link from parent node to current
        @type painter: QPanter
        @param style: style of node join link
        @type style: string
        '''
        if self.__linkPoint is None: return
        paintEdge = {'bezier': self._paintBezierEdge,
                'sharp_bezier': self._paintSharpBezierEdge,
                'linear': self._paintLinearEdge,
                'sharp_linear': self._paintSharpLinearEdge}
        painter.setPen(QPen(self.__node.effectiveEdgeColor, self.__node.effectiveEdgeWidth,
                Qt.SolidLine, Qt.FlatCap, Qt.RoundJoin))
        startPoint = QPointF(self.__parentLinkPoint)
        endPoint = QPointF(self.__linkPoint)
        paintEdge[self.__node.effectiveEdgeStyle](painter, startPoint, endPoint)

    def _paintBezierEdge(self, painter, startPoint, endPoint):
        '''
        Paint edge with bezier style
        '''
        #pylint: disable=R0201
        curvePath = QPainterPath()
        curveRect = QRectF(startPoint, endPoint)
        p1 = curveRect.topRight()
        p1.setX(curveRect.topRight().x() - curveRect.width() / 2.0)
        p2 = curveRect.bottomLeft()
        p2.setX(curveRect.bottomLeft().x() + curveRect.width() / 2.0)
        curvePath.moveTo(startPoint)
        curvePath.cubicTo(p1, p2, endPoint)
        painter.drawPath(curvePath)

    def _paintSharpBezierEdge(self, painter, startPoint, endPoint):
        '''
        Paint edge with sharp bezier style
        '''
        curvePath = QPainterPath()
        painter.setPen(QPen(self.__node.effectiveEdgeColor, 0.5))
        curveRect = QRectF(startPoint, endPoint)
        p1 = curveRect.topRight()
        p1.setX(curveRect.topRight().x() - curveRect.width() / 2.0)
        p2 = curveRect.bottomLeft()
        p2.setX(curveRect.bottomLeft().x() + curveRect.width() / 2.0)
        curvePath.moveTo(startPoint.x(), startPoint.y() + \
                self.__node.effectiveEdgeWidth / 2.0)
        curvePath.cubicTo(p1, p2, endPoint)
        startPoint.setY(startPoint.y() - self.__node.effectiveEdgeWidth / 2.0)
        curvePath.cubicTo(p2, p1, startPoint)
        painter.drawPath(curvePath)
        painter.fillPath(curvePath, self.__node.effectiveEdgeColor)

    def _paintLinearEdge(self, painter, startPoint, endPoint):
        '''
        Paint edge with linear style
        '''
        if self.__node.left:
            startPoint, endPoint = endPoint, startPoint
        points = QPolygonF()
        points.append(startPoint)
        points.append(QPointF(startPoint.x() + self.__node.effectiveEdgeWidth / 2.0,
                startPoint.y()))
        points.append(QPointF(endPoint.x() - self.__node.effectiveEdgeWidth / 2.0,
                endPoint.y()))
        points.append(endPoint)
        painter.drawPolyline(points)

    def _paintSharpLinearEdge(self, painter, startPoint, endPoint):
        '''
        Paint edge with sharp linear style
        '''
        curvePath = QPainterPath()
        painter.setPen(QPen(self.__node.effectiveEdgeColor, 0.1))
        curvePath.moveTo(startPoint.x(), startPoint.y() + \
                self.__node.effectiveEdgeWidth / 2.0)
        curvePath.lineTo(endPoint)
        curvePath.lineTo(startPoint.x(), startPoint.y() - \
                self.__node.effectiveEdgeWidth / 2.0)
        painter.drawPath(curvePath)
        painter.fillPath(curvePath, self.__node.effectiveEdgeColor)

    def boundingRect(self):
        '''Bounding rectangle of the edge'''
        if self.__linkPoint is None: return QRectF()
        x = min(self.__parentLinkPoint.x(), self.__linkPoint.x())
        y = min(self.__parentLinkPoint.y(), self.__linkPoint.y())
        width = abs(self.__parentLinkPoint.x() - self.__linkPoint.x())
        height = abs(self.__parentLinkPoint.y() - self.__linkPoint.y())
        return QRectF(x, y, width, height)


class NodeLocation(object):
    '''
    Class for holding position of a node

    @author: Oleg Kandaurov
    '''
    writable('right', 'top', 'position')

    def __init__(self, right = None, top = None, position = None):
        '''
        @param right: horizontal position. True if right position, otherwise False
        @type right: bool

        @param top: vertical position. True if top position, otherwise False
        @type top: bool

        @param position: vertical order
        @type position: int
        '''
        self.__right = right
        self.__top = top
        self.__position = position

    @property
    def valid(self):
        '''
        True if all the data necessary to use this object is valid, otherwise False.

        @rtype: bool
        '''
        return self.__position is not None or self.__right is not None


class NodeMapCreator(QObject):
    '''
    Create and process various node containers
    @author: Oleg Kandaurov
    '''

    def __init__(self):
        '''Constructor'''
        QObject.__init__(self)

    def createNodeMap(self, rootNode):
        '''
        Create dict containing all node of the map.
        Format is {0: [n0], 1: [n1, n2], -1: [n3, n4], 2:[...] ...}
        @param rootNode: Root node of the mindmap
        @type rootNode: Node
        @return: dict of Nodes
        '''
        nodeMap = {}
        self._extendNodeMap(rootNode, 0, nodeMap)
        return nodeMap

    @classmethod
    def includeParentNodes(cls, nodeMap):
        '''
        Include parents of all nodes in the nodeMap
        @param nodeMap: Node map that need to be processed
        '''
        depths = nodeMap.keys()
        depths.sort(lambda x, y: cmp (abs(y), abs(x))) # 4 -3 3 traverse from edges of mindmap
        for depth in depths:
            for node in nodeMap[depth]:
                upperLevel = depth + 1 if node.left else depth - 1
                if upperLevel not in depths:
                    nodeMap[upperLevel] = []
                if node.parent not in nodeMap[upperLevel]:
                    nodeMap[upperLevel].append(node.parent)

    @classmethod
    def createSortedNodeList(cls, nodeMap):
        '''
        Create list of nodes sorted from left to right
        @param nodeMap:
        @return: list of nodes
        '''
        nodes = []
        keys = nodeMap.keys()
        keys.sort()
        for key in keys:
            nodes.extend(nodeMap[key])
        return nodes

    @classmethod
    def createFilteredNodeMap(cls, nodeMap, condition):
        '''
        Create new node map from the given one
        @param nodeMap: node map
        @param condition: function that returns True if node matches a condition
        @return: filtered node map
        '''
        filteredNodeMap = {}
        for depth in nodeMap:
            for node in nodeMap[depth]:
                if condition(node):
                    if depth not in filteredNodeMap:
                        filteredNodeMap[depth] = []
                    filteredNodeMap[depth].append(node)
        return filteredNodeMap

    def _extendNodeMap(self, node, depth, nodeMap):
        '''
        Append nodes of given depth to the node map
        @type node: Node
        @type depth: int
        '''
        if depth not in nodeMap:
            nodeMap[depth] = []
        nodeMap[depth].append(node)
        for childrenNode in node.children:
            deeperLevel = depth - 1 if childrenNode.left else depth + 1
            self._extendNodeMap(childrenNode, deeperLevel, nodeMap)


class MindMapFilter(QObject):
    '''
    Filter mind map
    @author: Oleg Kandaurov
    '''

    writable('mode', 'text', 'icons')

    def __init__(self, mindMapScene):
        '''Constructor'''
        QObject.__init__(self)
        self.__mindMapScene = mindMapScene
        self.__nodemapCreator = self.__mindMapScene.nodeMapCreator
        self.__foundNodes = []
        self.__isfilterModeEnabled = False
        self.__text = None
        self.__icons = []
        self.__mode = None
        self.__mindMapScene.curMapChanged.connect(self.refresh)
        self.__mindMapScene.mapChanged.connect(self.refresh)

    def _isMatchTextCondition(self, node):
        '''
        Decide whether node match a condition or not
        @param node: Node to be inspected
        @return: True if node matches a condition
        '''
        if self.__text is None or not len(self.__text):
            return True
        nodeText = QTextDocument()
        nodeText.setHtml(node.text)
        if unicode(self.__text.toLower()) in unicode(nodeText.toPlainText().toLower()):
            return True
        return False

    def _isMatchIconCondition(self, node):
        '''
        Decide whether node match a condition or not
        @param node: Node to be inspected
        @return: True if node matches a condition
        '''
        if not len(self.__icons):
            return True
        for icon in node.icons:
            if icon in self.__icons:
                return True
        return False

    def _setMatchedProperty(self, node):
        '''
        Traverses through all children nodes and sets "matched" property
        @param node: Parent node
        '''
        for node in node.children:
            nodeDelegate = self.__mindMapScene.getDelegate(node)
            if nodeDelegate is None: return
            nodeDelegate.matched = ((node in self.__foundNodes)
                    if self.__isfilterModeEnabled else True)
            self._setMatchedProperty(node)

    def refresh(self):
        '''Apply filter to all nodes'''
        if self.__isfilterModeEnabled:
            condition = (self._isMatchTextCondition if self.__mode == 'text'
                    else self._isMatchIconCondition)
            nodeMap = self.__nodemapCreator.createNodeMap(
                    self.__mindMapScene.rootDelegate.node)
            filteredNodeMap = self.__nodemapCreator.createFilteredNodeMap(nodeMap, condition)
            self.__nodemapCreator.includeParentNodes(filteredNodeMap)
            self.__foundNodes = self.__nodemapCreator.createSortedNodeList(filteredNodeMap)
        self._setMatchedProperty(self.__mindMapScene.rootDelegate.node)
        self.__mindMapScene.calculateVisibilty(self.__mindMapScene.rootDelegate.node,
                propagate = True)

    def toggleFilter(self, isfilterModeEnabled):
        '''
        Update filter trigger
        @param isfilterModeEnabled: Indicates that filtering is enabled
        '''
        self.__isfilterModeEnabled = isfilterModeEnabled
        self.refresh()


class NodeFinder(QObject):
    '''
    Node finder
    @author: Eldar Mamedov
    '''

    statusChanged = pyqtSignal(str)

    def __init__(self, controller):
        '''
        @type controller: MainWindowController
        '''
        QObject.__init__(self)
        self.__foundNodes = []
        self.__index = -1
        self.__controller = controller
        self.__mindMapScene = controller.mindMapScene
        self.__nodemapCreator = self.__mindMapScene.nodeMapCreator
        self.__findText = None
        self.__caseSensitive = False
        self.__highlight = False
        self.__mindMapScene.curMapChanged.connect(self.refresh)
        self.__mindMapScene.mapChanged.connect(self.refresh)

    def _condition(self, node):
        '''
        Decide whether node match a condition or not
        @param node: Node to be inspected
        @return: True if node matches a condition
        '''
        text = QTextDocument()
        text.setHtml(node.text)

        if self.__caseSensitive:
            if unicode(self.__findText) in unicode(text.toPlainText()):
                return True
        else:
            if unicode(self.__findText.toLower()) in unicode(text.toPlainText().toLower()):
                return True
        return False

    def prevFoundNode(self):
        '''
        Select previous found node
        '''
        if len(self.__foundNodes):
            self.__index -= 1
            if self.__index < 0:
                self.__index = len(self.__foundNodes) - 1
            foundNode = self.__foundNodes[self.__index]
            self.__controller.showNode(foundNode)
            self.__mindMapScene.currentNode = foundNode
        self._status()

    def nextFoundNode(self):
        '''
        Select next found node
        '''
        if len(self.__foundNodes):
            self.__index += 1
            if self.__index == len(self.__foundNodes):
                self.__index = 0
            foundNode = self.__foundNodes[self.__index]
            self.__controller.showNode(foundNode)
            self.__mindMapScene.currentNode = foundNode
        self._status()

    def highlightNodes(self, highlight = False):
        '''
        Highlight found nodes
        @type highlight: bool
        '''
        self.__highlight = highlight
        for node in self.__foundNodes:
            nodeDelegate = self.__mindMapScene.getDelegate(node)
            if nodeDelegate is None: continue
            if highlight:
                nodeDelegate.highlighted = True
            else:
                nodeDelegate.highlighted = False

    def findNodes(self, findText = None, caseSensitive = False, highlight = False):
        '''
        Find nodes
        @type findText: str
        @type caseSensitive: bool
        @type highlight: bool
        '''

        if self.__highlight: self.highlightNodes(False)
        self.__foundNodes = []
        self.__index = -1
        self.__findText = findText
        self.__caseSensitive = caseSensitive
        if findText is None or not len(findText): return
        nodeMap = self.__nodemapCreator.createNodeMap(self.__mindMapScene.rootDelegate.node)
        filteredNodeMap = self.__nodemapCreator.createFilteredNodeMap(nodeMap, self._condition)
        self.__foundNodes = self.__nodemapCreator.createSortedNodeList(filteredNodeMap)
        if highlight: self.highlightNodes(highlight)
        self._status()

    def refresh(self):
        '''
        Refresh list of found nodes
        '''
        self.findNodes(self.__findText, self.__caseSensitive, self.__highlight)

    def _status(self):
        '''
        Generates signal about status
        '''
        if not len(self.__foundNodes):
            self.statusChanged.emit('not_found')
            return
        else:
            if self.__index == len(self.__foundNodes) - 1:
                self.statusChanged.emit('last')
                return
            if self.__index == 0:
                self.statusChanged.emit('first')
                return
        self.statusChanged.emit('none')

