# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Core classes
'''

import hivemind.settings as settings
import time, random
from PyQt4.QtGui import QApplication, QFontMetricsF, QPalette, QTextDocument, \
 QAbstractTextDocumentLayout
from PyQt4.QtCore import QSizeF
from hivemind.attribute import Observable, readable, writable


class Element(object):
    '''
    Base class for mind map objects

    @authors: Andrew Golovchenko
    '''
    #pylint: disable=W0212

    _ATTRIBUTES = ()
    '''List of element attributes'''

    def __init__(self, **attributes):
        '''
        Initialize attributes with specified values and the others to defaults

        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        # class specified prefix of private field
        mangle = '_%s__' % self.__class__.__name__
        # init private fields and set it to default values
        for attr in self.__class__._ATTRIBUTES:
            self.__dict__[mangle + attr] = None
        self.initAttributes(attributes)

    def initAttributes(self, attributes):
        '''
        Set class attributes to specified values names and values

        @type attributes: dict
        '''
        # class specified prefix of private field
        mangle = '_%s__' % self.__class__.__name__
        # replace some of default values with new
        for attrName, attrValue in attributes.items():
            self.__dict__[mangle + attrName.lower()] = attrValue

    @property
    def attributes(self):
        '''
        Return dict of inner element attributes
        @rtype: dict
        '''
        elementAttrs = {}
        mangle = '_%s__' % self.__class__.__name__
        for attr in self.__class__._ATTRIBUTES:
            if self.__dict__[mangle + attr] is not None:
                elementAttrs[attr] = self.__dict__[mangle + attr]
        return elementAttrs


class MindMap(object):
    '''
    Mind map

    @authors: Andrew Golovchenko
    '''
    writable('attributeRegistry')
    readable('root')

    def __init__(self, root = None, attributeRegistry = None):
        '''
        Initialize mind map
        @param root: root node of the mind map
        @type root: Node
        @param attributeRegistry: the list of all available additional attributes of nodes
        @type attributeRegistry: AttributeRegistry
        '''
        self.__attributeRegistry = attributeRegistry
        if root is None:
            root = Node()
            root.setAttributes(text = QApplication.translate('MindMap', 'Root'))
        self.__root = root
        self.__idToNode = {}
        self._registerSubtree(root)

    def processNotification(self, sender, *values, **args):
        '''
        Update the id-to-node dictionary after the node addition/deletion

        @param sender: source of notification
        @type sender: Observable
        '''
        notifyType = args.get('type')
        if notifyType == 'childNodeAdded':
            self._registerSubtree(*values)
        elif notifyType == 'childNodeRemoved':
            self._unregisterSubtree(*values)

    def _registerSubtree(self, root):
        '''Register the node and its children in the id-to-node dictionary'''
        self.__idToNode[root.id] = root
        root.addObserver(self)
        for child in root.children:
            self._registerSubtree(child)

    def _unregisterSubtree(self, root):
        '''Unregister the node and its children in the id-to-node dictionary'''
        self.__idToNode.pop(root.id)
        for child in root.children:
            self._unregisterSubtree(child)

    def getNodeById(self, nodeId):
        '''
        @return: node with the passed id attribute or None if no node found
        @rtype: Node
        '''
        return self.__idToNode.get(nodeId)

    def clone(self):
        '''
        Create a clone of the current mind map
        @rtype: MindMap
        '''
        rootClone = self.__root.cloneTree()
        if self.__attributeRegistry:
            attributeRegistry = AttributeRegistry(**self.__attributeRegistry.attributes)
        else:
            attributeRegistry = None
        return MindMap(rootClone, attributeRegistry)


class Node(Element, Observable):
    '''
    Mind map node

    @authors: Ilya Paramonov, Andrew Golovchenko, Andrew Vasilev
    '''
    #pylint: disable=W0212

    _ATTRIBUTES = ('created', 'id', 'modified', 'style', 'left', 'folded', 'link', 'hgap',
                'vgap', 'vshift', 'encrypted_content', 'backgroundColor', 'color',
                'propagateNodeFormatting')
    '''List of node attributes'''
    readable('created', 'id', 'modified', 'encrypted_content', 'nodeAttribute', 'note',
             'parent', 'icons', 'children', 'font', 'color', 'edgeColor',
             'edgeWidth', 'edgeStyle', 'hgap', 'vgap', 'vshift', 'cloud', 'folded',
             'backgroundColor', 'link', 'style', 'left',
             'labelText', 'labelFont', 'labelColor', 'propagateNodeFormatting',
             'propagateLabelFormatting')

    def _setStyle(self, style):
        '''Set style value of Node'''
        self.__style = style
        self._propagateParentStyle()

    def _propagateParentStyle(self):
        '''Store and send to children current parent style attribute'''
        if self.__style:
            newParentStyle = self.__style
        else:
            newParentStyle = self.__parentStyle
        for child in self.__children:
            if child.__parentStyle == newParentStyle: continue
            child.__parentStyle = newParentStyle
            child._propagateParentStyle()

    @property
    def effectiveStyle(self):
        '''Style of the node witch should be used to draw the node'''
        if self.__style:
            return self.__style
        elif self.__parentStyle:
            return self.__parentStyle
        else:
            return settings.get('defaultStyle')

    def _setLeft(self, left):
        '''Set left position flag of Node according inheritance policy'''
        if self.__left == left: return
        self.__left = left
        for child in self.__children:
            child._setLeft(left)

    def _setLabelColor(self, color):
        '''Set label color attribute to Node'''
        self.__labelColor = color
        self._propagateParentLabelColor()

    def _propagateParentLabelColor(self):
        '''Store and send to children parent label color attribute'''
        if self.__labelColor and self.__propagateLabelFormatting:
            newParentLabelColor = self.__labelColor
        else:
            newParentLabelColor = self.__parentLabelColor
        for child in self.__children:
            if equal(child.__parentLabelColor, newParentLabelColor): continue
            child.__parentLabelColor = newParentLabelColor
            child._propagateParentLabelColor()

    @property
    def effectiveLabelColor(self):
        '''Color of the label that should be used to draw the node'''
        if self.__labelColor:
            return self.__labelColor
        elif self.__parentLabelColor:
            return self.__parentLabelColor
        else:
            return settings.get('defaultLabelColor')

    def _setLabelFont(self, font):
        '''Set label font attribute to Node'''
        self.__labelFont = font
        self._propagateParentLabelFont()

    def _propagateParentLabelFont(self):
        '''Set label font attribute to Node according inheritance policy'''
        self.__labelFontMetrics = QFontMetricsF(self.effectiveLabelFont)
        if self.__labelFont and self.__propagateLabelFormatting:
            newParentLabelFont = self.__labelFont
        else:
            newParentLabelFont = self.__parentLabelFont
        for child in self.__children:
            if equal(child.__parentLabelFont, newParentLabelFont): continue
            child.__parentLabelFont = newParentLabelFont
            child._propagateParentLabelFont()

    @property
    def effectiveLabelFont(self):
        '''Font of the label text that should be used to draw the node'''
        if self.__labelFont:
            return self.__labelFont
        elif self.__parentLabelFont:
            return self.__parentLabelFont
        else:
            return settings.get('defaultLabelFont')

    def _setEdgeColor(self, color):
        '''Set edge color attribute to Node'''
        self.__edgeColor = color
        self._propagateParentEdgeColor()

    def _propagateParentEdgeColor(self):
        '''Set edge color attribute to Node according inheritance policy'''
        if self.__edgeColor:
            newParentEdgeColor = self.__edgeColor
        else:
            newParentEdgeColor = self.__parentEdgeColor
        for child in self.__children:
            if equal(child.__parentEdgeColor, newParentEdgeColor): continue
            child.__parentEdgeColor = newParentEdgeColor
            child._propagateParentEdgeColor()

    @property
    def effectiveEdgeColor(self):
        '''Edge color that should be used to draw the node'''
        if self.__edgeColor:
            return self.__edgeColor
        elif self.__parentEdgeColor:
            return self.__parentEdgeColor
        else:
            return settings.get('defaultEdgeColor')

    def _setEdgeWidth(self, width):
        '''Set edge width attribute to Node'''
        self.__edgeWidth = width
        self._propagateParentEdgeWidth()

    def _propagateParentEdgeWidth(self):
        '''Set edge width attribute to Node according inheritance policy'''
        if self.__edgeWidth:
            newParentEdgeWidth = self.__edgeWidth
        else:
            newParentEdgeWidth = self.__parentEdgeWidth
        for child in self.__children:
            if child.__parentEdgeWidth == newParentEdgeWidth: continue
            child.__parentEdgeWidth = newParentEdgeWidth
            child._propagateParentEdgeWidth()

    @property
    def effectiveEdgeWidth(self):
        '''Width of the node edge that should be used to draw the node'''
        if self.__edgeWidth:
            edgeWidth = self.__edgeWidth
        elif self.__parentEdgeWidth:
            edgeWidth = self.__parentEdgeWidth
        else:
            edgeWidth = settings.get('defaultEdgeWidth')
        return float(edgeWidth % 10)

    def _setEdgeStyle(self, style):
        '''Set edge style attribute to Node'''
        self.__edgeStyle = style
        self._propagateParentEdgeStyle()

    def _propagateParentEdgeStyle(self):
        '''Set edge style attribute to Node according inheritance policy'''
        if self.__edgeStyle:
            newParentEdgeStyle = self.__edgeStyle
        else:
            newParentEdgeStyle = self.__parentEdgeStyle
        for child in self.__children:
            if child.__parentEdgeStyle == newParentEdgeStyle: continue
            child.__parentEdgeStyle = newParentEdgeStyle
            child._propagateParentEdgeStyle()

    @property
    def effectiveEdgeStyle(self):
        '''Edge style that should be used to draw the node'''
        if self.__edgeStyle:
            return self.__edgeStyle
        elif self.__parentEdgeStyle:
            return self.__parentEdgeStyle
        else:
            return settings.get('defaultEdgeStyle')

    def _setFont(self, font):
        '''Set font value of Node text'''
        self.__font = font
        self._propagateParentFont()

    def _determineFontToBePropagated(self):
        '''Determine font that should be propagated to the child nodes'''
        if self.__font and self.__propagateNodeFormatting:
            return self.__font
        return self.__parentFont

    def _propagateParentFont(self):
        '''Set font value of Node text according inheritance policy'''
        self.__richcontent.setDefaultFont(self.effectiveFont)
        self.__textFontMetrics = QFontMetricsF(self.effectiveFont)
        self._updateTextWidth()
        newParentFont = self._determineFontToBePropagated()
        for child in self.__children:
            if equal(child.__parentFont, newParentFont): continue
            child.__parentFont = newParentFont
            child._propagateParentFont()

    @property
    def effectiveFont(self):
        '''The font of the node text that should be used to draw the node'''
        if self.__font:
            return self.__font
        elif self.__parentFont:
            return self.__parentFont
        else:
            return settings.get('defaultFont')

    def _setColor(self, color):
        '''Set color value of Node text content'''
        self.__color = color
        self._propagateParentColor()

    def _determineColorToBePropagated(self):
        '''Determine the color that should be propagated to the child nodes'''
        if self.__color and self.__propagateNodeFormatting:
            return self.__color
        return self.__parentColor

    def _propagateParentColor(self):
        '''Set color value of Node text content according inheritance policy'''
        self.__colorPalette.setColor(QPalette.Text, self.effectiveColor)
        newParentColor = self._determineColorToBePropagated()
        for child in self.__children:
            if equal(child.__parentColor, newParentColor): continue
            child.__parentColor = newParentColor
            child._propagateParentColor()

    @property
    def effectiveColor(self):
        '''The color of the node text that should be used to draw the node'''
        if self.__color:
            return self.__color
        elif self.__parentColor:
            return self.__parentColor
        else:
            return settings.get('defaultNodeTextColor')

    def _setBackgroundColor(self, color):
        '''Set background color value of Node'''
        self.__backgroundColor = color
        self._propagateParentBackgroundColor()

    def _determineBackgroundColorToBePropagated(self):
        '''Determine the background color that should be propagated to the child nodes'''
        if self.__backgroundColor and self.__propagateNodeFormatting:
            return self.__backgroundColor
        return self.__parentBackgroundColor

    def _propagateParentBackgroundColor(self):
        '''Set background color value of Node according inheritance policy'''
        newParentBackgroundColor = self._determineBackgroundColorToBePropagated()
        for child in self.__children:
            if equal(child.__parentBackgroundColor, newParentBackgroundColor): continue
            child.__parentBackgroundColor = newParentBackgroundColor
            child._propagateParentBackgroundColor()

    @property
    def effectiveBackgroundColor(self):
        '''The background color of the node text that should be used to draw the node'''
        if self.__backgroundColor:
            return self.__backgroundColor
        elif self.__parentBackgroundColor:
            return self.__parentBackgroundColor
        else:
            return settings.get('defaultNodeBackgroundColor')

    def _updateTextWidth(self):
        '''Update text width'''
        if self.__textFontMetrics.width(self.__richcontent.toPlainText()) < \
                self.__maxWidth:
            self.__richcontent.setTextWidth(-1)
        else:
            self.__richcontent.setTextWidth(self.__maxWidth)

    def updateProperties(self):
        '''Update cached properties of the node'''
        self.__richcontent.setDefaultFont(self.effectiveFont)
        self.__colorPalette.setColor(QPalette.Text, self.effectiveColor)
        self.__labelFontMetrics = QFontMetricsF(self.effectiveLabelFont)
        self.__textFontMetrics = QFontMetricsF(self.effectiveFont)
        self.__maxWidth = settings.get('maxNodeTextWidth')
        self._updateTextWidth()
        for child in self.__children:
            child.updateProperties()

    def __init__(self, **attributes):
        '''
        Initialize node attributes

        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        Element.__init__(self, **attributes)
        Observable.__init__(self)
        # has similar name properties
        self.__parent = None
        self.__labelText = None
        self.__labelFont = self.__parentLabelFont = None
        self.__labelFontMetrics = QFontMetricsF(settings.get('defaultLabelFont'))
        self.__labelColor = self.__parentLabelColor = None
        self.__propagateLabelFormatting = False
        self.__note = None
        self.__left = None
        self.__edgeStyle = self.__parentEdgeStyle = None
        self.__style = self.__parentStyle = None
        self.__edgeColor = self.__parentEdgeColor = None
        self.__edgeWidth = self.__parentEdgeWidth = None
        self.__cloud = None
        self.__font = self.__parentFont = None
        self.__color = self.__parentColor = None
        self.__backgroundColor = self.__parentBackgroundColor = None
        self.__propagateNodeFormatting = False
        self.__icons = []
        self.__nodeAttribute = None
        self.__children = []
        if self.__id is None: #pylint: disable=E0203
            self.__modified = self.__created = int(time.time() * 1000.0)
            self.__id = 'ID_%d' % random.getrandbits(31)
        # local fields
        self.__isHtmlContent = False
        self.__colorPalette = QPalette()
        self.__colorPalette.setColor(QPalette.Text, settings.get('defaultNodeTextColor'))
        self.__richcontent = QTextDocument()
        self.__richcontent.setDefaultFont(settings.get('defaultFont'))
        self.__textFontMetrics = QFontMetricsF(settings.get('defaultFont'))
        self.__maxWidth = settings.get('maxNodeTextWidth')

    def _setField(self, field, val):
        '''Create/set private field value'''
        self.__dict__['_Node__' + field] = val

    def _getField(self, fieldName):
        '''Return private field value'''
        return self.__dict__['_Node__' + fieldName]

    __LOCAL_ATTRIBUTES = set(('cloud', 'folded', 'icons', 'labelText', 'link'))
    '''Properties which emit "localAttributeChanged" notification and not has a setter'''

    __INHERITED_ATTRIBUTES = set(('font', 'color', 'backgroundColor', 'edgeWidth',
            'edgeStyle', 'edgeColor', 'labelFont', 'labelColor', 'left', 'style',
            'propagateNodeFormatting', 'propagateLabelFormatting'))
    '''Node properties which emit "inheritedAttributeChanged" notification'''

    __NONEVENT_ATTRIBUTES = set(('created', 'id', 'modified', 'encrypted_content',
            'nodeAttribute', 'note', 'hgap', 'vgap', 'vshift'))
    '''Properties which not emit any notifications and not has a setter'''

    __TEXT_ATTRIBUTES = set(('text', 'html'))
    '''
    Node text properties, they are part of the local attributes that need special treatment.
    '''

    __LOCAL_CHANGE_LEVEL, __INHERITED_CHANGE_LEVEL = range(2)
    '''Numerical constants of signal types'''

    __SIGNALS = ('localAttributeChanged', 'inheritedAttributeChanged')
    '''Main signals to notify about mind map changes'''

    def setAttributes(self, **attributes):
        '''
        Set one or many node attributes by name

        @param attributes: name = value pairs of attributes
        @type attributes: dict
        '''
        maxNotifyLevel = -1 # -1 <=> no notification will be send
        for attrName, attrValue in attributes.items():
            if attrName in Node.__INHERITED_ATTRIBUTES:
                self._setAttribute(attrName, attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName in Node.__TEXT_ATTRIBUTES:
                self._setAttribute(attrName, attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__LOCAL_CHANGE_LEVEL)
            elif attrName in Node.__LOCAL_ATTRIBUTES:
                self._setField(attrName, attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__LOCAL_CHANGE_LEVEL)
            elif attrName in Node.__NONEVENT_ATTRIBUTES:
                self._setField(attrName, attrValue)
            else: raise Exception('Unknown attribute name: ' + attrName)
        # emit only one signal with max level of change
        if maxNotifyLevel != -1:
            self.notifyObservers(self, type = Node.__SIGNALS[maxNotifyLevel])

    def _setAttribute(self, attributeName, attributeValue):
        '''
        Use setter to set new value for the attribute identified by attributeName
        @type attributeName: str
        @param attributeValue: value for the attribute
        '''
        setter = getattr(self, '_set%s%s' % (attributeName[0].upper(), attributeName[1:]))
        setter(attributeValue)

    WRITABLE_PROPERTIES = __LOCAL_ATTRIBUTES.union(__INHERITED_ATTRIBUTES).union(
            __NONEVENT_ATTRIBUTES).union(__TEXT_ATTRIBUTES)
    '''Node writable properties that can be set with setAttributes() method'''

    def getAttributes(self, *propNames, **options):
        '''
        Return dict of specified property values. If used method call with
        argument 'getAll = True' returns overall writable values

        @param propNames: list of attributes to be returned
        @type propNames: tuple

        @param options: named control options (one yet): getAll = True
        @type options: dict
        '''
        resultAttributes = {}
        if options.get('getAll'):
            propNames = Node.WRITABLE_PROPERTIES
        propNames = list(propNames)
        # automatically choose content type to be returned)
        found = False
        for textAttribute in Node.__TEXT_ATTRIBUTES:
            if textAttribute in propNames:
                propNames.remove(textAttribute)
                found = True
        if found:
            if self.__isHtmlContent:
                resultAttributes['html'] = self.text
            else:
                resultAttributes['text'] = self.text
        for attrName in propNames:
            resultAttributes[attrName] = self._getField(attrName)
        return resultAttributes

    @property
    def text(self):
        '''
        Return text of the node in either plain text or HTML
        @rtype: unicode
        '''
        # unicode type casting must be used for QString
        if self.__isHtmlContent:
            return unicode(self.__richcontent.toHtml())
        return unicode(self.__richcontent.toPlainText())

    def isHtml(self):
        '''
        Returns True if node has rich content; otherwise return False
        '''
        return self.__isHtmlContent

    def _setText(self, text):
        '''
        Set node content as simple text
        '''
        self.__isHtmlContent = False
        self.__richcontent.setPlainText(text if text else '')
        self._updateTextWidth()

    def _setHtml(self, html):
        '''
        Set node content as formated html
        '''
        self.__isHtmlContent = True
        self.__richcontent.setHtml(html if html else '')
        self._updateTextWidth()

    def _setPropagateNodeFormatting(self, propagateFormatting):
        '''
        Set propagate formatting switch of the node text
        @type propagateFormatting: bool
        '''
        self.__propagateNodeFormatting = propagateFormatting
        self._propagateParentFont()
        self._propagateParentColor()
        self._propagateParentBackgroundColor()

    def _setPropagateLabelFormatting(self, propagateFormatting):
        '''
        Set propagate formatting switch of the label text
        @type propagateFormatting: bool
        '''
        self.__propagateLabelFormatting = propagateFormatting
        self._propagateParentLabelColor()
        self._propagateParentLabelFont()

    def addChild(self, child, position = None):
        '''
        Add a child node to the current node

        @type child: Node

        @param position: position in children list
        @type position: int
        '''
        # prevent duplicates
        if child in self.__children: return
        child.__parent = self
        if position is not None:
            self.__children.insert(position, child)
        else:
            self.__children.append(child)
        self._propagateParentStyle()
        self._propagateParentLabelColor()
        self._propagateParentLabelFont()
        self._propagateParentEdgeColor()
        self._propagateParentEdgeWidth()
        self._propagateParentEdgeStyle()
        self._propagateParentFont()
        self._propagateParentColor()
        self._propagateParentBackgroundColor()
        self.updateProperties()
        self.notifyObservers(child, type = 'childNodeAdded')

    def removeChild(self, child):
        '''
        Remove a child node from the current node

        @type child: Node
        '''
        self.notifyObservers(child, type = 'childNodeRemoved')
        self.__children.remove(child)
        child.__parent = None

    def getChildPosition(self, child):
        '''
        Return position of a child in the current node

        @type child: Node
        '''
        if not self.isAncestorOf(child):
            return None
        return self.__children.index(child)

    def moveNode(self, dest, position = None):
        '''
        Relocate current node to a destination node

        @type dest: Node

        @param position: position in children list
        @type position: int
        '''
        if self.__parent is None or self.isAncestorOf(dest):
            return
        if self.__parent == dest:
            if position - dest.children.index(self) > 1:
                position -= 1
        self.__parent.__children.remove(self)
        self.__parent = dest
        if position is None:
            dest.__children.append(self)
        else:
            dest.__children.insert(position, self)
        self._setLeft(self.left if dest.parent is None else dest.left)
        dest._propagateParentStyle()
        dest._propagateParentLabelColor()
        dest._propagateParentLabelFont()
        dest._propagateParentEdgeColor()
        dest._propagateParentEdgeWidth()
        dest._propagateParentEdgeStyle()
        dest._propagateParentFont()
        dest._propagateParentColor()
        dest._propagateParentBackgroundColor()
        self.notifyObservers(self, dest, type = 'nodeMoved')

    def isAncestorOf(self, child):
        '''
        Check if the current node is an ancestor of the child

        @type child: Node
        @rtype: bool
        '''
        node = child
        while node is not None:
            if self == node:
                return True
            node = node.parent
        return False

    def depth(self):
        '''
        Return the depth of the node
        @rtype: int
        '''
        node = self
        nodeDepth = 0
        while node.parent is not None:
            nodeDepth += 1
            node = node.parent
        return nodeDepth * (-1 if self.left else 1)

    def createChildPrototype(self, **attributes):
        '''
        Create a new node without adding it into the children list.
        Attributes of the created node are set according to the inheritance policy.
        Prototype is not a child of this node yet. You must add it afterwards with the
        addChilde method

        @param attributes: initialization attributes
        @type attributes: dict

        @return: created child node
        @rtype: Node
        '''
        child = Node(**attributes)
        child.__parent = self
        # set inheritable attributes
        child.__left = self.__left
        child.__parentFont = self._determineFontToBePropagated()
        child.__parentColor = self._determineColorToBePropagated()
        child.__parentBackgroundColor = self._determineBackgroundColorToBePropagated()
        return child

    @property
    def allNodeAttributes(self):
        '''
        Return list of pairs (attributeName, attributeValue)
        @rtype: list
        '''
        if self.__nodeAttribute:
            # Attribute.allAttributes always has not None value
            return self.__nodeAttribute.allAttributes.items()
        return []

    def addNodeAttribute(self, attrName, attrValue):
        '''
        Add attribute to the current node

        @type attrName: str
        @type attrValue: str
        '''
        if self.__nodeAttribute is None:
            self.__nodeAttribute = Attribute()
        self.__nodeAttribute.addAttribute(attrName, attrValue)

    def paint(self, painter):
        '''
        Paint node text content

        @type painter: QPainter
        '''
        painter.save()
        paintContext = QAbstractTextDocumentLayout.PaintContext()
        paintContext.palette = self.__colorPalette
        self.__richcontent.documentLayout().draw(painter, paintContext)
        painter.restore()

    def textSize(self):
        '''
        Return area size of node text content

        @rtype: QSizeF
        '''
        size = self.__richcontent.size()
        return QSizeF(self.__richcontent.idealWidth(), size.height())

    @property
    def labelSize(self):
        '''
        Return minimal area size of label text

        @rtype: QSizeF
        '''
        if self.__labelText is None: return QSizeF(0.0, 0.0)
        return QSizeF(self.__labelFontMetrics.width(self.__labelText),
                      self.__labelFontMetrics.overlinePos() -
                      self.__labelFontMetrics.underlinePos())

    __PARENT_ATTRIBUTES = set(('parentFont', 'parentColor', 'parentBackgroundColor',
            'parentStyle', 'parentLabelColor', 'parentLabelFont', 'parentEdgeWidth',
            'parentEdgeStyle', 'parentEdgeColor'))
    def clone(self):
        '''
        Clone current node and return it's cloned copy
        @rtype: Node
        '''
        clone = Node()
        clone.setAttributes(**self.getAttributes(getAll = True))
        for attribute in Node.__PARENT_ATTRIBUTES:
            attrName = '_Node__%s' % attribute
            clone.__dict__[attrName] = self.__dict__[attrName]
        return clone

    def cloneTree(self):
        '''
        Create a clone of the current node and it's subtree
        @rtype: Node
        '''
        clone = self.clone()
        for child in self.__children:
            clone.addChild(child.cloneTree())
        return clone


class Cloud(Element):
    '''
    Cloud inner node

    @authors: Andrew Golovchenko
    '''
    _ATTRIBUTES = ('style', 'color', 'width')
    '''List of cloud attributes'''
    writable(*_ATTRIBUTES)

    def __init__(self, **attributes):
        '''
        Initialize cloud attributes

        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        Element.__init__(self, **attributes)


class AttributeRegistry(Element):
    '''
    Mind map attribute registry

    @authors: Andrew Golovchenko
    '''
    _ATTRIBUTES = ('restricted', 'font_size', 'show_attributes')
    '''List of "attribute registry" element attributes'''
    writable('viewOptions', 'attrValues')
    writable(*_ATTRIBUTES)

    def __init__(self, **attributes):
        '''
        Initialize empty attribute registry

        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        Element.__init__(self, **attributes)
        # dict of attributes options in following format:
        # {'name_A' : {'option_x' : 'value_x', 'option_y' : 'value_y'}
        #  'name_B' : {'option_z' : 'value_z'} ...}
        self.__viewOptions = {}
        # dict of attributes values in following format:
        # {'name_A' : ['value_1', 'value_2'], 'name_B' : ['value_1'] ...}
        self.__attrValues = {}


class Attribute(Element):
    '''
    Composite container class for attribute and attribute_layout inner nodes

    @authors: Andrew Golovchenko
    '''
    _ATTRIBUTES = ('name_width', 'value_width')
    '''List of "attribute" element attributes'''
    readable('allAttributes')
    writable(*_ATTRIBUTES)

    def __init__(self, **attributes):
        '''
        Initialize class attributes

        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        Element.__init__(self, **attributes)
        # dict of attributes in following format:
        # {'name_A' : 'value_A', 'name_B' : 'value_B' ...}
        self.__allAttributes = {}

    def addAttribute(self, attrName, attrValue):
        '''
        Add attribute to the container

        @type attrName: str
        @type attrValue: str
        '''
        self.__allAttributes[attrName] = attrValue

def equal(objectOne, objectTwo):
    '''
    Compare two objects for equality. Both objects can be None.
    @return: True if objects are equal, False otherwise
    @rtype: bool
    '''
    if objectOne is None and objectTwo is None: return True
    if objectOne is None or objectTwo is None: return False
    return objectOne == objectTwo
