# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Transport layer classes
'''

from twisted import application
from twisted.internet.error import ConnectionDone
from twisted.names.srvconnect import SRVConnector
from twisted.python import failure
from twisted.words.protocols import jabber
from twisted.words.xish.xmlstream import STREAM_ERROR_EVENT
from twisted.python import log
from twisted.internet import reactor
from wokkel.subprotocols import StreamManager, XMPPHandlerCollection
from hivemind.attribute import Observable
import hivemind.network.core

#pylint: disable=C0103

class HivemindXmlStreamFactory(jabber.xmlstream.XmlStreamFactory, Observable):
    '''
    Factory for Jabber XmlStream objects as a reconnecting client.
    @author: Oleg Kandaurov
    '''

    def __init__(self, authenticator):
        jabber.xmlstream.XmlStreamFactory.__init__(self, authenticator)
        Observable.__init__(self)
        self.initialDelay = 1.0
        self.factor = 1.4
        self.maxDelay = 20

    def startedConnecting(self, connector):
        '''
        Called when a connection has been started.
        @param connector: a Connector object.
        '''
        log.msg('Connecting started at %s' % self.__class__.__name__)
        self.notifyObservers(connection = hivemind.network.core.connectionState.connecting)
        jabber.xmlstream.XmlStreamFactory.startedConnecting(self, connector)


class HivemindStreamManager(StreamManager):
    '''
    Business logic representing a managed XMPP connection.

    This fixes two related issues with adding a new subprotocol handler
    (Changeset 99:2c8dc93fbef4 03/07/2011)
    1) Adding a handler when the stream is not yet initialized (authenticated)
    does not cause connectionMade to be called.
    2) Adding a handler in connectionMade, connectionInitialized, or connectionLost
    modifies the lists of handlers iterated over, causing some methods being called
    too often.

    @author: Oleg Kandaurov
    '''

    def addHandler(self, handler):
        '''Add protocol handler.'''
        XMPPHandlerCollection.addHandler(self, handler)
        if self.xmlstream:
            handler.makeConnection(self.xmlstream)
        if self._initialized:
            handler.connectionInitialized()

    def _connected(self, xs):
        '''Called when the transport connection has been established.'''
        def logDataIn(buf):
            '''Log incoming data.'''
            log.msg("RECV: %r" % buf)

        def logDataOut(buf):
            '''Log outgoing data.'''
            log.msg("SEND: %r" % buf)

        if self.logTraffic:
            xs.rawDataInFn = logDataIn
            xs.rawDataOutFn = logDataOut
        self.xmlstream = xs
        for e in list(self):
            e.makeConnection(xs)

    def _authd(self, xs):
        '''Called when the stream has been initialized.'''
        for p in self._packetQueue:
            xs.send(p)
        self._packetQueue = []
        self._initialized = True
        for e in list(self):
            e.connectionInitialized()

    def _disconnected(self, reason):
        '''Called when the stream has been closed.'''
        self.xmlstream = None
        self._initialized = False
        if not hasattr(reason, 'trap'):
            reason = failure.Failure(ConnectionDone())
        for e in list(self):
            e.connectionLost(reason)


class HivemindXMPPClient(HivemindStreamManager, application.service.Service, Observable):
    '''
    Service that initiates an XMPP client connection
    @author: Oleg Kandaurov
    '''

    #pylint: disable=W0212

    def __init__(self, authenticator, factory, host, port):
        '''
        @type authenticator: HybridAuthenticator
        @type factory: XmlStreamFactory
        @type host: str
        @type port: int
        '''
        self.jid = authenticator.jid
        self.domain = authenticator.jid.host
        self.server = jabber.jid.internJID(self.domain)
        self.host = host
        self.port = port
        self.logTraffic = True
        self._connection = None
        HivemindStreamManager.__init__(self, factory)
        Observable.__init__(self)

    def startService(self):
        '''Start service'''
        application.service.Service.startService(self)
        self._connection = self._createConnector()

    def stopService(self):
        '''Stop service'''
        application.service.Service.stopService(self)
        self.factory.stopTrying()
        self._connection.disconnect()

    def initializationFailed(self, reason):
        '''
        Called when stream initialization has failed
        @param reason: A failure instance indicating why stream initialization failed
        @type reason: Failure
        '''
        log.msg('Initialization failed at %s' % (self.__class__.__name__))
        self.notifyObservers(error = reason)

    def _authd(self, xs):
        '''
        Called when the stream has been initialized.
        Save the JID that we were assigned by the server, as the resource might
        differ from the JID we asked for. This is stored on the authenticator
        by its constituent initializers.
        '''
        self.jid = self.factory.authenticator.jid
        HivemindStreamManager._authd(self, xs)
        log.msg('Connection initialized at %s' % (self.__class__.__name__))
        self.notifyObservers(connection = hivemind.network.core.connectionState.initialized)

    def _connected(self, xs):
        '''Called when the transport connection has been established'''
        def _onError(error):
            '''Notify about error'''
            self.notifyObservers(error = error)
        HivemindStreamManager._connected(self, xs)
        log.msg('Transport connected at %s' % (self.__class__.__name__))
        self.notifyObservers(connection = hivemind.network.core.connectionState.connected)
        xs.addObserver(STREAM_ERROR_EVENT, _onError)

    def _disconnected(self, reason):
        '''Called when the stream has been closed'''
        HivemindStreamManager._disconnected(self, reason)
        log.msg('Stream closed at %s' % (self.__class__.__name__))
        self.notifyObservers(connection = hivemind.network.core.connectionState.disconnected)

    def _createConnector(self):
        '''Create connector instance'''
        connector = HivemindXMPPClientConnector(reactor, self.domain, self.port, self.factory)
        connector.connect()
        return connector

    connection = property(lambda self: self._connection, None)


class HivemindXMPPClientConnector(SRVConnector):
    '''
    Hivemind implementation of XMPP connector
    @author: Oleg Kandaurov
    '''
    def __init__(self, reactorInstance, domain, port, factory):
        '''Constructor'''
        SRVConnector.__init__(self, reactorInstance, 'xmpp-client', domain, factory)
        self.__port = port

    def pickServer(self):
        host, port = SRVConnector.pickServer(self)
        if not self.servers and not self.orderedServers:
            port = 5222
        if self.__port is not None:
            port = self.__port
        return host, port
