# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Action bag class for actions storing

@author: Alexander Kulikov
'''

from hivemind.attribute import readable
import hivemind.resource as resource
from PyQt4.QtCore import QObject, Qt
from PyQt4.QtGui import QKeySequence, QAction, QIcon
from hivemind.gui_widgets import StatusAction
import hivemind.sysconfig as sysconfig
import hivemind.settings as settings


def _setShortcut(action, shortcut):
    '''
    Set shortcut to action
    @type action: QAction
    @type shortcut: string or int
    '''
    #pylint: disable = E1103
    shortcut = unicode(QKeySequence(shortcut).toString()).lower()
    if 'return' in shortcut:
        shortcut = [shortcut, shortcut.replace('return', 'enter')]
    if 'enter' in shortcut:
        shortcut = [shortcut, shortcut.replace('enter', 'return')]
    if isinstance(shortcut, list):
        action.setShortcuts(shortcut)
    else:
        action.setShortcut(shortcut)


class ActionBag(QObject):
    '''
    Action bag for actions storing

    @author: Alexander Kulikov
    '''

    readable('fileNewAction', 'fileOpenAction', 'fileSaveAction',
             'fileSaveAsAction', 'editEdgeAction', 'editLabelAction', 'editNodeAction',
             'editNodeIconsAction', 'foldNodeAction', 'addNodeAction',
             'addSiblingNodeBelowAction', 'removeNodeAction', 'enterTransferModeAction',
             'undoAction', 'redoAction', 'zoomInAction', 'recentFileActions',
             'zoomOutAction', 'exitAction', 'aboutAction', 'settingsAction',
             'parentWidthAction', 'thinWidthAction', 'oneWidthAction', 'twoWidthAction',
             'fourWidthAction', 'eightWidthAction', 'parentStyleAction',
             'linearStyleAction', 'bezierStyleAction', 'sharpLinearStyleAction',
             'sharpBezierStyleAction', 'connectionStatusAction', 'sidebarAction',
             'addSiblingNodeAboveAction', 'editAffiliationAction', 'findAction',
             'cancelTransferModeAction', 'putNodeAction', 'putNodeBelowAction',
             'putNodeAboveAction', 'connectXMPPAction', 'disconnectXMPPAction',
             'createServiceAction', 'createClientAction', 'stopProtocolAction',
             'startServiceAction', 'startClientAction', 'showContextMenuAction',
             'exportToImageAction', 'editHyperlinkAction', 'openHyperlinkAction',
             'printAction', 'networkSettingsAction', 'networkHelpAction',
             'selectLeftNodeAction', 'selectRightNodeAction', 'selectUpNodeAction',
             'selectDownNodeAction', 'selectRootNodeAction', 'startSmartConferenceRobot',
             'stopSmartConferenceRobot', 'filterAction')

    def __init__(self, parent = None):
        QObject.__init__(self, parent)
        self._createActions()
        self._initializeActions()
        self._updateShortcuts()
        if sysconfig.PLATFORM == 'maemo':
            self._initializeMaemoActions()
        else:
            self._addEllipsesToActionText()
        settings.connect(self._updateRecentFileActions)
        settings.connect(self._updateShortcuts)

    def _createActions(self):
        '''Create actions'''
        #pylint: disable=W0201,R0915
        self.__fileNewAction = self.createAction(self.tr('&New'), QKeySequence.New,
                'document-new')
        self.__fileOpenAction = self.createAction(self.tr('&Open'), QKeySequence.Open,
                'document-open')
        self.__fileSaveAction = self.createAction(self.tr('&Save'), QKeySequence.Save,
                'document-save')
        self.__fileSaveAsAction = self.createAction(self.tr('Save as'),
                QKeySequence.SaveAs, 'document-save-as')
        self.__recentFileActions = []
        for _ in range(0, settings.get('maxRecentFiles')):
            self.__recentFileActions.append(QAction(self))
        self.__editEdgeAction = self.createAction(self.tr('Edit edge'),
                settings.get('editEdgeShortcut'))
        self.__editLabelAction = self.createAction(self.tr('Edit label'),
                settings.get('editLabelShortcut'))
        self.__editNodeAction = self.createAction(self.tr('Edit node'),
                settings.get('editNodeShortcut'), 'document-edit')
        self.__editNodeIconsAction = self.createAction(self.tr('Select icons'),
                settings.get('editNodeIconsShortcut'))
        self.__foldNodeAction = self.createAction(self.tr('Fold node'),
                settings.get('foldNodeShortcut'))
        self.__addNodeAction = self.createAction(self.tr('New child node'),
                settings.get('addNodeShortcut'), 'list-add')
        self.__addSiblingNodeBelowAction = self.createAction(self.tr('New sibling node' +
                ' below'), settings.get('addSiblingNodeBelowShortcut'), 'list-add')
        self.__addSiblingNodeAboveAction = self.createAction(self.tr('New sibling node' +
                ' above'), settings.get('addSiblingNodeAboveShortcut'), 'list-add')
        self.__removeNodeAction = self.createAction(self.tr('Remove node'),
                settings.get('removeNodeShortcut'), 'list-remove')
        self.__enterTransferModeAction = self.createAction(self.tr('Move node'),
                settings.get('enterTransferModeShortcut'))
        self.__cancelTransferModeAction = self.createAction(
                self.tr('Cancel transfer mode'),
                settings.get('cancelTransferModeShortcut'))
        self.__putNodeAction = self.createAction(self.tr('Paste moved node'),
                settings.get('putNodeShortcut'))
        self.__putNodeBelowAction = self.createAction(self.tr('Put node below'),
                settings.get('putNodeBelowShortcut'))
        self.__putNodeAboveAction = self.createAction(self.tr('Put node above'),
                settings.get('putNodeAboveShortcut'))
        self.__undoAction = self.createAction(self.tr('&Undo'), QKeySequence.Undo,
                'edit-undo')
        self.__redoAction = self.createAction(self.tr('&Redo'), QKeySequence.Redo,
                'edit-redo')
        self.__zoomInAction = self.createAction(self.tr('Zoom in'), QKeySequence.ZoomIn,
                'zoom-in')
        self.__zoomOutAction = self.createAction(self.tr('Zoom out'),
                QKeySequence.ZoomOut, 'zoom-out')
        self.__aboutAction = self.createAction(self.tr('&About'), None, 'help-about')
        self.__exitAction = self.createAction(self.tr('E&xit'),
                QKeySequence.Quit, 'application-exit')
        self.__findAction = self.createAction(self.tr('&Find'), QKeySequence.Find,
                'edit-find')
        self.__filterAction = self.createAction(self.tr('F&ilter'), 'Ctrl+J')
        self.__settingsAction = self.createAction(self.tr('Settings'),
                QKeySequence.Preferences, 'preferences-other')
        self.__parentWidthAction = self.createAction(self.tr('Parent'))
        self.__thinWidthAction = self.createAction(self.tr('Thin'))
        self.__oneWidthAction = self.createAction('1')
        self.__twoWidthAction = self.createAction('2')
        self.__fourWidthAction = self.createAction('4')
        self.__eightWidthAction = self.createAction('8')
        self.__parentStyleAction = self.createAction(self.tr('Parent'))
        self.__linearStyleAction = self.createAction(self.tr('Linear'))
        self.__bezierStyleAction = self.createAction(self.tr('Bezier'))
        self.__sharpLinearStyleAction = self.createAction(self.tr('Sharp linear'))
        self.__sharpBezierStyleAction = self.createAction(self.tr('Sharp bezier'))
        self.__connectionStatusAction = StatusAction(self.tr('Connection status'),
                'connection', 'disconnected')
        self.__editAffiliationAction = self.createAction(self.tr('&Edit permissions'))
        self.__disconnectXMPPAction = self.createAction(self.tr('&Go offline'))
        self.__stopProtocolAction = self.createAction(self.tr('&Finish teamwork'))
        self.__startServiceAction = self.createAction(self.tr('&Host teamwork'))
        self.__startClientAction = self.createAction(self.tr('&Participate in teamwork'))
        self.__networkSettingsAction = self.createAction(self.tr('&Settings'))
        self.__networkHelpAction = self.createAction(self.tr('He&lp'))
        self.__showContextMenuAction = self.createAction(self.tr('Show context menu'),
                Qt.Key_Menu)
        self.__exportToImageAction = self.createAction(self.tr('Export to image'))
        self.__editHyperlinkAction = self.createAction(self.tr('Edit hyperlink'),
                settings.get('editHyperlinkShortcut'))
        self.__openHyperlinkAction = self.createAction(self.tr('Open hyperlink'),
                settings.get('openHyperlinkShortcut'))
        self.__printAction = self.createAction(self.tr('Print'), QKeySequence.Print,
                'document-print')
        self.__sidebarAction = StatusAction(self.tr('Open sidebar'), 'sidebar', 'closed')
        self.__selectLeftNodeAction = self.createAction('',
                settings.get('selectLeftNodeShortcut'))
        self.__selectRightNodeAction = self.createAction('',
                settings.get('selectRightNodeShortcut'))
        self.__selectUpNodeAction = self.createAction('',
                settings.get('selectUpNodeShortcut'))
        self.__selectDownNodeAction = self.createAction('',
                settings.get('selectDownNodeShortcut'))
        self.__selectRootNodeAction = self.createAction('',
                settings.get('selectRootNodeShortcut'))
        self.__startSmartConferenceRobot = self.createAction(self.tr('&Join SmartConference'))
        self.__stopSmartConferenceRobot = self.createAction(self.tr('&Leave SmartConference'))

    def _updateShortcuts(self):
        '''Update action shortcuts'''
        _setShortcut(self.__editEdgeAction, settings.get('editEdgeShortcut'))
        _setShortcut(self.__editLabelAction, settings.get('editLabelShortcut'))
        _setShortcut(self.__editNodeAction, settings.get('editNodeShortcut'))
        _setShortcut(self.__editNodeIconsAction, settings.get('editNodeIconsShortcut'))
        _setShortcut(self.__foldNodeAction, settings.get('foldNodeShortcut'))
        _setShortcut(self.__addNodeAction, settings.get('addNodeShortcut'))
        _setShortcut(self.__addSiblingNodeBelowAction,
                settings.get('addSiblingNodeBelowShortcut'))
        _setShortcut(self.__addSiblingNodeAboveAction,
                settings.get('addSiblingNodeAboveShortcut'))
        _setShortcut(self.__removeNodeAction, settings.get('removeNodeShortcut'))
        _setShortcut(self.__enterTransferModeAction,
                settings.get('enterTransferModeShortcut'))
        _setShortcut(self.__cancelTransferModeAction,
                settings.get('cancelTransferModeShortcut'))
        _setShortcut(self.__putNodeAction, settings.get('putNodeShortcut'))
        _setShortcut(self.__putNodeBelowAction, settings.get('putNodeBelowShortcut'))
        _setShortcut(self.__putNodeAboveAction, settings.get('putNodeAboveShortcut'))
        _setShortcut(self.__editHyperlinkAction, settings.get('editHyperlinkShortcut'))
        _setShortcut(self.__openHyperlinkAction, settings.get('openHyperlinkShortcut'))
        _setShortcut(self.__selectLeftNodeAction, settings.get('selectLeftNodeShortcut'))
        _setShortcut(self.__selectRightNodeAction,
                settings.get('selectRightNodeShortcut'))
        _setShortcut(self.__selectUpNodeAction, settings.get('selectUpNodeShortcut'))
        _setShortcut(self.__selectDownNodeAction, settings.get('selectDownNodeShortcut'))
        _setShortcut(self.__selectRootNodeAction, settings.get('selectRootNodeShortcut'))

    def _initializeMaemoActions(self):
        '''Initialize maemo actions'''
        self.__undoAction.setIcon(QIcon(resource.getIcon('undo')))
        self.__redoAction.setIcon(QIcon(resource.getIcon('redo')))
        self.__zoomInAction.setIcon(resource.getIcon('zoom_in'))
        self.__zoomInAction.setAutoRepeat(True)
        self.__zoomOutAction.setIcon(resource.getIcon('zoom_out'))
        self.__zoomOutAction.setAutoRepeat(True)

    def _initializeActions(self):
        '''Make specific action initializations'''
        self.__fileSaveAction.setEnabled(False)
        self.__networkHelpAction.setEnabled(False)
        self._updateRecentFileActions()

    def _addEllipsesToActionText(self):
        '''Add ellipses to the text of the actions displaying dialogs'''
        actionsWithEllipses = [
                self.__fileOpenAction, self.__fileSaveAsAction,
                self.__settingsAction, self.__exportToImageAction,
                self.__addNodeAction, self.__addSiblingNodeBelowAction,
                self.__addSiblingNodeAboveAction, self.__editNodeAction,
                self.__editNodeIconsAction, self.__editLabelAction,
                self.__editEdgeAction, self.__editAffiliationAction,
                self.__startClientAction, self.__networkSettingsAction,
                self.__printAction, self.__findAction, self.__filterAction,
                self.__editHyperlinkAction, self.__startServiceAction]
        for action in actionsWithEllipses:
            action.setText(action.text() + '...')

    def _updateRecentFileActions(self):
        '''
        Update recentFileList of recent files
        '''
        recentFileList = settings.get('recentFiles')
        for row, action in enumerate(self.__recentFileActions):
            if row < len(recentFileList):
                text = '&' + str(row + 1) + ' ' + recentFileList[row]
                action.setText(text)
                action.setData(recentFileList[row])
                action.setVisible(True)
            else:
                action.setVisible(False)

    def createAction(self, text, shortcut = None, icon = None):
        '''
        Create action
        @param text: descriptive text for tool buttons
        @type text: string
        @param handler: slot
        @type shortcut: string
        @rtype: QAction
        '''
        action = QAction(text, self)
        if shortcut:
            _setShortcut(action, shortcut)
        if icon:
            action.setIcon(QIcon.fromTheme(icon))
        return action
