# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Ping related classes
'''

from twisted.python import log
from twisted.internet.task import LoopingCall
from twisted.words.protocols import jabber
from wokkel.subprotocols import XMPPHandler
from wokkel.ping import NS_PING
from hivemind.attribute import Observable
from hivemind import enum

#pylint: disable=C0103


class Pinger(XMPPHandler, Observable):
    '''
    Hivemind ping client
    @author: Oleg Kandaurov
    '''

    entityState = enum.enum('alive', 'timeout')

    def __init__(self, entity, interval, maxCount, isServer = False):
        '''
        Constructor
        @param entity: Entity to be pinged
        @type entity: str
        @param interval: Interval between sending ping
        @type interval: int
        @param maxCount: if maxCount pings without response has been occured, connection is
        considered lost
        @type maxCount: int
        @param server: If True then we ping jabber server and we must treat errors
        from server as successful pongs
        @type server: bool
        '''
        Observable.__init__(self)
        XMPPHandler.__init__(self)
        self.__entity = entity
        self.__interval = interval
        self.__maxCount = maxCount
        self.__isServer = isServer
        self.__pingLoop = None
        self.__count = 0

    def ping(self, entity, sender = None):
        '''
        Send out a ping request and wait for a response
        @param entity: Entity to be pinged
        @type entity: L{jid.JID}
        @return: A deferred that fires upon receiving a response.
        @rtype: L{defer.Deferred}
        @param sender: Optional sender address.
        @type sender: L{jid.JID}
        '''
        if self.xmlstream is None: return
        request = jabber.xmlstream.IQ(self.xmlstream, 'get')
        request.addElement((NS_PING, 'ping'))
        if sender:
            request['from'] = unicode(sender)
        defer = request.send(entity.full())
        return defer

    def _suppressError(self, failure): #pylint: disable=R0201
        '''Suppress error responses from server'''
        failure.trap(jabber.error.StanzaError)
        exc = failure.value
        if exc.condition in ['service-unavailable', 'feature-not-implemented']:
            return None
        return failure

    def _onSuccess(self, result):
        '''Handler of alive event'''
        aliveEvent = {'state': self.entityState.alive, 'entity': self.__entity}
        if self.__isServer:
            self.notifyObservers(transportPing = aliveEvent)
        else:
            self.notifyObservers(protocolPing = aliveEvent)
        self.__count = 0

    def _sendPing(self):
        '''Send ping to entity and wait for response'''
        if self.__count > self.__maxCount:
            timeoutEvent = {'state': self.entityState.timeout, 'entity': self.__entity}
            if self.__isServer:
                self.notifyObservers(transportPing = timeoutEvent)
            else:
                self.notifyObservers(protocolPing = timeoutEvent)
        self.__count += 1
        defer = self.ping(self.__entity)
        if self.__isServer:
            defer.addErrback(self._suppressError)
        defer.addCallback(self._onSuccess)
        defer.addErrback(self._suppressError)
        defer.addErrback(lambda failure: self.notifyObservers(error = failure))

    def stopPing(self):
        '''Stop ping loop'''
        if self.__pingLoop and self.__pingLoop.running:
            self.__pingLoop.stop()
            self.__pingLoop = None
            log.msg('Ping to %s stopped' % self.__entity)

    def startPing(self):
        '''Start ping loop'''
        self.__count = 0
        self.__pingLoop = LoopingCall(self._sendPing)
        self.__pingLoop.start(self.__interval, now = False)
        log.msg('Ping to %s started' % self.__entity)


class PingManager(object):
    '''
    Manages multiple ping protocols
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent, transport):
        '''
        Constructor
        @param parent: Object that handles events from ping protocols
        @param transport: Ping protocols will sent ping through transport
        '''
        object.__init__(self)
        self.__parent = parent
        self.__transport = transport
        self.__pinger = dict()

    def startPing(self, entity):
        '''
        Start ping to entity
        @type entity: JID
        '''
        if entity not in self.__pinger:
            return
        self.__pinger[entity].startPing()

    def stopPing(self, entity):
        '''
        Stop ping to entity
        @type entity: JID
        '''
        if entity not in self.__pinger:
            return
        self.__pinger[entity].stopPing()

    def addPinger(self, entity, interval, maxCount, isServer = False):
        '''
        Add pinger to manager
        @param entity: Entity to be pinged
        @type entity: JID
        @param interval: Interval between sending ping
        @type interval: int
        @param maxCount: if maxCount pings without response has been occured,
        connection is considered lost
        @type maxCount: int
        @param isServer: True means that we ping jabber server
        @param isServer: bool
        '''
        if entity not in self.__pinger:
            return
        self.__pinger[entity] = Pinger(entity, interval, maxCount,
                isServer)
        self.__pinger[entity].addObserver(self.__parent)
        self.__pinger[entity].setHandlerParent(self.__transport)

    def removePinger(self, entity):
        '''
        Remove specific pinger from manager
        @type entity: JID
        '''
        if entity not in self.__pinger:
            return
        self.stopPing(entity)
        if self.__transport:
            self.__pinger[entity].disownHandlerParent(self.__transport)
        self.__pinger[entity].deleteObserver(self.__parent)
        del self.__pinger[entity]

    def removeAll(self, exclude = None):
        '''
        Remove all pingers from manager except entities from exclude list
        @param exclude: exclude entities
        @type exclude: list
        '''
        for entity in self.__pinger.keys():
            if exclude and entity not in exclude:
                self.removePinger(entity)
