# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Abstract main window class
'''

import hivemind.settings as settings
from PyQt4.QtGui import QPainter, QGraphicsView, QPalette, QMainWindow
from PyQt4.QtCore import Qt, SIGNAL, pyqtSignal
from hivemind.gui_widgets import SearchingTrMixin
from hivemind.attribute import readable

class MainView(QGraphicsView):
    '''
    Main view of the application displaying mind map.
    
    @author: Andrew Vasilev
    '''

    readable('draggedPath')

    def __init__(self, scene, actionBag):
        QGraphicsView.__init__(self, scene)
        self.setRenderHints(QPainter.Antialiasing)
        self.setDragMode(QGraphicsView.ScrollHandDrag)
        self.setResizeAnchor(QGraphicsView.AnchorViewCenter)
        palette = self.palette()
        palette.setColor(QPalette.Base, Qt.white)
        self.setPalette(palette)
        self.__draggedPath = 0
        self.__actionBag = actionBag
        self.__zoomLevel = 0
        self.__actionBag.zoomInAction.triggered.connect(self._zoomIn)
        self.__actionBag.zoomOutAction.triggered.connect(self._zoomOut)
        self.__startPoint = None
        scene.mapChanged.connect(self._resetView)

    def scrollContentsBy(self, dx, dy):
        '''
        Scrolls scene contents

        @type dx: int
        @type dy: int
        '''
        QGraphicsView.scrollContentsBy(self, dx, dy)
        self.__draggedPath += abs(dx) + abs(dy)

    def mousePressEvent(self, event):
        '''
        Mouse press event handler

        @type event: QMouseEvent
        '''
        QGraphicsView.mousePressEvent(self, event)
        self.__draggedPath = 0
        self.__startPoint = event.pos()

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler.
        Emits itemClicked, mapClicked signals.

        @type event: QMouseEvent
        '''
        QGraphicsView.mouseReleaseEvent(self, event)
        self.__draggedPath += self._length(self.__startPoint, event.pos())

    @classmethod
    def _length(cls, point1, point2):
        '''
        Calculates approximate path between passed points

        @type point1: QPoint
        @type point2: QPoint

        @rtype: int
        '''
        diff = point2 - point1
        return abs(diff.x()) + abs(diff.y())

    __ZOOM_IN_COEFFICIENT = 1.25
    '''View scale coefficient for zoom in action'''

    __MAX_ZOOM_LEVEL = 8
    '''Maximum value for zoom level'''

    __MIN_ZOOM_LEVEL = -8
    '''Minimum value for zoom level'''

    def _zoom(self, scaleFactor):
        '''
        Change the scale factor of the mind map.
        @parame scaleFactor: which way to zoom and how many times, e.g. -1 zoom out once
        @type scaleFactor: int
        '''
        scale = MainView.__ZOOM_IN_COEFFICIENT ** scaleFactor
        self.scale(scale, scale)
        self.__zoomLevel += scaleFactor
        self._updateZoomActionsState()

    def _updateZoomActionsState(self):
        '''Update the state of the actions according to
        the zoom level'''
        if self.__zoomLevel == MainView.__MAX_ZOOM_LEVEL:
            self.__actionBag.zoomInAction.setEnabled(False)
        elif self.__zoomLevel == MainView.__MIN_ZOOM_LEVEL:
            self.__actionBag.zoomOutAction.setEnabled(False)
        else:
            self.__actionBag.zoomInAction.setEnabled(True)
            self.__actionBag.zoomOutAction.setEnabled(True)

    def _zoomIn(self):
        '''Zoom in scene view'''
        self._zoom(1)

    def _zoomOut(self):
        '''Zoom out scene view'''
        self._zoom(-1)

    def _resetView(self):
        '''
        Reset zoom transformations. Scroll the contents of
        the viewport to ensure that the root item is centered in the view.
        '''
        self.resetTransform()
        scale = MainView.__ZOOM_IN_COEFFICIENT ** settings.get('defaultZoomLevel')
        self.scale(scale, scale)
        self.__zoomLevel = 0
        self._updateZoomActionsState()
        self.centerOn(self.scene().rootDelegate)

    def resizeEvent(self, event):
        '''
        Resize event handler
        @type event: QResizeEvent
        '''
        self.emit(SIGNAL('resized'), event.size())
        return QGraphicsView.resizeEvent(self, event)


class AbstractMainWindow(SearchingTrMixin, QMainWindow):
    '''
    Abstract main window class

    Subclasses of this class must implement 'mindMapView' property

    @author: Alexander Kulikov
    '''

    closeEvnt = pyqtSignal()

    def __init__(self, actionBag, mindMapScene):
        '''Create new window'''
        QMainWindow.__init__(self)
        SearchingTrMixin.__init__(self)
        self._actionBag = actionBag
        self._initNavigateActions(mindMapScene)
        self._initActions()

    def _initNavigateActions(self, mindMapScene):
        '''Appends the navigate actions to mainView's list of actions'''
        self.addAction(self._actionBag.selectLeftNodeAction)
        self.addAction(self._actionBag.selectRightNodeAction)
        self.addAction(self._actionBag.selectUpNodeAction)
        self.addAction(self._actionBag.selectDownNodeAction)
        self.addAction(self._actionBag.selectRootNodeAction)

    def _initActions(self):
        '''Appends the actions to mainView's list of actions'''
        self.addAction(self._actionBag.editEdgeAction)
        self.addAction(self._actionBag.editLabelAction)
        self.addAction(self._actionBag.editNodeAction)
        self.addAction(self._actionBag.editNodeIconsAction)
        self.addAction(self._actionBag.foldNodeAction)
        self.addAction(self._actionBag.addNodeAction)
        self.addAction(self._actionBag.removeNodeAction)
        self.addAction(self._actionBag.enterTransferModeAction)
        self.addAction(self._actionBag.cancelTransferModeAction)
        self.addAction(self._actionBag.putNodeAction)
        self.addAction(self._actionBag.putNodeBelowAction)
        self.addAction(self._actionBag.putNodeAboveAction)
        self.addAction(self._actionBag.editHyperlinkAction)
        self.addAction(self._actionBag.openHyperlinkAction)
        self.addAction(self._actionBag.addSiblingNodeBelowAction)
        self.addAction(self._actionBag.addSiblingNodeAboveAction)
        self.addAction(self._actionBag.zoomInAction)
        self.addAction(self._actionBag.zoomOutAction)
        self.addAction(self._actionBag.showContextMenuAction)

    def closeEvent(self, event):
        '''Close event handler'''
        self.closeEvnt.emit()
        event.ignore()
