# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Protocol layer classes
'''

from twisted.internet import reactor
from twisted.python import log
from twisted import internet
from wokkel.xmppim import RosterClientProtocol
from wokkel.pubsub import PubSubClient, PubSubService, NS_PUBSUB_ERRORS, PUBSUB_REQUEST
from idavoll import error
from idavoll.backend import PubSubResourceFromBackend
from idavoll.memory_storage import Storage, LeafNode
from hivemind.attribute import Observable
from hivemind import enum, settings
import hivemind.network.core
from hivemind.network.changeset import Changeset, ChangesetStack

#pylint: disable=C0103


''' XMPP PubSub node name '''
XMPP_NODE_NAME = unicode('HiveMind')

class HivemindRosterClientProtocol(RosterClientProtocol, Observable):
    '''
    Client side roster protocol for Hivemind
    @author: Oleg Kandaurov
    '''

    '''Possible event types'''
    eventType = enum.enum('roster', 'remove', 'set')

    def __init__(self):
        '''Constructor'''
        RosterClientProtocol.__init__(self)
        Observable.__init__(self)

    def connectionInitialized(self):
        '''Called when connection has been initialized'''
        RosterClientProtocol.connectionInitialized(self)
        defer = self.getRoster()
        defer.addCallback(lambda roster: self.notifyObservers(roster =
                {'type': self.eventType.roster, 'data': roster}))
        defer.addErrback(lambda failure: self.notifyObservers(error = failure))

    def onRosterRemove(self, entity):
        '''
        Called when a roster push for the removal of an item was received
        @param entity: The entity for which the roster item has been removed
        @type entity: JID
        '''
        self.notifyObservers(roster = {'type': self.eventType.remove, 'data': entity})

    def onRosterSet(self, item):
        '''
        Called when a roster push for a new or update item was received
        @param item: The pushed roster item
        @type item: RosterItem
        '''
        self.notifyObservers(roster = {'type': self.eventType.set, 'data': item})


class HivemindPubSubService(PubSubService, Observable):
    '''
    Pubsub service for Hivemind
    @author: Oleg Kandaurov
    '''

    def __init__(self, resource = None):
        '''Constructor'''
        PubSubService.__init__(self, resource)
        Observable.__init__(self)
        self.jid = None

    def connectionInitialized(self):
        '''The XML stream has been initialized'''
        self.notifyObservers(connection = hivemind.network.core.connectionState.collaboration)
        self.jid = self.parent.jid

    def close(self, *args, **kw):
        '''Close service'''
        #Observer are not removed at all. Seems like bug in Wokkel.
        self.xmlstream.removeObserver(PUBSUB_REQUEST, self.handleRequest)
        self.notifyObservers(connection = hivemind.network.core.connectionState.initialized)
        return internet.defer.succeed(None)

    def proposeChangeset(self, changeset):
        '''
        Propose changeset to the HiveMind network
        @param changeset: changeset to publish inside network
        @type changeset: Changeset
        '''
        return self.resource.proposeChangeset(changeset, self.jid)


class HivemindResourceFromBackend(PubSubResourceFromBackend):
    '''
    Pubsub resource from backend for Hivemind
    @author: Oleg Kandaurov
    '''

    def __init__(self, backend):
        '''
        Constructor
        @param backend: Generic publish - subscribe backend service
        @type backend: BackendService
        '''
        PubSubResourceFromBackend.__init__(self, backend)
        self.serviceJID = None

    def subscribe(self, request):
        '''Called when a subscribe request has been received'''
        def subscribeSucceded(subscription):
            '''Notify observers about subscription state'''
            defer = self._isWhitelistModel(request.nodeIdentifier)
            defer.addCallback(lambda whitelist: None if whitelist else
                    self._assignAffiliation(
                    request.nodeIdentifier, request.subscriber,
                    settings.get('defaultAffiliation')))
            return subscription
        defer = self.backend.subscribe(request.nodeIdentifier, request.subscriber,
                request.sender)
        defer.addCallback(subscribeSucceded)
        return defer.addErrback(self._mapErrors)

    def unsubscribe(self, request):
        '''Called when a unsubscribe request has been received'''
        def unsubscribeSucceded(result):
            '''Notify observers about subscription state'''
            defer = self._isWhitelistModel(request.nodeIdentifier)
            defer.addCallback(lambda whitelist: None if whitelist else
                    self._assignAffiliation(request.nodeIdentifier, request.subscriber, None))
            return result
        defer = self.backend.unsubscribe(request.nodeIdentifier, request.subscriber,
                request.sender)
        defer.addCallback(unsubscribeSucceded)
        return defer.addErrback(self._mapErrors)

    def _isWhitelistModel(self, nodeIdentifier):
        '''Check if node has whitelist access model'''
        defer = self.backend.getNodeConfiguration(nodeIdentifier)
        defer.addCallback(lambda config: config['pubsub#access_model'] == 'whitelist')
        return defer

    def _assignAffiliation(self, nodeIdentifier, entity, affiliation):
        '''Set affiliation for entity'''
        defer = self.backend.storage.getNode(nodeIdentifier)
        defer.addCallback(lambda node: node.setAffiliation(entity, affiliation))
        return defer

    def proposeChangeset(self, changeset, owner):
        '''
        Propose changeset to the HiveMind network
        @param changeset: changeset to publish inside network
        @type changeset: Changeset
        '''
        def notify(subscriptions, node):
            '''Notify all subscribers'''
            for subscription in subscriptions:
                self._notify({'items': node.stack.getItems(1),
                        'nodeIdentifier': node.nodeIdentifier,
                        'subscription': subscription})
        defer = self.backend.storage.getNode(XMPP_NODE_NAME)
        defer.addCallback(lambda node: (node.storeChangeset(changeset, owner), node))
        defer.addCallback(lambda (_, node): (node.getSubscriptions(), node))
        defer.addCallback(lambda (defer, node): defer.addCallback(notify, node))
        return defer


class HivemindClient(PubSubClient, Observable):
    '''
    Pubsub client for HiveMind service
    @author: Andrew Vasilev
    '''

    def __init__(self, hivemindService):
        '''
        @param hivemindService: jid of hivemind service
        @type hivemindService: JID
        '''
        PubSubClient.__init__(self)
        Observable.__init__(self)
        self.__service = hivemindService
        self.jid = None

    def connectionInitialized(self):
        '''The XML stream has been initialized.'''
        PubSubClient.connectionInitialized(self)
        self.jid = self.parent.jid
        defer = self.subscribe(self.__service, XMPP_NODE_NAME, self.jid)
        defer.addCallback(self.retrieveItems)
        defer.addCallback(lambda success: self.notifyObservers(
                connection = hivemind.network.core.connectionState.collaboration))
        defer.addErrback(lambda failure: self.notifyObservers(error = failure))

    def close(self, *args, **kw):
        '''Close client'''
        defer = self._unsubscribe()
        return defer

    def _unsubscribe(self):
        '''Unsubscribe from the service'''
        def notifyUnsubscription(result):
            ''' Notify observers about unsubscribe event '''
            log.msg('Unsubscribed from %s' % self.__service.full())
            if unsubscribeTimer.active():
                unsubscribeTimer.cancel()
            self.notifyObservers(connection =
                    hivemind.network.core.connectionState.initialized)
            return None

        unsubscribeTimer = reactor.callLater(hivemind.network.core.UNSUBSCRIBE_TIMEOUT,
                notifyUnsubscription, self)
        if self.xmlstream is None:
            notifyUnsubscription(None)
            return internet.defer.succeed(None)
        defer = self.unsubscribe(self.__service, XMPP_NODE_NAME, self.jid)
        defer.addBoth(notifyUnsubscription)
        return defer

    def retrieveItems(self, result = None):
        '''Retrieve all items from the server'''
        defer = self.items(self.__service, XMPP_NODE_NAME)
        defer.addCallback(self._parseInitItems)
        defer.addErrback(lambda failure: self.notifyObservers(error = failure))

    def _parseInitItems(self, items):
        '''
        Parse items on initial request
        @param items: a set of items retrieved from the service
        @type items: list
        '''
        for item in items:
            changeset = Changeset.fromElement(item)
            self.notifyObservers(changeset = changeset)

    def itemsReceived(self, event):
        '''
        Called when an items notification has been received for a node.
        @param event: The items event.
        @type event: ItemsEvent
        '''
        if event.sender != self.__service or event.nodeIdentifier != XMPP_NODE_NAME:
            return
        self._parseInitItems(event.items)

    def proposeChangeset(self, changeset):
        '''
        Propose changeset to the hivemind network
        @param changeset: changeset to send to the controller
        @type changeset: str
        '''
        changeset.publisher = self.jid
        defer = self.publish(self.__service, XMPP_NODE_NAME,
                [changeset.toElement()])
        defer.addErrback(lambda failure: self.notifyObservers(error = failure))


class HivemindStorage(Storage):
    '''
    Hivemind storage
    @author: Oleg Kandaurov
    '''

    def createNode(self, nodeIdentifier, owner, config):
        if nodeIdentifier in self._nodes:
            return internet.defer.fail(error.NodeExists())
        if config['pubsub#node_type'] != 'leaf':
            raise error.NoCollections()
        node = HivemindNode(nodeIdentifier, owner, config)
        self._nodes[nodeIdentifier] = node
        return internet.defer.succeed(None)


class HivemindNode(LeafNode, Observable, object):
    '''
    Hivemind node
    @author: Oleg Kandaurov
    '''

    def __init__(self, nodeIdentifier, owner, config):
        '''Constructor'''
        Observable.__init__(self)
        LeafNode.__init__(self, nodeIdentifier, owner, config)
        config['pubsub#send_last_published_item'] = 'never'
        config['pubsub#persist_items'] = True
        config['pubsub#deliver_payloads'] = True
        self.stack = ChangesetStack()
        self.owner = owner

    def addSubscription(self, subscriber, state, options):
        '''Add new subscription to this node with given state.'''
        def subscriptionAdded(result):
            ''' Process successful subscription addition'''
            self.notifyObservers(subscription = {
                    'state': hivemind.network.core.subscriptionState.subscribed,
                    'entity': subscriber})
            return result

        defer = LeafNode.addSubscription(self, subscriber, state, options)
        defer.addCallback(subscriptionAdded)
        return defer

    def removeSubscription(self, subscriber):
        '''Remove subscription to this node.'''
        def subscriptionRemoved(result):
            ''' Process successful subscription removing'''
            self.notifyObservers(subscription = {
                    'state': hivemind.network.core.subscriptionState.unsubscribed,
                    'entity': subscriber})
            return result

        defer = LeafNode.removeSubscription(self, subscriber)
        defer.addCallback(subscriptionRemoved)
        return defer

    def storeItems(self, items, publisher):
        '''Store items in persistent storage for later retrieval'''
        for element in items:
            element.attributes['publisher'] = publisher.full()
            self.stack.addUnsignedChangeset(Changeset.fromElement(element))
            self.notifyObservers(changeset = Changeset.fromElement(element))
        return internet.defer.succeed(None)

    def storeChangeset(self, changeset, publisher):
        '''
        Store changeset in persistent storage for later retrieval
        @param items: The changeset to be stored
        @type items: Changeset
        @param publisher: JID of the publishing entity
        @type publisher: JID
        '''
        changeset.publisher = publisher
        self.stack.addUnsignedChangeset(changeset)
        self.notifyObservers(changeset = changeset)

    def getItems(self, maxItems = None):
        '''Get items'''
        return internet.defer.succeed(self.stack.getItems(maxItems))

    def getItemsById(self, itemIdentifiers):
        '''Get list of items by id'''
        items = []
        for itemIdentifier in itemIdentifiers:
            items.append(self.stack.getItemById(itemIdentifier))
        return internet.defer.succeed(items)

    def setAffiliation(self, entity, affiliation):
        '''Set affiliation'''
        LeafNode.setAffiliation(self, entity, affiliation)
        defer = self.getAffiliation(entity)
        defer.addCallback(lambda affiliation:
                self.notifyObservers(affiliation = (entity.userhost(), affiliation)))
        return internet.defer.succeed(None)


def isPubSubError(err):
    '''
    Check if error is PubSubError
    @type error: StanzaError
    @rtype bool
    '''
    if err.appCondition is None or err.appCondition.uri != NS_PUBSUB_ERRORS:
        return False
    return True


def pubsubConditionFromStanzaError(err):
    '''
    Return pubsub specific condition and feature from stanza error
    @type error: StanzaError
    @rtype: tuple of str, str
    '''
    if not isPubSubError(err):
        return None, None
    return err.appCondition.name, err.appCondition.getAttribute('feature')
