# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Set of changeset related classes
'''

import uuid
from twisted.words.xish.domish import Element
from twisted.words.protocols.jabber.jid import JID
from wokkel.generic import stripNamespace
from hivemind.attribute import readable

#pylint: disable=C0103

class Changeset(object):
    '''
    Changeset representation
    @author: Andrew Vasilev
    '''

    def __init__(self, data = None, changesetType = None, publisher = None,
            changesetId = None, number = None):
        '''
        Initialize object from the several components
        @param data: xml-serialized data of the changeset
        @type data: str
        @param changesetType: type of the changeset
        @type changesetType: str
        @param changesetId: unique id of the changeset
        @type changesetId: str
        @param publisher: publisher of the changeset
        @type publisher: JID
        @param number: the number of the changeset
        @type number: int
        '''
        #pylint: disable=R0913
        self.id = changesetId
        self.number = number
        self.data = data
        self.type = changesetType
        self.publisher = publisher

    def __str__(self):
        return str(self.number) + ' : ' + self.publisher.full() + ' : ' + \
                str(self.id) + ' : ' + str(self.type) + ' : ' + str(self.data)

    def toElement(self):
        '''
        Convert this changeset into domish xml Element
        @return: Element representation of current object
        @rtype: Element
        '''
        element = Element((None, 'item'))
        if self.id: element.attributes['id'] = self.id
        if self.number: element.attributes['number'] = str(self.number)
        if self.type: element.attributes['type'] = self.type
        if self.publisher: element.attributes['publisher'] = self.publisher.full()
        element.addRawXml(self.data.decode('utf-8'))
        return element

    def isValid(self, fieldList):
        '''
        Check fields of the changeset to be valid
        @param fieldList: fields of the changeset to test for None
        @type fieldList: list of str
        @rtype: bool
        '''
        for field in fieldList:
            if self.__getattribute__(field) is None:
                raise ChangesetException('Field ' + field + ' cannot be None')
        return True

    @staticmethod
    def fromElement(element):
        '''
        Create Changeset object from xml Element
        @param element: xml representation of the object
        @type element: Element
        '''
        xmlChangeset = element.firstChildElement()
        stripNamespace(xmlChangeset)
        changeset = xmlChangeset.toXml().encode('utf8')
        return Changeset(changeset, element.getAttribute('type'),
                JID(element.getAttribute('publisher')), element.getAttribute('id'),
                element.getAttribute('number'))


class ChangesetStack(object):
    '''
    Stack of the changesets
    @author: Andrew Vasilev, Oleg Kandaurov
    '''

    def __init__(self):
        self.__changesets = []
        self.__elements = []

    def getItemById(self, itemId):
        '''
        Get item by item id. Item is a xml Element
        @param itemId: Item id
        @type itemId: str
        @rtype: Element
        '''
        for element in self.__elements:
            if element.getAttribute('id') == itemId:
                return element
        return None

    def getItems(self, maxItems = None):
        '''
        Get items from stack
        @param maxItems: number of recently published items or all item if None
        @type maxItems: int
        @rtype: list of Element
        '''
        if maxItems:
            return self.__elements[-maxItems:]
        else:
            return self.__elements

    def addChangeset(self, changeset):
        '''
        Add changeset to the stack
        @param changeset: fully-functional changeset
        @type changeset: Changeset
        @raise ChangesetError: if passed changeset is invalid
        @raise ChangesetNumberError: if changeset has invalid humber
        @raise ChangesetIdError: if changeset has invalid id
        '''
        changeset.isValid(['id', 'number', 'data', 'type', 'publisher'])
        self._checkChangesetNumber(changeset)
        if not self._isUniqueId(changeset.id):
            raise ChangesetIdException('Changeset has not unique id')
        self.__changesets.append(changeset)
        self.__elements.append(changeset.toElement())

    def addUnsignedChangeset(self, changeset):
        '''
        Add unsigned changeset to the stack. Warning! Changes passed object!
        @param changeset: changeset with data and type fields present
        @type changeset: Changeset
        @raise ChangesetError: if passed changeset is invalid
        '''
        self.signChangeset(changeset)
        self.addChangeset(changeset)

    def signChangeset(self, changeset):
        '''
        Set appropriate number and hash for the changeset if needed
        @type changeset: Changeset
        '''
        changeset.number = len(self.__changesets)
        if not self._isUniqueId(changeset.id):
            changeset.id = self._generateUniqueId()

    def _checkChangesetNumber(self, changeset):
        '''
        Check number of the changeset to have next number in the list
        @type changeset: Changeset
        @raise ChangesetException: if changeset has wrong number
        '''
        if changeset.number != len(self.__changesets):
            raise ChangesetNumberException('Wrong number of the changeset.' +
                    '%d expected, but %d got' % (len(self.__changesets), changeset.number))

    def _generateUniqueId(self):
        '''
        Generate unique identifier for the changeset
        @return: new unique identifier for the changeset
        @rtype: str
        '''
        while True:
            newId = str(uuid.uuid4())
            if not self._isUniqueId(newId): continue
            return newId

    def _isUniqueId(self, changesetId):
        '''
        Check for the uniqueness of the id
        @type id: str
        @return: False if there is a changeset with the same id
        @rtype: bool
        '''
        if changesetId is None or changesetId is '': return False
        for changeset in self.__changesets:
            if changeset.id == changesetId:
                return False
        return True

    def __len__(self):
        '''Add support for getting number of changesets via len()'''
        return len(self.__changesets)

    def lastId(self):
        '''
        Get the id of the last changeset we are holding
        @rtype: str
        '''
        if len(self.__changesets) == 0: return None
        return self.__changesets[-1].id

    def items(self, changesetId = None):
        '''
        Retrieve the list of last items in the stack. If no id is passed,
        then return all available items
        @param id: id of the changeset known to the user
        @type id: str
        @rtype: list of Element
        '''
        if changesetId is None: return self.__elements
        index = 0
        for changeset in self.__changesets:
            index += 1
            if changeset.id == changesetId:
                return self.__elements[index:]
        return self.__elements

    def itemsByNumber(self, number = 1):
        '''
        Retrieve the list of last items in the stack.
        @param number: maximum number of items to recieve
        @type number: int
        @rtype: list of Element
        '''
        if number < 0 or number is None:
            return self.__elements
        count = min(number, len(self.__changesets))
        return self.__elements[-count:]


class ChangesetException(Exception):
    '''
    Base exception class for all exception that can be raised by ChangesetStack
    @author: Andrew Vasilev
    '''

    readable('reason')
    def __init__(self, reason):
        '''
        @param reason: reason for the exception to be raised
        @type reason: str
        '''
        Exception.__init__(self)
        self.__reason = reason

    def __str__(self):
        return self.__reason


class ChangesetNumberException(ChangesetException):
    '''
    Exception indicates that something is wrong with the number of the changeset
    @author: Andrew Vasilev
    '''

    def __init__(self, reason):
        ChangesetException.__init__(self, reason)


class ChangesetIdException(ChangesetException):
    '''
    Exception indicates that something is wrong with the identification of the changeset
    @author: Andrew Vasilev
    '''

    def __init__(self, reason):
        ChangesetException.__init__(self, reason)