# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA
'''
Module contains special robots for filling mind map with external data in
'''

from hivemind.smartconference.kp import M3_AVAILABLE, SmartSpace, VisitorChecker, \
    EventConverter, TalkChangeHandler, SlideChangeHandler
from hivemind.dialogs import SmartSpaceConnectionDialog
from hivemind import gui_factory
from hivemind.attribute import readable, writable
from PyQt4.QtCore import QObject, SIGNAL
from threading import Event, Lock
import copy

class SmartConferenceRobot(object):
    '''
    Creates specialized data from the smart conference system. This robot may work for a long
    time.

    @author: Andrew Vasilev
    '''

    readable('connection', 'lock', 'event')
    writable('talks', 'currentTalk')

    def __init__(self, mindMapDelegate, networkController, actionBag):
        '''
        Initialize the smart conference robot
        @param mindMapDelegate: mind map delegate, holds current mind map
        @type mindMapDelegate: MindMapDelegate
        @param networkController: controller to publish commands to
        @type networkController: NetworkController
        @type actionBag: ActionBag
        '''
        self.__connection = None
        self.__mindMapDelegate = mindMapDelegate
        self.__networkController = networkController
        self.__talks = []
        self.__externalTalks = []
        self.__currentTalk = None
        self.__event = Event()
        self.__lock = Lock()
        self.__actionBag = actionBag
        self._configureActions(True)
        self.__eventConverter = None
        self.__mindMapDelegate.mapChanged.connect(self._mindMapChanged)

    def _configureActions(self, startEnabled, stopEnabled = None):
        '''
        Setup actions related to this robot
        @param startEnabled: whether star action should be enabled
        @type startEnabled: bool
        @param stopEnabled: set something here if you want to explicitly set action state, it
        would be set to opposite to startEnabled
        @type stopEnabled: bool
        '''
        self.__actionBag.startSmartConferenceRobot.setEnabled(startEnabled)
        if not stopEnabled:
            self.__actionBag.stopSmartConferenceRobot.setEnabled(not startEnabled)
        else:
            self.__actionBag.stopSmartConferenceRobot.setEnabled(stopEnabled)

    def start(self):
        '''Begin data collection from the smart space'''
        def connect():
            '''Enable connection to smart space'''
            try:
                self.__connection = SmartSpace(dialog.name, dialog.host, dialog.port)
            except:
                raise Exception('Unable to connect to the smart space')
            self._configureActions(False)
            self.__eventConverter = EventConverter(self.__event, self.__mindMapDelegate)
            self.__eventConverter.received.connect(self._synchronizeStatus)
            self._setupSubscriptions()
            self.__eventConverter.start()
        if not M3_AVAILABLE:
            raise Exception('Smart M3 python support is not installed')
        dialog = SmartSpaceConnectionDialog(gui_factory.defaultParent())
        QObject.connect(dialog, SIGNAL('accepted()'), connect)
        dialog.exec_()

    def _setupSubscriptions(self):
        '''Create subscription handlers and subscribe them for changes'''
        VisitorChecker(self, self.__mindMapDelegate.mindMap.root, gui_factory.defaultParent())
        TalkChangeHandler(self)
        SlideChangeHandler(self)

    def _synchronizeStatus(self):
        '''Synchronize current status of the talks with the mind map'''
        self.__lock.acquire()
        for talk in self.__externalTalks:
            self.__talks.append(copy.copy(talk))
        self.__externalTalks = []
        for talk in self.__talks:
            for command in talk.synchronizeStatus(self.__mindMapDelegate.mindMap):
                self.__networkController.publishCommand(command)
        self.__lock.release()

    def _mindMapChanged(self):
        '''When mind map changes we need to clear the cache of the nodes and stop the robot'''
        self.__talks = []
        self.stop()

    def stop(self):
        '''Stop data collection from the smart space'''
        if self.__eventConverter:
            self.__eventConverter.received.disconnect(self._synchronizeStatus)
            self.__eventConverter.terminate()
            self.__eventConverter = None
        try:
            if self.__connection:
                self.__connection.close()
        finally:
            self.__connection = None
            self._configureActions(True)

    def addTalk(self, talk):
        '''
        Add new talk to the conference
        @type talk: NodeRepresented
        '''
        self.__externalTalks.append(talk)
